package stream

import (
	"bufio"
	"fmt"
	"os/exec"
	"time"

	"github.com/stvp/rollbar"

	"code.justin.tv/creative/streamer/lib/missioncontrol"
	"code.justin.tv/creative/streamer/lib/session_tracker"
)

// streamCount is the number of simulataneous streams per box
const streamCount int = 1

// poll Indefinitely consumes Stream sessions from mission control making sure
// that the number of streams does not exceed streamCount
func poll(streamSessionsChan chan missioncontrol.StreamSession, doneChan chan bool, streamCount int) {
	for i := 0; i < streamCount; i++ {
		streamSessionsChan <- missioncontrol.PopStreamSession(3 * time.Second)
	}
	for {
		<-doneChan
		streamSessionsChan <- missioncontrol.PopStreamSession(10 * time.Second)
	}
}

// Streamer is the main entry point for streaming content. It relies on the
// server running. It polls misison control for new streams then whenever it
// receives a new stream session it runs it in ffmpeg making sure to update
// mission control with the status of the stream
func Streamer() {
	streamSessionsChan := make(chan missioncontrol.StreamSession)
	doneChan := make(chan bool)
	go poll(streamSessionsChan, doneChan, streamCount)

	for {
		ss := <-streamSessionsChan
		go runStream(ss, doneChan)
	}
}

func runStream(ss missioncontrol.StreamSession, doneChan chan bool) {
	restartStream := true // Set stream to start by default
	restartChan := make(chan bool)
	for restartStream {
		go BeginStream(ss, restartChan)
		restartStream = <- restartChan // Sets restartStream true if reset, false if cancel
	}
	doneChan <- true
}

// BeginStream starts a StreamSession and sends a message to the done channel
// when it is finished
func BeginStream(ss missioncontrol.StreamSession, restartChan chan bool) {
	cmdName := "ffmpeg"
	cmdArgs := []string{
		"-protocol_whitelist", "file,http,https,tcp,tls",
		"-re",
		"-auto_convert", "1",
		"-segment_time_metadata", "1",
		"-f", "concat",
		"-i", fmt.Sprintf("http://localhost:8080/stream_sessions-%d-file_list-%d.ffconcat", ss.ID, 1),
		"-c", "copy",
		"-f", "flv",
		fmt.Sprintf("rtmp://live.twitch.tv/app/%s", ss.Channel.StreamKey),
		"-loglevel", "trace",
	}

	cmd := exec.Command(cmdName, cmdArgs...)

	// Channel to receive cancels and stops
	cancelChan := make(chan bool)

	go cancelStreamOnCancel(ss, cancelChan)

	pipe, err := cmd.StderrPipe()
	if err != nil {
		rollbar.Error(rollbar.ERR, err)
	}

	scanner := bufio.NewScanner(pipe)
	go func() {
		err = cmd.Start()
		if err != nil {
			rollbar.Error(rollbar.ERR, err)
		}

		err = cmd.Wait()
		if err != nil {
			rollbar.Error(rollbar.INFO, err)
		}
	}()

	go trackSession(scanner, cancelChan)

	// Blocks until stream is canceled or reset (on 36 hour timeout )
	select {
		case cancelled := <- cancelChan:
			if cancelled {
				_ = cmd.Process.Kill()
				missioncontrol.UpdateStreamSession(ss.ID, "finished")
				restartChan <- false
			} else {
				resp := missioncontrol.GetStreamSession(ss.ID)
				if resp.State != "finished" {
					restartChan <- true
				}
			}
		case <-time.After(time.Hour * 36):
			_ = cmd.Process.Kill()
			restartChan <- true
	}

}

func trackSession(scanner *bufio.Scanner, cancelChan chan bool) {
	sessiontracker.ProcessFFMPEGLogs(scanner)
	cancelChan <- false
}

func cancelStreamOnCancel(ss missioncontrol.StreamSession, cancelChan chan bool) {
	for {
		resp := missioncontrol.GetStreamSession(ss.ID)
		if resp.State == "canceled" {
			cancelChan <- true
			break
		} else if resp.State == "finished" {
			break
		}
		time.Sleep(1 * time.Second)
	}
}
