defmodule Manager.Video do
  use Manager.Web, :model
  alias ExAws.S3
  alias Manager.Channel
  alias Manager.Video
  alias Manager.Repo
  alias Experimental.Flow
  import FFmpex
  use FFmpex.Options
  import FFprobe
  require IEx

  @bucket "twitch-creative-video-repository"
  schema "videos" do
    field :title, :string
    field :show, :string
    field :season, :string
    field :s3_path, :string
    field :channel_id, :integer
    field :length, :integer
    field :unlocked_by, :string

    timestamps()
  end

  def load_from_s3(prefix) do
    videos = S3.list_objects(@bucket, prefix: prefix)
    |> ExAws.stream!
    |> Stream.take(5)
    |> Stream.map(&(&1.key))
    |> Stream.map(&video_from_key/1)
    |> Stream.filter(&(&1))
    |> Flow.from_enumerable(stages: 20)
    |> Flow.partition(stages: 20)
    |> Flow.map(&add_duration/1)
    |> Enum.to_list
    |> Enum.map(&(Video.changeset(%Video{}, &1)))
    |> Enum.map(&Repo.insert/1)
    # |> Enum.map(&Map.from_struct/1)
    # |> Enum.map(&Map.drop(&1, [:__meta__, :id]))

    # IEx.pry
    # Repo.insert_all(Video, videos)
  end

  def video_from_key(key) do
    pieces = String.split(key, "/")
    case length(pieces) do
      4 ->
        [channel_name, show, season, title] = String.split(key, "/")
        channel = from(c in Channel, where: c.name == ^channel_name) |> Repo.one!
        %{
          channel_id: channel.id,
          show: show,
          season: season,
          title: title,
          s3_path: key
        }
      _ -> nil
    end
  end

  def signed_url(video) do
    {:ok, signed_url} = ExAws.S3.presigned_url(ExAws.Config.new(:s3, []), :get, @bucket, video.s3_path)
    signed_url
  end

# ExAws.S3.presigned_url(ExAws.Config.new(:s3, []), :get, "twitch-creative-video-repository", "food/A_Taste_Of_History/001/001_martha.mp4")
  def add_duration(video) do
    signed_url = signed_url(video)
    {output, 0} = System.cmd("ffprobe", [
      "-v", "error",
      "-select_streams", "v:0",
      "-show_entries",
      "stream=duration",
      "-of", "default=noprint_wrappers=1:nokey=1",
      signed_url,
      ], stderr_to_stdout: true)

    [num_str, _] = String.split(output, "\n")

    duration = String.to_float(num_str)

    Map.put(video, :length, round(duration))
  end

  @doc """
  Builds a changeset based on the `struct` and `params`.
  """
  def changeset(struct, params \\ %{}) do
    struct
    |> cast(params, [:title, :show, :season, :s3_path, :channel_id, :length, :unlocked_by])
    |> validate_required([:title, :show, :season, :s3_path, :channel_id, :length])
  end
end
