alias Experimental.GenStage
defmodule Timelapser.TimelapseJobProducer do
  use GenStage
  alias Timelapser.Repo
  alias Timelapser.Timelapse
  import Ecto.Query
  def start_link do
    GenStage.start_link(__MODULE__, 0, name: __MODULE__)
  end

  def init(counter) do
     {:producer, counter}
  end

  def handle_demand(demand, state) do
    limit = demand + state
    {:ok, {count, events}} = take(limit)
    {:noreply, events, limit - count}
  end

  def handle_info(:timelapse_enqueued, state) do
    limit = state
    {:ok, {count, events}} = take(limit)
    {:noreply, events, limit - count}
  end

  def take(limit) do
    Repo.transaction fn ->
      ids = Repo.all waiting_timelapses(limit)
      {count, ids} = Repo.update_all timelapses_by_ids(ids),
                          [set: [status: "running", updated_at: Ecto.DateTime.utc]],
                          [returning: [:id]]
      id_list = ids |> Enum.flat_map(&Map.values/1) |> IO.inspect
      timelapses = Repo.all(from t in Timelapse,
                            where: t.id in ^id_list,
                            preload: [:building_blocks])

      {count, timelapses}
    end
  end

  defp timelapses_by_ids(ids) do
    from t in "timelapses", where: t.id in ^ids
  end

  defp waiting_timelapses(demand) do
    from t in "timelapses",
      where: t.status == "waiting" or (t.updated_at < ago(2, "minute") and t.status == "running"),
      limit: ^demand,
      select: t.id,
      lock: "FOR UPDATE SKIP LOCKED"
  end
end
