defmodule Timelapser.TimelapseControllerTest do
  use Timelapser.ConnCase

  alias Timelapser.Timelapse
  @valid_attrs %{}
  @invalid_attrs %{}

  setup %{conn: conn} do
    {:ok, conn: put_req_header(conn, "accept", "application/json")}
  end

  test "lists all entries on index", %{conn: conn} do
    conn = get conn, timelapse_path(conn, :index)
    assert json_response(conn, 200)["data"] == []
  end

  test "shows chosen resource", %{conn: conn} do
    timelapse = Repo.insert! %Timelapse{}
    conn = get conn, timelapse_path(conn, :show, timelapse)
    assert json_response(conn, 200)["data"] == %{"id" => timelapse.id}
  end

  test "renders page not found when id is nonexistent", %{conn: conn} do
    assert_error_sent 404, fn ->
      get conn, timelapse_path(conn, :show, -1)
    end
  end

  test "creates and renders resource when data is valid", %{conn: conn} do
    conn = post conn, timelapse_path(conn, :create), timelapse: @valid_attrs
    assert json_response(conn, 201)["data"]["id"]
    assert Repo.get_by(Timelapse, @valid_attrs)
  end

  test "does not create resource and renders errors when data is invalid", %{conn: conn} do
    conn = post conn, timelapse_path(conn, :create), timelapse: @invalid_attrs
    assert json_response(conn, 422)["errors"] != %{}
  end

  test "updates and renders chosen resource when data is valid", %{conn: conn} do
    timelapse = Repo.insert! %Timelapse{}
    conn = put conn, timelapse_path(conn, :update, timelapse), timelapse: @valid_attrs
    assert json_response(conn, 200)["data"]["id"]
    assert Repo.get_by(Timelapse, @valid_attrs)
  end

  test "does not update chosen resource and renders errors when data is invalid", %{conn: conn} do
    timelapse = Repo.insert! %Timelapse{}
    conn = put conn, timelapse_path(conn, :update, timelapse), timelapse: @invalid_attrs
    assert json_response(conn, 422)["errors"] != %{}
  end

  test "deletes chosen resource", %{conn: conn} do
    timelapse = Repo.insert! %Timelapse{}
    conn = delete conn, timelapse_path(conn, :delete, timelapse)
    assert response(conn, 204)
    refute Repo.get(Timelapse, timelapse.id)
  end
end
