defmodule Timelapser.Timelapse do
  import FFmpex
  use FFmpex.Options
  import FFprobe
  alias Timelapser.Chunk
  alias Timelapser.Timelapse
  alias Timelapser.Repo
  alias Timelapser.BuildingBlock
  alias Experimental.Flow
  use Timelapser.Web, :model

  @timelapse_bot_id 135514044
  @timelapse_bot_login "timelapse_bot"
  schema "timelapses" do
    field :status, :string, default: "waiting"
    field :clip_url, :string
    has_many :building_blocks, Timelapser.BuildingBlock

    timestamps()
  end

  def build_timelapse(%Timelapser.Timelapse{id: _id, building_blocks: [%Timelapser.BuildingBlock{vod_id: vod_id}]} = timelapse) do
    with  vod                                                                 <- Timelapser.Vinyl.get_vod(vod_id),
          %{"channel" => %{"name" => channel}}                                <- Timelapser.TwitchApi.get_vod(vod_id),
          %{"duration" => duration, "title" => title, "owner_id" => owner_id} <- vod,
          url                                                                 <- Timelapser.TwitchApi.get_playlist_url(vod_id),
          timelapse_path                                                      <- generate_timelapse(url, 810 * 2),
          timelapse_vod                                                       <- Timelapser.TwitchApi.create_timelapse_upload("Timelapse_of_something",  "timelapse_bot"),
          %{"upload" => %{"token" => upload_token}}                           <- timelapse_vod,
          %{"video" => %{"_id" => timelapse_vod_id}}                          <- timelapse_vod,
          title_path                                                          <- generate_title(channel),
          credits_path                                                        <- generate_credits(channel),
          concat_timelapse_path                                               <- concat_op_tl_ed(title_path, timelapse_path, credits_path),
          full_timelapse_path                                                 <- add_audio(concat_timelapse_path),
          :ok                                                                 <- Timelapser.VodUploader.upload(timelapse_vod_id, upload_token, full_timelapse_path),
          _response                                                           <- Timelapser.Vinyl.update_vod(timelapse_vod_id, %{owner_id: owner_id}),
          %{"clip_url" => clip_url}                                           <- Timelapser.ClipCreator.create_clip(timelapse_vod_id, @timelapse_bot_login),
          _clean                                                              <- File.rm(timelapse_path),
          %{"clip_url" => clip_url}                                           <- Timelapser.ClipCreator.create_clip(timelapse_vod_id, channel),
          _response                                                           <- Timelapser.Vinyl.update_vod(timelapse_vod_id, %{owner_id: @timelapse_bot_id}),
          _result                                                             <- update_timelapse(timelapse, [clip_url: clip_url])
    do
      clip_url
    else
      result ->  result
    end
  end


  def generate_timelapse(filelist_url, frames) do
    with duration                <- FFprobe.duration(filelist_url) ,
         extraction_timestamps   <- get_extraction_timestamps(duration, frames),
         chunks                  <- Chunk.extract_chunks(filelist_url),
         extraction_points       <- get_extraction_points(chunks, extraction_timestamps),
         frame_paths             <- download_frames(extraction_points),
         {:ok, frame_diff_cache} <- Agent.start_link(fn -> %{} end),
         chosen_frame_paths      <- choose_frames(frame_paths, 810, frame_diff_cache),
         timelapse_path          <- concat_frames(frame_paths),
         _clean                  <- Enum.each(frame_paths, &File.rm/1)
    do
      timelapse_path
    end
  end

  def generate_title(channel) do
    {:ok, title_path} = Briefly.create([extname: ".mp4"])
    {_, 0} = System.cmd("ffmpeg",[
     "-y",
     "-f", "lavfi",
     "-i",
     "color=c=black:s=1280x800:d=2.0", "-vf",
     "drawtext=fontfile=assets/Korolev_bold.otf:fontsize=30:fontcolor=white:x=(w-text_w)/2:y=(h-text_h)/2:text='Recorded by #{channel} on Twitch'",
     "-r", "30",
     title_path
   ], stderr_to_stdout: true)
   title_path
  end

  def generate_credits(channel) do
    {:ok, credits_path} = Briefly.create([extname: ".mp4"])
    {_, 0} = System.cmd("ffmpeg",[
     "-y",
     "-f", "lavfi",
     "-i",
     "color=c=black:s=1280x800:d=1.0", "-vf",
     "drawtext=fontfile=assets/Korolev_bold.otf:fontsize=30:fontcolor=white:x=(w-text_w)/2:y=(h-text_h)/2:text='twitch.tv/#{channel}'",
     "-r", "30",
     credits_path
   ], stderr_to_stdout: true)
   credits_path
  end


  def concat_frames(frame_paths) do
    {:ok, timelapse_path} = Briefly.create([extname: ".mp4"])
    input_string = "concat:"<>Enum.join(frame_paths, "|")
     {_, 0} = System.cmd("ffmpeg",[
      "-framerate", "30",
      "-i", input_string,
      "-y",
      "-c:v", "libx264",
      "-preset", "fast",
      "-pix_fmt", "yuv420p",
      "-s", "1280x800",
      "-threads", "0",
      "-f", "flv",
      "-r", "30",
      "-fflags", "+genpts",
       timelapse_path
    ], stderr_to_stdout: true)
    # command = FFmpex.new_command
    # |> add_global_option(option_y)
    # |> add_input_file(input_string)
    # |> add_output_file(timelapse_path)
    #   |> add_file_option(option_vcodec("libx264"))
    #   |> add_file_option(option_preset("fast"))
    #   |> add_file_option(option_pix_fmt("yuv420p"))
    #   |> add_file_option(option_pix_fmt("yuv420p"))
    #   |> add_file_option(option_s("1280x800"))
    #   |> add_file_option(option_f("flv"))
    #   |> add_file_option(option_flags("+genpts"))
    # :ok = execute(command)
    timelapse_path
  end

  def choose_frames(frame_paths, length, _cache) when length(frame_paths) <= length, do: frame_paths
  def choose_frames(frame_paths, length, frame_diff_cache) do
    # {idx, {score, frame_path}}
    frames_with_differences_and_idx = add_differences_to_frames(frame_paths, frame_diff_cache)
      |> Enum.with_index
      |> Enum.sort_by(fn({{score, _}, _idx}) -> score end)
      |> Enum.drop(1)
      |> Enum.sort_by(fn({{_score, _}, idx}) -> idx end)
      |> Enum.map(fn({{_score, path}, _idx}) -> path end)
      |> choose_frames(length, frame_diff_cache)
  end

  def add_differences_to_frames(frame_paths, frame_diff_cache) do
    forward_neighbor_paths = Enum.drop(frame_paths, 1)
    backward_neighbor_paths = frame_paths
    frame_diffs = Enum.zip(forward_neighbor_paths, backward_neighbor_paths)
    |> Enum.with_index
    |> Flow.from_enumerable
    |> Flow.partition
    |> Flow.map(fn(args) -> compare_frames(args, frame_diff_cache) end)
    |> Enum.sort_by(fn({_, idx}) -> idx end)
    |> Enum.map(fn({result, _idx}) -> result end)


    forward_neighbor_diffs = [nil] ++ frame_diffs
    backward_neighbor_diffs = frame_diffs ++ [nil]

    Enum.zip(forward_neighbor_diffs, backward_neighbor_diffs)
      |> Enum.map(&calculate_difference/1)
      |> Enum.zip(frame_paths)
  end

  def compare_frames({{f_neighboor, b_neighboor} = cache_key, idx}, frame_diff_cache) do
    result = case Agent.get(frame_diff_cache, fn(cache) -> Map.get(cache, cache_key) end) do
      nil -> do_compare_frames({f_neighboor, b_neighboor})
      result -> result
    end
    Agent.update(frame_diff_cache, fn(cache) -> Map.put(cache, cache_key, result) end)
    {result, idx}
  end

  def do_compare_frames({f_neighboor, b_neighboor}) do
    {response, 1} = System.cmd("compare",[
      "-metric", "PSNR",
      f_neighboor, b_neighboor, "/dev/null"
   ], stderr_to_stdout: true)
   case Float.parse(response) do
     :error -> 999999999999
     {val, _} -> val
   end
  end

  def calculate_difference({nil, b_neighboor}), do: b_neighboor * 2
  def calculate_difference({f_neighboor, nil}), do: f_neighboor * 2
  def calculate_difference({f_neighboor, b_neighboor}), do: f_neighboor + b_neighboor



  def concat_op_tl_ed(title_path, timelapse_path, credits_path) do
    {:ok, na_timelapse_path} = Briefly.create([extname: ".mp4"])
    {_, 0} = System.cmd("ffmpeg",[
     "-y",
     "-i", title_path,
     "-i", timelapse_path,
     "-i", credits_path,
     "-filter_complex", "[0:0] [1:0] [2:0] concat=n=3:v=1 [v]",
     "-map", "[v]",
     "-c:v", "libx264",
     "-preset", "fast",
     "-pix_fmt", "yuv420p",
     "-s", "1280x800",
     "-threads", "0",
     "-f", "flv",
     "-r", "30",
     "-fflags", "+genpts",
     na_timelapse_path
   ], stderr_to_stdout: true)
   na_timelapse_path
  end

  def add_audio(concat_timelapse_path) do
    {:ok, full_timelapse_path} = Briefly.create([extname: ".mp4"])
    {_, 0} = System.cmd("ffmpeg",[
      "-y",
      "-t", "30",
      "-i", concat_timelapse_path,
      "-t", "30",
      "-i", "assets/timelapse_audio.mp3",
      "-c", "copy",
      "-map", "0:v:0",
      "-map", "1:a:0",
      "-c:v", "libx264",
      "-preset", "fast",
      "-pix_fmt", "yuv420p",
      "-s", "1280x800",
      "-threads", "0",
      "-f", "flv",
      "-acodec", "aac",
      "-r", "30",
      "-fflags", "+genpts",
      full_timelapse_path
    ], stderr_to_stdout: true)
    full_timelapse_path
  end

  def download_frames(extraction_points) do
    extraction_points
    |> Flow.from_enumerable
    |> Flow.partition
    |> Flow.map(&extraction_point_to_frame/1)
    |> Enum.to_list
    |> Enum.sort_by(fn {idx, _} -> idx end)
    |> Enum.map(fn {_, file} -> file end)
    |> Enum.filter(&File.exists?/1)
  end

  def extraction_point_to_frame({chunk, ts, idx}) do
    {_mega, sec, micro} = :os.timestamp
    scheduler_id = :erlang.system_info(:scheduler_id)
    frame_path = "/tmp/#{sec}-#{micro}-#{scheduler_id}.png"
    #  System.cmd("ffmpeg",[
    #    "-i", chunk.url,
    #    "-ss", "#{ts}",
    #    "-vframes", "1",
    #    "-y",
    #     frame_path
    #  ])
    command = FFmpex.new_command
    |> add_global_option(option_y)
    |> add_input_file(chunk.url)
    |> add_output_file(frame_path)
      |> add_file_option(option_ss(round(ts)))
      |> add_file_option(option_vframes(1))
    :ok = execute(command)
    {idx, frame_path}
  end

  def  get_extraction_points(chunks, extraction_timestamps) do
    Enum.map(extraction_timestamps, &(get_extraction_point(chunks, &1)))
  end

  def get_extraction_point(chunks, ts) do
    too_far_idx = Enum.find_index(chunks, fn chunk -> chunk.duration > ts end) || Enum.count(chunks) - 1
    chunk = Enum.at(chunks, too_far_idx - 1)
    {chunk, ts - chunk.duration, too_far_idx - 1}
  end

  def get_extraction_timestamps(duration, frames) do
    take_every = duration / frames
    1..frames
    |> Enum.map(&(&1 * take_every))
  end


  def by_id(id) do
    from t in "timelapses", where: t.id == ^id
  end

  def update_timelapse(timelapse, attrs) do
    Timelapser.Repo.update_all(by_id(timelapse.id),
                    [set: attrs])
  end

  @doc """
  Builds a changeset based on the `struct` and `params`.
  """
  def changeset(struct, params \\ %{}) do
    struct
    |> cast(params, [:status, :clip_url, :building_blocks])
  end

  @doc """
  Builds a api_changeset based on the `struct` and `params`.
  """
  def api_changeset(struct, params \\ %{}) do
    struct
    |> cast(params, [:status, :clip_url])
    |> cast_assoc(:building_blocks)
    |> Map.put_new(:status, "waiting")
  end

  def get_by_vod_id(vod_id) do
    Repo.one!(from t in Timelapse,
      join: bb in assoc(t, :building_blocks),
      where: bb.vod_id == ^vod_id,
      preload: [building_blocks: bb])
  end
end

# Timelapser.Timelapse.build_timelapse(%{id: 1, building_blocks: [%{vod_id: "90997909"}]})
# Timelapser.Timelapse.generate_timelapse("http://vod006-ttvnw.akamaized.net/v1/AUTH_system/vods_2ddb/programming_23268928288_522983546/high/index-dvr.m3u8", 10)
