defmodule TwirpEx.Router do
  alias TwirpEx.{Protobufs, Parser, Status}
  use Plug.Router
  plug Plug.Parsers, parsers: [Parser], pass:  ["text/*", "application/*"]
  plug :match
  plug :dispatch

  for {{:service,service},rpcs} <- Protobufs.defs do
    for {:rpc, procedure, _arg, response_type, _, _,_} <- rpcs do
      namespace = Protobufs.namespace(service)
      procedure = Atom.to_string(procedure)
      response_type = Protobufs.modularize(response_type)

      post "/twirp/#{namespace}/#{procedure}" do
        arg = conn.params["_twirp"]
        case arg do
          nil ->
            send_error(conn, :invalid_argument, msg: "body is required")
          _ ->
            apply(unquote(Protobufs.modularize("#{namespace}.#{procedure}")), :run, [arg])
            |> case do
              {:ok, result, options} ->
                headers = Keyword.get(options, :headers, %{})

                conn = Enum.reduce(headers, conn, fn
                  {key, value}, conn -> Plug.Conn.put_resp_header(conn, key, value)
                end)

                resp_body = conn
                |> Plug.Conn.get_req_header("content-type")
                |> encode(unquote(response_type), result)

                send_resp(conn, :ok, resp_body)

              {:error, status, error_details} ->
                send_error(conn, status, error_details)
              result ->
                resp_body = conn
                |> Plug.Conn.get_req_header("content-type")
                |> encode(unquote(response_type), result)

                send_resp(conn, :ok, resp_body)
            end
        end
      end
    end
  end

  match _ do
    path = conn.path_info
    |> Enum.join("/")

    send_error(conn, :bad_route, msg: "no route found", meta: %{path: "/" <> path})
  end

  defp send_error(conn, status, error_details \\ %{}) do
    code = Status.code_for_status(status)

    resp_body = error_details
    |> Enum.into(%{})
    |> Map.merge(%{
      type: status
    })
    |> Poison.encode!()

    send_resp(conn, code, resp_body)
  end

  def encode(["application/protobuf"], response_type, result), do: response_type.encode(result)
  def encode(["application/json"], _, result), do: Poison.encode!(result)
end
