defmodule FakeRouter do
  use Plug.Router
  use TwirpEx.Router.Phoenix
  plug :match
  plug :dispatch

  post "/other" do
    send_resp(conn, 200, "untouched")
  end
end

defmodule TwirpEx.RouterPhoenixTest do
  use ExUnit.Case, async: true
  use Plug.Test
  alias Code.Justin.Tv.Example.Haberdasher.{Hat, Size}

  @opts FakeRouter.init([])

  test "does not intercept non-twirp routes" do
    conn = conn(:post, "/other")

    conn = FakeRouter.call(conn, @opts)

    assert conn.state == :sent
    assert conn.status == 200
    assert conn.resp_body == "untouched"
  end

  test "invalid_argument if no body" do
    conn = conn(:post, "/twirp/code.justin.tv.example.haberdasher.Haberdasher/MakeHat")

    conn = FakeRouter.call(conn, @opts)

    assert conn.state == :sent
    assert conn.status == 400
    assert conn.resp_body == Poison.encode!(%{
      type: "invalid_argument",
      msg: "body is required",
    })
  end

  test "responds to json with json" do
    params = Poison.encode!(%{"inches" => 10})
    conn = conn(:post, "/twirp/code.justin.tv.example.haberdasher.Haberdasher/MakeHat", params)
    |> put_req_header("content-type", "application/json")

    conn = FakeRouter.call(conn, @opts)

    assert conn.state == :sent
    assert conn.status == 200
    assert conn.resp_body == Poison.encode!(%{
      "name" => "bowler",
      "inches" => 10,
      "color" => "yellow"
      })
  end

  test "respons to protobuf with protobuf" do
    params = Size.encode(%Size{inches: 10})
    conn = conn(:post, "/twirp/code.justin.tv.example.haberdasher.Haberdasher/MakeHat", params)
    |> put_req_header("content-type", "application/protobuf")

    conn = FakeRouter.call(conn, @opts)

    assert conn.state == :sent
    assert conn.status == 200
    assert Hat.decode(conn.resp_body) == %Hat{
      name: "bowler",
      inches: 10,
      color: "yellow"
      }
  end
end
