defmodule TwirpEx.RouterTest do
  use ExUnit.Case, async: true
  use Plug.Test
  alias Code.Justin.Tv.Example.Haberdasher.{Hat, Size, Errah, CacheSize}

  alias TwirpEx.Router

  @opts Router.init([])

  test "invalid_argument if no body" do
    conn = conn(:post, "/twirp/code.justin.tv.example.haberdasher.Haberdasher/MakeHat")

    conn = Router.call(conn, @opts)

    assert conn.state == :sent
    assert conn.status == 400
    assert conn.resp_body == Poison.encode!(%{
      type: "invalid_argument",
      msg: "body is required",
    })
  end

  test "responds to json with json" do
    params = Poison.encode!(%{"inches" => 10})
    conn = conn(:post, "/twirp/code.justin.tv.example.haberdasher.Haberdasher/MakeHat", params)
    |> put_req_header("content-type", "application/json")

    conn = Router.call(conn, @opts)

    assert conn.state == :sent
    assert conn.status == 200
    assert conn.resp_body == Poison.encode!(%{
      "name" => "bowler",
      "inches" => 10,
      "color" => "yellow"
      })
  end

  test "responds to protobuf with protobuf" do
    params = Size.encode(%Size{inches: 10})
    conn = conn(:post, "/twirp/code.justin.tv.example.haberdasher.Haberdasher/MakeHat", params)
    |> put_req_header("content-type", "application/protobuf")

    conn = Router.call(conn, @opts)

    assert conn.state == :sent
    assert conn.status == 200
    assert Hat.decode(conn.resp_body) == %Hat{
      name: "bowler",
      inches: 10,
      color: "yellow"
      }
  end

  test "responds with cache header to json with json" do
    params = Poison.encode!(%{"inches" => 10, "seconds" => 60})
    conn = conn(:post, "/twirp/code.justin.tv.example.haberdasher.Haberdasher/MakeCachedHat", params)
    |> put_req_header("content-type", "application/json")

    conn = Router.call(conn, @opts)

    assert conn.state == :sent
    assert conn.status == 200
    assert conn.resp_headers == [{"cache-control", "public, max-age=60"}]
    assert conn.resp_body == Poison.encode!(%{
      "name" => "bowler",
      "inches" => 10,
      "color" => "yellow"
      })
  end

  test "responds with cache header protobuf with protobuf" do
    params = CacheSize.encode(%CacheSize{inches: 10, seconds: 60})
    conn = conn(:post, "/twirp/code.justin.tv.example.haberdasher.Haberdasher/MakeCachedHat", params)
    |> put_req_header("content-type", "application/protobuf")

    conn = Router.call(conn, @opts)

    assert conn.state == :sent
    assert conn.status == 200
    assert conn.resp_headers == [{"cache-control", "public, max-age=60"}]
    assert Hat.decode(conn.resp_body) == %Hat{
      name: "bowler",
      inches: 10,
      color: "yellow"
      }
  end

  test "responds to a bad route with a bad route error" do
    params = Size.encode(%Size{inches: 10})
    conn = conn(:post, "/twirp/code.justin.tv.example.haberdasher.Haberdasher/NotARoute", params)
    |> put_req_header("content-type", "application/protobuf")

    conn = Router.call(conn, @opts)


    assert conn.state == :sent
    assert conn.status == 404
    assert conn.resp_body == Poison.encode!(%{
      type: "bad_route",
      msg: "no route found",
      meta: %{
        path: "/twirp/code.justin.tv.example.haberdasher.Haberdasher/NotARoute"
      }
    })
  end

  test "responds to json with json error" do
    errors = [ canceled: 408,
      unknown: 500,
      invalid_argument: 400,
      deadline_exceeded: 408,
      not_found: 404,
      bad_route: 404,
      already_exists: 409,
      permission_denied: 403,
      unauthenticated: 401,
      resource_exhaused: 403,
      failed_precondition: 412,
      aborted: 409,
      out_of_range: 400,
      unimplemented: 501,
      internal: 500,
      unavailable: 503,
      dataloss: 500,
    ]
    Enum.each(errors, fn
      {error, code} ->
        error = Atom.to_string(error)
        perform_json_error_request(error)
        |> assert_error_response(error, code)

        perform_protobuf_error_request(error)
        |> assert_error_response(error, code)
    end)
  end

  defp perform_json_error_request(error_name) do
    Poison.encode!(%{error_name: error_name})
    |> perform_error_request("application/json")
  end

  defp perform_protobuf_error_request(error_name) do
    Errah.encode(%Errah{error_name: error_name})
    |> perform_error_request("application/protobuf")
  end

  defp perform_error_request(params, content_type) do
    :post
    |> conn("/twirp/code.justin.tv.example.haberdasher.Haberdasher/MakeErrah", params)
    |> put_req_header("content-type", content_type)
    |> Router.call(@opts)
  end

  defp assert_error_response(conn, error_name, status_code) do
    assert conn.state == :sent
    assert conn.status == status_code, "expected '#{error_name}' to return status: #{status_code}"
    assert conn.resp_body == Poison.encode!(%{
      type: error_name,
      msg: "message for #{error_name}",
      meta: %{
        error_cause: "cause for #{error_name}"
      }
    })
  end
end
