'''
Transcode a set of video files for 24/7 stream with subtitles burned in.
'''

from __future__ import print_function
import argparse
import os
import re
import sys
import subprocess
from transcode import is_video


def ffmpeg_split(input, output, start=None, end=None):
    '''
    Export a portion of the input video (between start and end) into
    the output video filename.
    '''
    command = [
        'ffmpeg',
        '-i', input,
        '-c', 'copy',
        output
    ]
    if start:
        command = [command[0], '-ss', str(start)] + command[1:]
    if end:
        command = command[:-1] + ['-t', str(end - start), command[-1]]
    print('[CALL] `%s`' % subprocess.list2cmdline(command))
    subprocess.check_output(command)


def ffmpeg_concat(input, output):
    '''
    Merge a number of videos (`input` provided as a ffmpeg file list)
    into a single `output` video.
    '''
    command = [
        'ffmpeg',
        '-f', 'concat',
        '-i', input,
        '-c', 'copy',
        output
    ]
    print('[CALL] `%s`' % subprocess.list2cmdline(command))
    subprocess.check_output(command)


def split(video, splits):
    '''
    Split video into splits length videos named video-X where X is the split
    index starting from 1. Return a list of each partial video created.
    '''
    parts = []
    for index, split in enumerate(splits):
        parts.append(('-%d' % (index + 1)).join(os.path.splitext(video)))
        ffmpeg_split(video, parts[-1], start=split[0], end=split[1])
    return parts


def join(splits, output):
    '''
    Concat a ffmpeg list file into a single video

        ffmpeg -f concat -i CONCAT_FILE \
          -c copy OUTPUT_FILE
    '''
    concat_file = '%s.concat' % video
    with open(concat_file, 'w') as f:
        f.write(''.join([('file \'%s\'\n' % f) for f in splits]))
    ffmpeg_concat(concat_file, output)


def find_darkness(video):
    '''
    Return list of tuples that represent the start and end of each section
    of darkness such as [(start, end), (start, end)]

        ffmpeg -loglevel 32 -i FILENAME \
          -vf blackdetect=d=2 \
          -an -f null - 2>&1 | grep "blackdetect"
    '''
    command = [
        'ffmpeg',
        '-loglevel', '32',
        '-i', video,
        '-vf', 'blackdetect=d=2',
        '-an', '-f', 'null', '-'
    ]
    print('[CALL] `%s`' % subprocess.list2cmdline(command))
    output = subprocess.check_output(command, stderr=subprocess.STDOUT)
    res = []
    for detection in [i for i in output.split('\r') if i.startswith('[black')]:
        matches = re.search('black_start:(\d+.?\d+?) black_end:(\d+.?\d+?) '
                            'black_duration:(\d+.?\d+?).*', detection)
        if matches:
            res.append((float(matches.group(1)), float(matches.group(2))))
    return res


def reverse_splits(splits):
    '''
    Return the opposite time periods than provided.
    '''
    res = []
    next_start = 0
    for i, split in enumerate(splits):
        res.append((next_start, split[0]))
        next_start = split[1]
    res.append((next_start, None))
    return res


def remove_darkness(video):
    '''
    Find and remove all moments of darkness (probably commercials) in the
    supplied video file
    '''
    print('[STARTING] %s' % video)
    darkness = find_darkness(video)
    if darkness:
        splits = reverse_splits(darkness)
        parts = split(video, splits)
        join(parts, '_trimmed'.join(os.path.splitext(video)))


def config():
    '''
    Configure script from args
    '''
    parser = argparse.ArgumentParser()
    parser.add_argument('path', help='path to video files')
    parser.add_argument('-r', '--recursive', help='find videos in dirs',
                        action='store_true')
    parser.add_argument('-v', '--verbose', help='display verbose information',
                        action='store_true')
    return parser.parse_args()


if __name__ == '__main__':
    args = config()
    files = []
    if os.path.isfile(args.path):
        files.append(args.path)
    if os.path.isdir(args.path):
        if args.recursive:
            for dirpath, dirnames, filenames in os.walk(args.path):
                for filename in [f for f in filenames if is_video(f)]:
                    files.append(os.path.join(dirpath, filename))
        else:
            videos = [f for f in os.listdir(args.path) if is_video(f)]
            for filename in videos:
                files.append(os.path.join(args.path, filename))
    for video in files:
        remove_darkness(video)
