'''
Transcode a set of video files for 24/7 stream with subtitles burned in.
'''

from __future__ import print_function
import argparse
import os
import re
import sys
import subprocess


SUPPORTED_VIDEO_EXTENSIONS = ['.mpg', '.mov', '.avi', '.mp4', '.wmv']


def srt2ass(srt):
    '''
    Convert subtitles to styled ass format.

        ffmpeg -i $episode.srt -f ass - | \
        sed 's/16,&Hffffff,&Hffffff/10,\&H999999,\&H999999/' > $episode.ass
    '''
    ass = srt.replace('.srt', '.ass')
    command = [
        'ffmpeg', '-i', srt, '-f', 'ass', '-',
    ]
    with open(ass, 'w') as output, open(os.devnull) as devnull:
        subtitles = subprocess.check_output(command, stderr=devnull)
        subtitles = subtitles.replace('16,&Hffffff,&Hffffff',
                                      '10,&H999999,&H999999')
        output.write(subtitles)
        return ass
    return None


def ffmpeg_transcode(video, subtitles=None):
    '''
    Transcode video file into stream-ready format.

    ffmpeg -i $i -vcodec libx264 -profile:v main -preset:v medium \
        -r 30 -g 60 -keyint_min 60 -sc_threshold 0 -b:v 4000k \
        -maxrate 4000k -bufsize 4000k \
        -filter:v scale="trunc(oh*a/2)*2:720" \
        -sws_flags lanczos+accurate_rnd -acodec aac -b:a 96k -ar 48000 \
        -ac 2 `basename $i .mpg`.mp4
    '''
    filename, extension = os.path.splitext(video)
    command = [
        'ffmpeg',
        '-i', video,
        '-vcodec', 'libx264',
        '-profile:v', 'main',
        '-preset:v', 'medium',
        '-r', '30',
        '-g', '60',
        '-keyint_min', '60',
        '-sc_threshold', '0',
        '-b:v', '3000k',
        '-minrate', '3000k',
        '-maxrate', '3000k',
        '-bufsize', '3000k',
        '-filter:v', 'scale=\'trunc(oh*a/2)*2:720\'',
        '-pix_fmt', 'yuv420p',
        '-sws_flags', 'lanczos+accurate_rnd',
        '-acodec', 'aac',
        '-b:a', '96k',
        '-ar', '48000',
        '-ac', '2',
        '%s.mp4' % filename
    ]
    if subtitles is not None:
        subtitles = srt2ass(subtitles)
        command = (command[:-1] + ['-vf', 'subtitles=%s' % subtitles] +
                   ['_titled.'.join(command[-1].split('.'))])
    print('[CALL] `%s`' % subprocess.list2cmdline(command))
    return subprocess.Popen(command,
                            stdout=subprocess.PIPE,
                            stderr=subprocess.STDOUT)


def ffmpeg_burn_subtitles(video, subtitles):
    '''
    Burn in provided subtitles to the video.

        ffmpeg -i $episode.srt -f ass - | \
        sed 's/16,&Hffffff,&Hffffff/10,\&H999999,\&H999999/' \
        > $episode.ass

    '''
    subtitles = srt2ass(subtitles)
    output = video.replace('.mp4', '_titled.mp4')
    command = [
        'ffmpeg', '-i', video,
        '-vf', 'subtitles=%s' % subtitles,
        output
    ]
    return subprocess.Popen(command,
                            stdout=subprocess.PIPE,
                            stderr=subprocess.STDOUT)


def requires_transcode(video):
    '''
    Return true if video is not a supported video file
    TODO: This needs to check the actual encoding and details to ensure
    the file is playable regardless off the filename
    '''
    return not video.endswith('mp4')


def subtitles_for_video(video):
    '''
    Return a subtitles file for the provided video
    # TODO: Support multiple subtitle formats
    '''
    name, extension = os.path.splitext(video)
    subtitles = ''.join([name, '.srt'])
    if os.path.isfile(subtitles):
        return subtitles
    return None


def transcode(video):
    '''
    Transcode video file into stream ready format and burn in subtitles.
    '''
    subtitles = subtitles_for_video(video)
    proc = None
    if requires_transcode(video):
        print('[TRANSCODE %s SUBTITLES] %s' % (
            subtitles and 'with' or 'without', video))
        return ffmpeg_transcode(video, subtitles)
    elif subtitles is not None:
        print('[BURN SUBTITLES] %s' % video)
        return ffmpeg_burn_subtitles(video, subtitles)
    print('[NO OPERATION] %s' % video)
    return None


def is_video(video):
    '''
    Return True when path points to a video file
    '''
    extension = os.path.splitext(video)[-1]
    return extension in SUPPORTED_VIDEO_EXTENSIONS


def config():
    '''
    Configure script from args
    '''
    parser = argparse.ArgumentParser()
    parser.add_argument('path', help='path to video files')
    parser.add_argument('-l', '--limit', help='limit concurrent processes',
                        type=int)
    parser.add_argument('-r', '--recursive', help='find videos in dirs',
                        action='store_true')
    parser.add_argument('-v', '--verbose', help='display verbose information',
                        action='store_true')
    return parser.parse_args()


if __name__ == '__main__':
    args = config()
    files = []
    if os.path.isfile(args.path):
        files.append(args.path)
    if os.path.isdir(args.path):
        if args.recursive:
            for dirpath, dirnames, filenames in os.walk(args.path):
                for filename in [f for f in filenames if is_video(f)]:
                    files.append(os.path.join(dirpath, filename))
        else:
            videos = [f for f in os.listdir(args.path) if is_video(f)]
            for filename in videos:
                files.append(os.path.join(args.path, filename))
    encoding_limit = args.limit or 10
    to_encode = files
    currently_encoding = []
    while (to_encode or currently_encoding):
        if to_encode and len(currently_encoding) < encoding_limit:
            video = to_encode.pop()
            video_process = transcode(video)
            if video_process is not None:
                print('[STARTING] %s' % os.path.basename(video))
                currently_encoding.append(video_process)
        for process in currently_encoding:
            nextline = process.stdout.readline()
            if args.verbose:
                sys.stdout.write(nextline)
                sys.stdout.flush()
            if nextline == '' and process.poll() is not None:
                print('[FINISHED] %s' % os.path.basename(video))
                currently_encoding.remove(process)
