'''
Validate that all videos in the provided path are valid for streaming
with `ffmpeg` and provide warning and issues for those that are not
'''

from __future__ import print_function
import argparse
import datetime
import os
import re
import subprocess
import sys


TOTAL_TIME = datetime.timedelta(seconds=0)
VALID_VIDEOS = 0
INVALID_VIDEOS = 0


def ffprobe_property(video, prop):
    '''
    Return property from ffprobe on video file. Currently supports

    - playable: True if ffpronbe can read video
    - duration: Duration string hh:mm:ss.ms from ffprobe
    '''
    command = [
        'ffprobe',
        video
    ]
    try:
        output = subprocess.check_output(command, stderr=subprocess.STDOUT)
    except subprocess.CalledProcessError, e:
        return None
    if prop == 'playable':
        return True
    elif prop == 'duration':
        matches = re.search(".*Duration: (.*), start:.*", output)
        if matches:
            return matches.group(1)


def is_playable_video(video):
    '''
    Return True if ffprobe believes this is a playable video
    '''
    return ffprobe_property(video, 'playable') is not None


def video_duration(video):
    '''
    Return a timedelta of the video's duration
    '''
    duration = ffprobe_property(video, 'duration')
    if duration:
        (hours, minutes, _seconds) = duration.split(':')
        seconds, microseconds = _seconds.split('.')
        return datetime.timedelta(hours=int(hours),
                                  minutes=int(minutes),
                                  seconds=int(seconds),
                                  microseconds=int(microseconds))
    return None


def validate(video, expected_length=1500, verbose=False):
    '''
    Test and output validation messages related to provided video.
    The video will test against expected length
    '''
    global VALID_VIDEOS
    global INVALID_VIDEOS
    global TOTAL_TIME
    warnings = []
    normalized_video = os.path.basename(video)
    if is_playable_video(video) is False:
        print('INVALID: %s is not a playable video' % normalized_video,
              file=sys.stderr)
        return
    duration = video_duration(video)
    if duration is None:
        warnings.append('- Duration not found')
    elif duration.total_seconds() < expected_length:  # More than 25 minutes
        warnings.append('- Duration: %s' % duration.total_seconds())
    TOTAL_TIME += duration
    if warnings:
        INVALID_VIDEOS += 1
        print('INVALID: %s\n%s' % (normalized_video, '\n    '.join(warnings)),
              file=sys.stderr)
    else:
        VALID_VIDEOS += 1
        if verbose:
            print('VALID: %s [%s]' % (normalized_video, duration))


def config():
    '''
    Configure script from args
    '''
    parser = argparse.ArgumentParser()
    parser.add_argument('path', help='path to video files')
    parser.add_argument('-l', '--length', type=int,
                        help='expected episode length in seconds')
    parser.add_argument('-r', '--recursive', help='find videos in dirs',
                        action='store_true')
    parser.add_argument('-v', '--verbose', help='display verbose information',
                        action='store_true')
    return parser.parse_args()


if __name__ == '__main__':
    args = config()
    expected_length = args.length or 1500
    if os.path.isfile(args.path):
        validate(args.path, expected_length=expected_length,
                 verbose=args.verbose)
    if os.path.isdir(args.path):
        if args.recursive:
            subtitle_path = os.path.expanduser(args.path)
            for dirpath, dirnames, filenames in os.walk(subtitle_path):
                for filename in filenames:
                    if filename.endswith('.mp4'):
                        validate(os.path.join(dirpath, filename),
                                 expected_length=expected_length,
                                 verbose=args.verbose)
        else:
            videos = [f for f in os.listdir(args.path) if f.endswith('.mp4')]
            for filename in videos:
                validate(os.path.join(args.path, filename),
                         expected_length=expected_length, verbose=args.verbose)
    average_length = int(TOTAL_TIME.total_seconds() / len(videos))
    print('Valid videos: %d' % VALID_VIDEOS)
    print('Invalid videos: %d' % INVALID_VIDEOS)
    print('Average length: %d seconds' % average_length)
    print('Total play time: %d seconds' % TOTAL_TIME.total_seconds())
