package log

import (
	"bytes"
	"fmt"
	"testing"
	"time"

	"code.justin.tv/creator-collab/log/errors"
)

type logEventCase struct {
	desc     string
	logEvent LogEvent
}

func getLogEvents() []logEventCase {
	return []logEventCase{
		{
			desc:     "an empty log event",
			logEvent: LogEvent{},
		},
		{
			desc: "a typical info log event",
			logEvent: LogEvent{
				Message: "starting service",
				Level:   LevelInfo,
				Time:    time.Date(2019, 2, 21, 3, 30, 0, 0, time.UTC),
				Fields: map[string]interface{}{
					"numWorkers": 2,
				},
			},
		},
		{
			desc: "a typical error log event",
			logEvent: LogEvent{
				Error: errors.Wrap(
					fmt.Errorf("db operation failed"),
					"api operation failed",
					errors.Fields{
						"id": "3243434",
					}),
				Level: LevelError,
				Time:  time.Date(2019, 2, 21, 3, 30, 0, 0, time.UTC),
				Fields: map[string]interface{}{
					"target": "222222",
				},
			},
		},
		{
			desc: "an info event with lots of types of fields",
			logEvent: LogEvent{
				Message: "msg",
				Level:   LevelInfo,
				Time:    time.Date(2019, 2, 21, 3, 30, 0, 0, time.UTC),
				Fields: map[string]interface{}{
					"strField":   "value1",
					"intField":   123,
					"nilField":   nil,
					"errField":   fmt.Errorf("an error"),
					"mapField":   map[string]interface{}{"nestedField": "nestedValue"},
					"sliceField": []string{"value1, value2, value3 "},
				},
			},
		},
		{
			desc: "an info event with a field that has spaces in the name",
			logEvent: LogEvent{
				Message: "msg",
				Level:   LevelInfo,
				Time:    time.Date(2019, 2, 21, 3, 30, 0, 0, time.UTC),
				Fields: map[string]interface{}{
					"field with spaces": "value1",
				},
			},
		},
	}
}

var testInfoEvent = &LogEvent{
	Message: "starting service",
	Level:   LevelInfo,
	Time:    time.Date(2019, 2, 21, 3, 30, 0, 0, time.UTC),
	Fields: map[string]interface{}{
		"numWorkers": 2,
	},
}

var testErrorEvent = &LogEvent{
	Error: fmt.Errorf("db operation failed"),
	Level: LevelError,
	Time:  time.Date(2019, 2, 21, 3, 30, 0, 0, time.UTC),
	Fields: map[string]interface{}{
		"target": "222222",
	},
}

func TestDevLogEventWriter(t *testing.T) {
	t.Run("log should write, and not panic", func(t *testing.T) {
		for _, logEventCase := range getLogEvents() {
			t.Run(logEventCase.desc, func(t *testing.T) {
				b := &bytes.Buffer{}
				logWriter := newDevLogEventWriter(b)
				logWriter.log(&logEventCase.logEvent)
			})
		}
	})
}
