package errors

import (
	"fmt"
	"strconv"
	"testing"

	"github.com/rollbar/rollbar-go"
	"github.com/stretchr/testify/require"
)

// TestRollbarCompatibility verifies that rollbar-go will recognize the stack traces attached to debugErrors.
// If rollbar-go doesn't recognize debugError's stack trace, then it attaches its own, which gives us
// a stack trace that points to where we log, instead of where the error was created.
func TestRollbarCompatibility(t *testing.T) {
	var _ rollbar.CauseStacker = &debugError{}
}

func TestNew(t *testing.T) {
	t.Run("should return an error, when given a message", func(t *testing.T) {
		err := New("query failed")
		require.Error(t, err)
		require.Contains(t, err.Error(), "query failed")
	})
}

func TestWrap(t *testing.T) {
	t.Run("should return nil, when given an nil error", func(t *testing.T) {
		outerErrMessage := "hosting channel failed"
		err := Wrap(nil, outerErrMessage)
		require.NoError(t, err)
	})

	t.Run("should return an error, when given an error and a message", func(t *testing.T) {
		innerErrMessage := "query failed"
		innerErr := fmt.Errorf(innerErrMessage)

		outerErrMessage := "hosting channel failed"
		err := Wrap(innerErr, outerErrMessage)
		require.Error(t, err)
		require.Contains(t, err.Error(), innerErrMessage)
		require.Contains(t, err.Error(), outerErrMessage)
	})

	t.Run("should return an error, when given an error, a message, and fields", func(t *testing.T) {
		innerErrMessage := "query failed"
		innerErr := fmt.Errorf(innerErrMessage)

		outerErrMessage := "hosting channel failed"
		err := Wrap(innerErr, outerErrMessage, Fields{
			"sourceID": "1111111",
			"targetID": "2222222",
		})
		require.Error(t, err)
		require.Contains(t, err.Error(), innerErrMessage)
		require.Contains(t, err.Error(), outerErrMessage)
	})

	t.Run("should return an error, when given an error, a message, and a nil field", func(t *testing.T) {
		innerErrMessage := "query failed"
		innerErr := fmt.Errorf(innerErrMessage)

		outerErrMessage := "hosting channel failed"
		err := Wrap(innerErr, outerErrMessage, nil)
		require.Error(t, err)
		require.Contains(t, err.Error(), innerErrMessage)
		require.Contains(t, err.Error(), outerErrMessage)
	})
}

func TestFormattingDetails(t *testing.T) {
	t.Run("formatting with %+v should give the error message, stack trace, and fields", func(t *testing.T) {
		err := followChannel("", "111111")
		require.Error(t, err)

		details := fmt.Sprintf("%+v", err)

		// Details should contain the error message
		require.Contains(t, details, paramEmptyErrorMessage)

		// Details should contain the function names in the stack trace, from the point where the  error is
		// created.
		require.Contains(t, details, "followChannel")

		// Details should contain the fields in the outer error.
		require.Contains(t, details, "param")
		require.Contains(t, details, "sourceID")
	})

	t.Run("formatting with verbose verb should not panic", func(t *testing.T) {
		for _, c := range errorTestCases() {
			t.Run(c.desc, func(*testing.T) {
				details := fmt.Sprintf("%+v", c.err)
				require.NotEmpty(t, details)
			})
		}
	})
}

func TestError(t *testing.T) {
	t.Run("calling Error on edge cases should not panic", func(t *testing.T) {
		for _, c := range errorTestCases() {
			if c.err == nil {
				continue
			}

			t.Run(c.desc, func(*testing.T) {
				_ = c.err.Error()
			})
		}
	})
}

func TestCause(t *testing.T) {
	t.Run("calling Cause on a wrapped error should return the underlying error", func(t *testing.T) {
		innerErr := fmt.Errorf("inner error")
		err := Wrap(innerErr, "outer error")

		causeErr := Cause(err)

		require.Equal(t, innerErr, causeErr)
	})

	t.Run("calling Cause on an error that doesn't implement the Causer interface should return nil", func(t *testing.T) {
		causeErr := Cause(fmt.Errorf("error"))
		require.Nil(t, causeErr)
	})

	// Cause() is expected to follow rollbar-go's semantics, instead of pkg/error
	t.Run("calling Cause on an error that wraps a wrapped error should return the wrapped error", func(t *testing.T) {
		err1 := fmt.Errorf("err1")
		err2 := Wrap(err1, "err2")
		err3 := Wrap(err2, "err3")

		causeErr := Cause(err3)

		require.Equal(t, err2, causeErr)
	})

	t.Run("calling Cause on edge cases should not panic", func(t *testing.T) {
		for _, c := range errorTestCases() {
			t.Run(c.desc, func(*testing.T) {
				_ = Cause(c.err)
			})
		}
	})
}

func TestUnwrap(t *testing.T) {
	t.Run("calling Unwrap on a wrapped error should return the underlying error", func(t *testing.T) {
		innerErr := fmt.Errorf("inner error")
		err := Wrap(innerErr, "outer error")

		causeErr := Unwrap(err)

		require.Equal(t, innerErr, causeErr)
	})

	t.Run("calling Unwrap on an error that doesn't have a Unwrap() method return nil", func(t *testing.T) {
		causeErr := Unwrap(fmt.Errorf("error"))
		require.Nil(t, causeErr)
	})

	// Cause() is expected to follow rollbar-go's semantics, instead of pkg/error
	t.Run("calling Unwrap on an error that wraps a wrapped error should return the wrapped error", func(t *testing.T) {
		err1 := fmt.Errorf("err1")
		err2 := Wrap(err1, "err2")
		err3 := Wrap(err2, "err3")

		causeErr := Unwrap(err3)

		require.Equal(t, err2, causeErr)
	})

	t.Run("calling Unwrap on edge cases should not panic", func(t *testing.T) {
		for _, c := range errorTestCases() {
			t.Run(c.desc, func(*testing.T) {
				_ = Unwrap(c.err)
			})
		}
	})
}

// httpError is an error with custom fields that we will use for testing Is and As.
type httpError struct {
	code int
	msg  string
}

func (h *httpError) Error() string {
	return fmt.Sprintf("code: %d, message: %s", h.code, h.msg)
}

// httpError implements the error interface.
var _ error = &httpError{}

func TestAs(t *testing.T) {
	t.Run("As should work with this package's wrapped errors", func(t *testing.T) {
		var err error
		err = &httpError{
			code: 500,
			msg:  "Internal Server Error",
		}
		wrappedError := Wrap(err, "operation failed")

		var unwrappedError *httpError
		ok := As(wrappedError, &unwrappedError)
		require.True(t, ok, "As() should have succeeded and returned true")
		require.Equal(t, err, unwrappedError)
	})
}

func TestIs(t *testing.T) {
	t.Run("Is should work with this package's wrapped errors", func(t *testing.T) {
		var err error
		err = &httpError{
			code: 500,
			msg:  "Internal Server Error",
		}
		wrappedError := Wrap(err, "operation failed")

		ok := Is(wrappedError, err)
		require.True(t, ok, "Is() should have succeeded and returned true")
	})
}

func TestUnwrapAll(t *testing.T) {
	t.Run("calling UnwrapAll on a nil error should return nil", func(t *testing.T) {
		unwrappedErr := UnwrapAll(nil)
		require.Nil(t, unwrappedErr)
	})

	t.Run("calling UnwrapAll on an error that doesn't implement Causer should return that error", func(t *testing.T) {
		err := fmt.Errorf("an error that does not implement the Causer interface")
		unwrappedErr := UnwrapAll(err)
		require.Equal(t, err, unwrappedErr)
	})

	t.Run("calling UnwrapAll on an error created by New should return that error", func(t *testing.T) {
		err := New("an error that implements the Causer interface, but does not have an inner error")
		unwrappedErr := UnwrapAll(err)
		require.Equal(t, err, unwrappedErr)
	})

	t.Run("calling UnwrapAll on a wrapped error should return the inner error", func(t *testing.T) {
		err := fmt.Errorf("an inner error")
		wrappedErr := Wrap(err, "a wrapped error")

		unwrappedErr := UnwrapAll(wrappedErr)
		require.Equal(t, err, unwrappedErr)
	})

	t.Run("calling UnwrapAll on an error chain should return the inner error", func(t *testing.T) {
		err := fmt.Errorf("an inner error")
		wrappedErr1 := Wrap(err, "middle error")
		wrappedErr2 := Wrap(wrappedErr1, "outermost error")

		unwrappedErr := UnwrapAll(wrappedErr2)
		require.Equal(t, err, unwrappedErr)
	})

	t.Run("calling Cause on edge cases should not panic", func(t *testing.T) {
		for _, c := range errorTestCases() {
			t.Run(c.desc, func(*testing.T) {
				_ = UnwrapAll(c.err)
			})
		}
	})
}

func TestMessage(t *testing.T) {
	t.Run("calling Message() on a wrapped error should return the wrapped error's message", func(t *testing.T) {
		innerErr := fmt.Errorf("inner error")
		err := Wrap(innerErr, "outer error")

		msg := Message(err)

		require.Equal(t, "outer error", msg)
	})

	t.Run("calling Message() on an error that doesn't implement the Messager interface should return an empty string", func(t *testing.T) {
		msg := Message(fmt.Errorf("error"))

		require.Empty(t, msg)
	})

	t.Run("calling Message() should not panic", func(t *testing.T) {
		for _, c := range errorTestCases() {
			t.Run(c.desc, func(*testing.T) {
				_ = Message(c.err)
			})
		}
	})
}

func TestFrames(t *testing.T) {
	t.Run("calling Frames() on a debugError should return stack frames", func(t *testing.T) {
		err := New("something failed")

		frames := Frames(err)

		require.NotEmpty(t, frames)
		firstFrame := frames[0]
		require.Contains(t, firstFrame.Function, "TestFrames")
	})

	t.Run("calling Frames() on an error that doesn't implement the Framer interface should return nil", func(t *testing.T) {
		frames := Frames(fmt.Errorf("error"))

		require.Nil(t, frames)
	})

	t.Run("calling Frames() should not panic", func(t *testing.T) {
		for _, c := range errorTestCases() {
			t.Run(c.desc, func(*testing.T) {
				_ = Frames(c.err)
			})
		}
	})
}

func TestGetFields(t *testing.T) {
	t.Run("calling GetFields() on a debugError should return stack frames", func(t *testing.T) {
		fields := map[string]interface{}{
			"field1": "value1",
		}
		err := New("something failed", fields)

		gotFields := GetFields(err)

		require.Equal(t, fields, gotFields)
	})

	t.Run("calling GetFields() on an error that doesn't implement the Fieldser interface should return nil", func(t *testing.T) {
		fields := GetFields(fmt.Errorf("error"))

		require.Nil(t, fields)
	})

	t.Run("calling GetFields() should not panic", func(t *testing.T) {
		for _, c := range errorTestCases() {
			t.Run(c.desc, func(*testing.T) {
				_ = GetFields(c.err)
			})
		}
	})
}

type errorCase struct {
	err  error
	desc string
}

// Get a slice of errors created in a variety of different ways.
func errorTestCases() []errorCase {
	return []errorCase{
		{
			err:  nil,
			desc: "nil error",
		},
		{
			err:  fmt.Errorf("something failed"),
			desc: "fmt.Errorf()",
		},
		// errors.New() cases
		{
			err:  New(""),
			desc: "New(), given empty error message",
		},
		{
			err:  New("something failed"),
			desc: "New(), given a message",
		},
		{
			err:  New("something failed", nil),
			desc: "New(), given a message, and nil fields",
		},
		{
			err:  New("something failed", Fields{"field1": "value1"}),
			desc: "New(), given a message, and fields",
		},
		{
			err: New("something failed", Fields{
				"strField":   "value1",
				"intField":   123,
				"nilField":   nil,
				"errField":   fmt.Errorf("an error"),
				"mapField":   map[string]interface{}{"nestedField": "nestedValue"},
				"sliceField": []string{"value1, value2, value3 "},
			}),
			desc: "New(), given a message, and fields with different types of values",
		},
		{
			err: New("something failed", Fields{
				"multi word field": "value1",
			}),
			desc: "New(), given a message, and a field whose keys has spaces",
		},
		// errors.Wrap cases
		{
			err:  Wrap(fmt.Errorf("inner err"), ""),
			desc: "Wrap(), given error, and empty message",
		},
		{
			err:  Wrap(fmt.Errorf("inner err"), ""),
			desc: "Wrap(), given error, and empty message",
		},
		{
			err:  Wrap(fmt.Errorf("inner err"), "outer err", nil),
			desc: "Wrap(), given error, message, and a nil field",
		},
		{
			err: Wrap(fmt.Errorf("inner err"), "outer err", Fields{
				"field1": "value1",
			}),
			desc: "Wrap(), given error, message, and a field map",
		},
		{
			err: Wrap(fmt.Errorf("inner err"), "outer err", Fields{
				"field1": "value1",
			}, nil),
			desc: "Wrap(), given error, message, a field map, and a nil field map",
		},
		{
			err: Wrap(fmt.Errorf("inner err"), "outer err",
				Fields{"field1": "value1"},
				Fields{"field2": "value2"}),
			desc: "Wrap(), given error, message, and multiple field maps",
		},
		{
			err: Wrap(fmt.Errorf("inner err"), "outer err",
				Fields{"": "value1"}),
			desc: "Wrap(), given error, message, and a field with no key",
		},
		{
			err: Wrap(fmt.Errorf("inner err"), "outer err",
				Fields{"field1": nil}),
			desc: "Wrap(), given error, message, and a field with a nil key",
		},
		{
			err: Wrap(New("inner err", Fields{"innerField1": "innerValue1"}),
				"outer err", Fields{"field1": "value1"}),
			desc: "Wrap(), given an inner error created by New, a message, and a field",
		},
		{
			err: Wrap(
				Wrap(fmt.Errorf("inner inner error"), "inner error"),
				"outer err",
				Fields{"field1": "value1"}),
			desc: "Wrap(), given an inner error created by Wrap, a message, and a field",
		},
	}
}

// followChannel is a fake function meant to generate errors for tests.
const paramEmptyErrorMessage = "required param was empty"
const followChannelFailed = "follow channel failed"

func followChannel(sourceID, targetID string) error {
	if sourceID == "" {
		return New(paramEmptyErrorMessage, Fields{
			"param": "sourceID",
		})
	}
	if targetID == "" {
		return New(paramEmptyErrorMessage, Fields{
			"param": "targetID",
		})
	}

	err := updateDB(sourceID, targetID)
	if err != nil {
		return Wrap(err, followChannelFailed)
	}

	return nil
}

const updateTableFailed = "updating follow table failed"

// updateDB is a fake function meant to generate low level errors for tests.
func updateDB(sourceID, targetID string) error {
	_, err := strconv.Atoi(sourceID)
	if err != nil {
		return Wrap(err, updateTableFailed, Fields{
			"sourceID": sourceID, "targetID": targetID})
	}

	_, err = strconv.Atoi(targetID)
	if err != nil {
		return Wrap(err, updateTableFailed, Fields{
			"sourceID": sourceID, "targetID": targetID})
	}

	return nil
}
