package desk

import (
	"context"
	"fmt"
	"net/http"
	"time"

	"github.com/pkg/errors"
)

const (
	createCaseStatName      = "case.create"
	updateCaseStatName      = "case.update"
	readCaseMessageStatName = "case.message.read"
)

// Case is a case in Desk.
type Case struct {
	ID           int                    `json:"id,omitempty"`
	ExternalID   string                 `json:"external_id,omitempty"`
	Type         string                 `json:"type,omitempty"`
	LabelAction  string                 `json:"label_action"`
	Labels       []string               `json:"labels,omitempty"`
	Status       string                 `json:"status,omitempty"`
	Subject      string                 `json:"subject,omitempty"`
	CustomFields map[string]interface{} `json:"custom_fields,omitempty"`
	Created      *time.Time             `json:"created_at,omitempty"`
	Updated      *time.Time             `json:"updated_at,omitempty"`
	Changed      *time.Time             `json:"changed_at,omitempty"`
}

// CaseMessage is the original message that a case was opened with.
type CaseMessage struct {
	Direction string     `json:"direction,omitempty"`
	From      string     `json:"from,omitempty"`
	To        string     `json:"to,omitempty"`
	Subject   string     `json:"subject,omitempty"`
	Body      string     `json:"body,omitempty"`
	Created   *time.Time `json:"created_at,omitempty"`
	Updated   *time.Time `json:"updated_at,omitempty"`
}

func (c *client) CreateCase(ctx context.Context, ca *Case, m *CaseMessage) (*Case, error) {
	params := struct {
		*Case
		Message *CaseMessage `json:"message,omitempty"`
	}{
		Case:    ca,
		Message: m,
	}

	var createdCase Case
	path := "/api/v2/cases"
	err := c.do(ctx, http.MethodPost, path, createCaseStatName, &params, &createdCase)
	return &createdCase, errors.Wrap(err, "could not create case")
}

func (c *client) UpdateCase(ctx context.Context, ca *Case) (*Case, error) {
	var updatedCase Case
	path := fmt.Sprintf("/api/v2/cases/%d", ca.ID)
	err := c.do(ctx, http.MethodPatch, path, updateCaseStatName, ca, &updatedCase)
	return &updatedCase, errors.Wrap(err, "could not update case")
}

func (c *client) CaseMessage(ctx context.Context, caseID int) (*CaseMessage, error) {
	var caseMessage CaseMessage
	path := fmt.Sprintf("/api/v2/cases/%d/message", caseID)
	err := c.do(ctx, http.MethodGet, path, readCaseMessageStatName, nil, &caseMessage)
	return &caseMessage, errors.Wrap(err, "could not retrieve case message")
}
