package desk

import (
	"context"
	"fmt"
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/stretchr/testify/suite"
)

const testCaseID = 1337

type caseSuite struct {
	suite.Suite
	server *httptest.Server
	client Client
}

func (s *caseSuite) SetupSuite() {
	s.server = createServer()

	var err error
	s.client, err = createClient(s.server.URL)
	s.NoError(err)
}

func (s *caseSuite) TearDownSuite() {
	s.server.Close()
}

func (s *caseSuite) TestCreateCase() {
	c := &Case{
		Type:   TypeEmail,
		Labels: []string{"Awesome Label"},
	}

	m := &CaseMessage{
		Direction: DirectionIn,
		From:      "me@jordanpotter.com",
		To:        "support@twitch.tv",
		Subject:   "Test Test Test",
		Body:      "Test by Jordan Potter",
	}

	ctx := context.Background()
	_, err := s.client.CreateCase(ctx, c, m)
	s.NoError(err)
}

func (s *caseSuite) TestUpdateCase() {
	c := &Case{
		ID:     testCaseID,
		Status: StatusNew,
	}

	ctx := context.Background()
	_, err := s.client.UpdateCase(ctx, c)
	s.NoError(err)
}

func (s *caseSuite) TestCaseMessage() {
	ctx := context.Background()
	_, err := s.client.CaseMessage(ctx, testCaseID)
	s.NoError(err)
}

func TestIncrementSuite(t *testing.T) {
	suite.Run(t, new(caseSuite))
}

func createServer() *httptest.Server {
	return httptest.NewServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if r.Header.Get("Authorization") == "" {
			http.Error(w, "missing authorization header", http.StatusUnauthorized)
			return
		}

		if r.Method == http.MethodPost && r.URL.Path == "/api/v2/cases" {
			fmt.Fprint(w, caseJSON)
		} else if r.Method == http.MethodPatch && r.URL.Path == fmt.Sprintf("/api/v2/cases/%d", testCaseID) {
			fmt.Fprint(w, caseJSON)
		} else if r.Method == http.MethodGet && r.URL.Path == fmt.Sprintf("/api/v2/cases/%d/message", testCaseID) {
			fmt.Fprint(w, caseMessageJSON)
		} else {
			http.Error(w, "unexpected request", http.StatusNotFound)
		}
	}))
}

func createClient(endpoint string) (Client, error) {
	return NewClient(Config{
		Endpoint:       endpoint,
		ConsumerKey:    "consumer-key",
		ConsumerSecret: "consumer-secret",
		AccessToken:    "access-token",
		AccessSecret:   "access-secret",
	})
}

var caseJSON = `{
    "id": 1337,
    "external_id": null,
    "blurb": "Test by Jordan Potter",
    "priority": 4,
    "locked_until": null,
    "label_ids": [
        3368049
    ],
    "active_at": null,
    "changed_at": "2016-08-26T18:17:38Z",
    "created_at": "2016-08-26T18:17:37Z",
    "updated_at": "2016-08-26T18:17:38Z",
    "first_opened_at": null,
    "opened_at": null,
    "first_resolved_at": null,
    "resolved_at": null,
    "status": "new",
    "active_notes_count": 0,
    "active_attachments_count": 0,
    "has_pending_interactions": false,
    "has_failed_interactions": false,
    "description": null,
    "language": null,
    "received_at": "2016-08-26T18:17:37Z",
    "type": "email",
    "labels": [
        "Awesome Label"
    ],
    "subject": "Test Test Test",
    "route_status": "available",
    "custom_fields": {
        "is_spammer": null,
        "case_language": null,
        "category": null,
        "follow_up": null,
        "twitch_amazon_case_id": null,
        "twitch_display_name": null,
        "twitch_amazon_linked": null
    },
    "_links": {
        "self": {
            "href": "/api/v2/cases/11921659",
            "class": "case"
        },
        "message": {
            "href": "/api/v2/cases/11921659/message",
            "class": "email"
        },
        "customer": {
            "href": "/api/v2/customers/493200276",
            "class": "customer"
        },
        "labels": {
            "href": "/api/v2/cases/11921659/labels",
            "class": "label"
        },
        "assigned_user": null,
        "assigned_group": {
            "href": "/api/v2/groups/622367",
            "class": "group"
        },
        "locked_by": null,
        "history": {
            "href": "/api/v2/cases/11921659/history",
            "class": "history"
        },
        "case_links": {
            "href": "/api/v2/cases/11921659/links",
            "class": "case_link",
            "count": 0
        },
        "macro_preview": {
            "href": "/api/v2/cases/11921659/macros/preview",
            "class": "macro_preview"
        },
        "replies": {
            "href": "/api/v2/cases/11921659/replies",
            "class": "reply",
            "count": 0
        },
        "draft": {
            "href": "/api/v2/cases/11921659/replies/draft",
            "class": "reply"
        },
        "notes": {
            "href": "/api/v2/cases/11921659/notes",
            "class": "note",
            "count": 0
        },
        "attachments": {
            "href": "/api/v2/cases/11921659/attachments",
            "class": "attachment",
            "count": 0
        }
    }
}`

var caseMessageJSON = `{
    "created_at": "2016-08-26T02:04:48Z",
    "bcc": null,
    "body": "Ex Dee Lorem ipsum dolor sit amet, consectetur adipiscing elit. \"properly\" \\exp 🔫 <wrip> <script> kappa!\r\n\r\n你将不能改变英文ID的大小写。另外，设定%{language_name}昵称是一个永久性的设定，一旦设定完毕就不能改喔!",
    "cc": null,
    "client_type": "desk_portal",
    "direction": "in",
    "erased_at": null,
    "from": "fire@twitch.tv",
    "hidden": false,
    "hidden_at": null,
    "hidden_by": null,
    "id": 1490586401,
    "sent_at": null,
    "status": "received",
    "subject": "123 Something isn't working \"properly\" \\exp ???? <wrip> <script> kappa!",
    "to": null,
    "updated_at": "2016-08-26T02:04:48Z",
    "_links": {
        "self": {
            "class": "email",
            "href": "/api/v2/cases/11911035/message"
        },
        "outbound_mailbox": {
            "class": "outbound_mailbox",
            "href": "/api/v2/mailboxes/outbound/14931"
        },
        "sent_by": null,
        "erased_by": null,
        "case": {
            "class": "case",
            "href": "/api/v2/cases/11911035"
        },
        "customer": {
            "class": "customer",
            "href": "/api/v2/customers/57004888"
        },
        "attachments": {
            "class": "attachment",
            "href": "/api/v2/cases/11911035/message/attachments"
        },
        "splits": {
            "class": "split",
            "href": "/api/v2/cases/11911035/replies/1396916203/splits"
        }
    }
}
`
