// Package desk provides structs and functions for accessing version 2 of the
// Salesforce Desk API.
package desk

import (
	"bytes"
	"context"
	"encoding/json"
	"io"
	"net/http"

	"code.justin.tv/common/twitchhttp"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/dghubble/oauth1"
	"github.com/pkg/errors"
)

// Client performs API operations on a specified Desk portal, automatically
// handling authentication.
type Client interface {
	// CreateCase will create a new case in Desk. The supplied CaseMessage will
	// be the initial message that opened the case.
	CreateCase(ctx context.Context, c *Case, cm *CaseMessage) (*Case, error)

	// UpdateCase will update the specified case with the values provided. The
	// specified Case object must at least include an ID.
	UpdateCase(ctx context.Context, c *Case) (*Case, error)

	// CaseMessage will retrieve the original message that opened the case.
	CaseMessage(ctx context.Context, caseID int) (*CaseMessage, error)
}

// Config includes the necessary fields to successfully communicate with Desk.
type Config struct {
	Endpoint       string
	ConsumerKey    string
	ConsumerSecret string
	AccessToken    string
	AccessSecret   string
	Stats          statsd.Statter
}

type client struct {
	httpClient twitchhttp.Client
}

// NewClient returns a new client that communicates with Desk using the
// provided settings for authentication.
func NewClient(conf Config) (Client, error) {
	httpClient, err := httpClient(conf)
	return &client{httpClient}, err
}

func httpClient(conf Config) (twitchhttp.Client, error) {
	wrappedOAuthRoundTripper := func(rt http.RoundTripper) http.RoundTripper {
		config := oauth1.NewConfig(conf.ConsumerKey, conf.ConsumerSecret)
		token := oauth1.NewToken(conf.AccessToken, conf.AccessSecret)
		oauthHTTPClient := config.Client(oauth1.NoContext, token)
		oauthTransport := oauthHTTPClient.Transport.(*oauth1.Transport)
		oauthTransport.Base = rt
		return oauthTransport
	}

	return twitchhttp.NewClient(twitchhttp.ClientConf{
		Host:                 conf.Endpoint,
		RoundTripperWrappers: []func(http.RoundTripper) http.RoundTripper{wrappedOAuthRoundTripper},
		Stats:                conf.Stats,
	})
}

func (c *client) do(ctx context.Context, method, path, statName string, reqData, respData interface{}) error {
	req, err := c.createRequest(method, path, reqData)
	if err != nil {
		return errors.Wrap(err, "could not create request")
	}

	reqOpts := twitchhttp.ReqOpts{StatName: statName}
	resp, err := c.httpClient.DoJSON(ctx, respData, req, reqOpts)
	return errors.Wrapf(err, "request failed with status %s", resp.Status)
}

func (c *client) createRequest(method, path string, data interface{}) (*http.Request, error) {
	var body io.Reader
	if data != nil {
		bodyJSON, err := json.Marshal(data)
		if err != nil {
			return nil, errors.Wrap(err, "could not jsonify data")
		}
		body = bytes.NewBuffer(bodyJSON)
	}

	req, err := c.httpClient.NewRequest(method, path, body)
	if err != nil {
		return nil, errors.Wrap(err, "could not create http request object")
	}

	req.Header.Add("Accept", "application/json")
	req.Header.Add("Content-Type", "application/json")
	return req, nil
}
