package client

import (
	"encoding/json"
	"errors"

	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/lambda"
)

type Client interface {
	SubmitQuote(quote SubmitQuoteRequest) (string, error)
	DeleteQuote(quoteID string) error
}

func New(sess *session.Session, functionArn string) Client {
	return &client{
		lambda:      lambda.New(sess),
		functionArn: functionArn,
	}
}

type client struct {
	lambda      *lambda.Lambda
	functionArn string
}

func (c *client) SubmitQuote(quote SubmitQuoteRequest) (string, error) {
	bytes, err := json.Marshal(&quote)

	if err != nil {
		return "", err
	}

	resp, err := c.lambda.Invoke(&lambda.InvokeInput{
		FunctionName: &c.functionArn,
		Payload:      bytes,
	})

	if err != nil {
		return "", err
	}

	submitQuoteResp := SubmitQuoteResponse{}

	err = json.Unmarshal(resp.Payload, &submitQuoteResp)
	if err != nil {
		return "", err
	}

	if submitQuoteResp.Success {
		return submitQuoteResp.QuoteID, nil
	}

	return "", errors.New(submitQuoteResp.Message)
}

func (c *client) DeleteQuote(quoteID string) error {
	return errors.New("Not Implemented")
}
