package api

import (
	"encoding/json"
	"io"
	"net/http"
	"net/url"
	"strconv"
	"time"

	"code.justin.tv/cs/pong/tickets"
	"code.justin.tv/cs/pong/tickets/amazon"

	"github.com/pkg/errors"
)

type AmazonTicketRequest struct {
	Secret             string    `json:"secret"`
	DeskCaseID         int       `json:"desk_case_id,string"`
	DeskSubjectEscaped string    `json:"desk_subject"`
	DeskBodyEscaped    string    `json:"desk_body"`
	TwitchUserID       string    `json:"twitch_userid"`
	TwitchPartner      bool      `json:"twitch_partner,string"`
	AmazonPrime        bool      `json:"twitch_amazon_prime,string"`
	AmazonPrimeCountry string    `json:"twitch_amazon_prime_country"`
	Created            time.Time `json:"created_at"`
	LiquidErr          string    `json:"liquid_err"`
}

func (s *Server) createAmazonTicket(w http.ResponseWriter, r *http.Request) {
	var data AmazonTicketRequest
	ctx := r.Context()
	lr := io.LimitReader(r.Body, 10<<10)

	if err := json.NewDecoder(lr).Decode(&data); err != nil {
		handleError(w, http.StatusBadRequest, "failed to parse request", err, true)
		return
	}

	// We only want Desk to have access to this API endpoint. To accomplish
	// this, we verify that Desk is passing the secret that we expect. This
	// secret has been randomly generated and is unguessable by outside parties.
	if data.Secret != s.deskPayloadSecret {
		err := errors.New("incorrect payload secret")
		handleError(w, http.StatusForbidden, http.StatusText(http.StatusForbidden), err, false)
		return
	}

	// Desk's webhook logic runs Liquid when producing the webhook payload.
	// However Desk can't error out when it confronts an error while running
	// the Liquid engine. Instead, it passes that error to us to handle.
	if data.LiquidErr != "" {
		err := errors.New(data.LiquidErr)
		handleError(w, http.StatusBadRequest, "received Liquid error", err, true)
		return
	}

	if err := validateAmazonTicketRequest(&data); err != nil {
		handleError(w, http.StatusBadRequest, "invalid request", err, true)
		return
	}

	subject, err := url.QueryUnescape(data.DeskSubjectEscaped)
	if err != nil {
		handleError(w, http.StatusBadRequest, "failed to unescape desk subject", err, true)
		return
	}

	body, err := url.QueryUnescape(data.DeskBodyEscaped)
	if err != nil {
		handleError(w, http.StatusBadRequest, "failed to unescape desk body", err, true)
		return
	}

	user, err := s.usersClient.GetUserByID(ctx, data.TwitchUserID, nil)
	if err != nil {
		handleError(w, http.StatusInternalServerError, "failed to retrieve user information", err, false)
		return
	}

	t := tickets.Amazon{
		Subject:            subject,
		Body:               body,
		TwitchTicketID:     strconv.Itoa(data.DeskCaseID),
		TwitchUserID:       data.TwitchUserID,
		TwitchUsername:     *user.Displayname,
		TwitchPartner:      data.TwitchPartner,
		AmazonPrime:        data.AmazonPrime,
		AmazonPrimeCountry: data.AmazonPrimeCountry,
		Created:            data.Created,
		C2CID:              amazon.C2CIDForCountry(data.AmazonPrimeCountry),
	}
	if err := s.amazonClient.CreateTicket(ctx, t); err != nil {
		handleError(w, http.StatusInternalServerError, "failed to create Amazon ticket", err, false)
		return
	}
}

func validateAmazonTicketRequest(atr *AmazonTicketRequest) error {
	if atr.DeskCaseID == 0 {
		return errors.New("missing desk case id")
	} else if atr.DeskSubjectEscaped == "" {
		return errors.New("missing desk subject")
	} else if atr.DeskBodyEscaped == "" {
		return errors.New("missing desk body")
	} else if atr.TwitchUserID == "" {
		return errors.New("missing twitch user id")
	} else if atr.AmazonPrime && atr.AmazonPrimeCountry == "" {
		return errors.New("missing amazon prime country")
	}
	return nil
}
