package api

import (
	"encoding/json"
	"io"
	"net/http"
	"net/mail"
	"strconv"
	"strings"
	"unicode"

	desk "code.justin.tv/cs/desk-client"
	"code.justin.tv/cs/pong/logger"
	"code.justin.tv/web/users-service/models"
	"github.com/pkg/errors"
)

type ProcessorMessage struct {
	Secret             string `json:"secret"`
	DeskCaseID         string `json:"desk_case_id"`
	TwitchUserID       string `json:"twitch_userid,omitempty"`
	DeskCustomerEmails string `json:"desk_customer_emails"`
}

func (s *Server) processTicket(w http.ResponseWriter, r *http.Request) {
	var data ProcessorMessage
	ctx := r.Context()
	lr := io.LimitReader(r.Body, 10<<10)

	if err := json.NewDecoder(lr).Decode(&data); err != nil {
		handleError(w, http.StatusBadRequest, "failed to parse request", err, true)
		return
	}

	// We only want Desk to have access to this API endpoint. To accomplish
	// this, we verify that Desk is passing the secret that we expect. This
	// secret has been randomly generated and is unguessable by outside parties.
	if data.Secret != s.deskPayloadSecret {
		err := errors.New("incorrect payload secret")
		handleError(w, http.StatusForbidden, http.StatusText(http.StatusForbidden), err, false)
		return
	}

	if err := validateTicketProcessRequest(&data); err != nil {
		handleError(w, http.StatusBadRequest, "invalid request", err, true)
		return
	}

	var userList []string

	// * if `UserID` exists in the payload
	if data.TwitchUserID != "" {
		// 	1. Append it to the user list
		userList = append(userList, data.TwitchUserID)
	} else {
		// 	1. Split the emails string ( it's a comma delimited string )
		emailList := strings.Split(data.DeskCustomerEmails, ",")
		// Trim the rest of the input emails
		for i, email := range emailList {
			emailList[i] = strings.TrimSpace(email)
		}
		// 	2. Validate they're all at least email addresses ( `^\S+?@\S+?\.\S+?$` )
		emailList, err := filterValidEmails(emailList)
		if err != nil {
			handleError(w, http.StatusBadRequest, "invalid request", err, true)
			return
		}
		// 	3. Users Service -> Get users with the emails from the previous step
		results, err := s.usersClient.GetUsers(ctx, &models.FilterParams{
			Emails: emailList,
		}, nil)
		if err != nil {
			handleError(w, http.StatusInternalServerError, "failed to retrieve user information", err, false)
			return
		}

		users := results.ConvertToExternal().Results

		//	4. Append it to the user list
		for _, user := range users {
			userList = append(userList, user.ID)
		}

	}

	labelMap := make(map[string]struct{})

	// 	5. Check resulting userIDs against customer TuID set
	for i, user := range userList {
		// users-service can potential return hundreds/thousands of users.
		// Most users in our impact set will not be spamming account creation,
		// and non-legacy users can only have up to 25 accounts per email address.
		if i > 100 {
			break
		}
		result := s.ticketProcessorClient.GetSpecialUser(user)
		if result != nil && result.SirtImpacted {
			labelMap["SIRT-IMPACTED"] = struct{}{}
			labelMap[result.JiraID] = struct{}{}
		}
	}

	var labelList []string
	for key := range labelMap {
		labelList = append(labelList, key)
	}

	// * If the customer exists in the TuID set
	// 	1. Update the Desk case with the JIRA IDs from the impact set and the `SIRT-IMPACTED` label

	if len(labelList) > 0 {

		parsedID, err := strconv.Atoi(data.DeskCaseID)
		if err != nil {
			handleError(w, http.StatusInternalServerError, "failed to parse Desk ticket ID", errors.Wrapf(err, "failed to update Desk id %s with SIRT-IMPACTED/%s", data.DeskCaseID, labelList), false)
			return
		}
		//TODO: add a call to the service cloud client here as well
		_, err = s.deskClient.UpdateCase(ctx,
			&desk.Case{
				ID:     parsedID,
				Labels: labelList,
			})
		if err != nil {
			handleError(w, http.StatusInternalServerError, "failed to update Desk ticket", errors.Wrapf(err, "failed to update Desk id %s with SIRT-IMPACTED/%s", data.DeskCaseID, labelList), false)
			return
		}

		logger.Infof("Updated Desk ticket %s with SIRT-IMPACTED/%s", data.DeskCaseID, labelList)

		// err = s.serviceCloudClient.UpdateCase(ctx, string(parsedID),
		// 	salesforce.CustomerCase{
		// 		Labels: ptr.String(strings.Join(labelList[:], ",")),
		// 	})
		// if err != nil {
		// 	handleError(w, http.StatusInternalServerError, "failed to update Service Cloud ticket", errors.Wrapf(err, "failed to update Service Cloud id %s with SIRT-IMPACTED/%s", data.DeskCaseID, labelList), false)
		// 	return
		// }

		// logger.Infof("Updated Service Cloud ticket %s with SIRT-IMPACTED/%s", data.DeskCaseID, labelList)

	}

}

func validateTicketProcessRequest(atr *ProcessorMessage) error {
	if atr.DeskCaseID == "" {
		return errors.New("missing desk case id")
	}
	if atr.DeskCaseID == "0" {
		return errors.New("invalid desk case id")
	}
	if atr.DeskCustomerEmails == "" {
		return errors.New("missing customer email")
	}
	return nil
}

func filterValidEmails(emails []string) ([]string, error) {
	var filteredEmails []string

	for _, email := range emails {
		if isValidEmail(email) {
			filteredEmails = append(filteredEmails, email)
		}
	}

	if len(filteredEmails) > 0 {
		return filteredEmails, nil
	}

	return nil, errors.New("no valid customer emails")
}

func isValidEmail(email string) bool {
	if len(email) == 0 {
		return false
	}
	if unicode.IsSpace(rune(email[0])) {
		return false
	}
	if _, err := mail.ParseAddress(email); err != nil {
		return false
	}
	return true
}
