package api

import (
	"strings"
	"testing"

	"github.com/stretchr/testify/assert"
)

func Test_validEmail(t *testing.T) {

	assert.True(t, isValidEmail("test@twitch.tv"), "an email with a local part, domain, and tld are valid")
	assert.False(t, isValidEmail("test@test@twitch.tv"), "Emails with more than two domain parts are invalid")
	assert.False(t, isValidEmail(" test@twitch.tv"), "Emails with a space in the front are invalid")

}

func Test_validateTicketProcessRequest(t *testing.T) {
	atr := getSampleProcessorMessage()
	atr.DeskCaseID = "0"
	assert.Error(t, validateTicketProcessRequest(atr))

	atr = getSampleProcessorMessage()
	atr.DeskCaseID = ""
	assert.Error(t, validateTicketProcessRequest(atr))

	atr = getSampleProcessorMessage()
	atr.DeskCustomerEmails = ""
	assert.Error(t, validateTicketProcessRequest(atr))

	atr = getSampleProcessorMessage()
	atr.TwitchUserID = ""
	assert.NoError(t, validateTicketProcessRequest(atr))

}

func Test_filterValidEmails(t *testing.T) {
	atr := getSampleProcessorMessage()
	emails := strings.Split(atr.DeskCustomerEmails, ",")
	emails, err := filterValidEmails(emails)
	assert.NoError(t, err, "filtering a list with at least one valid email should not error")
	assert.Exactly(t, emails, []string{"test@twitch.tv"}, "invalid emails should be filtered out")

	atr = getSampleProcessorMessage()
	emails = strings.Split(atr.DeskCustomerEmails, ",")
	for i, e := range emails {
		emails[i] = strings.TrimSpace(e)
	}
	emails, err = filterValidEmails(emails)
	assert.NoError(t, err, "filtering a list with at least one valid email should not error")
	assert.Exactly(t, emails, []string{"test@twitch.tv", "test@mods.twitch.tv"}, "after trimming, otherwise valid emails with a space in front should become valid")

	atr = getSampleProcessorMessage()
	atr.DeskCustomerEmails = `testa]\s@twitch.tv, test@test2@twtich.tv, test\^@mods.twitch.tv`
	emails = strings.Split(atr.DeskCustomerEmails, ",")
	for i, e := range emails {
		emails[i] = strings.TrimSpace(e)
	}
	emails, err = filterValidEmails(emails)
	assert.Error(t, err, "filtering a list with zero valid emails should error")
	assert.Len(t, emails, 0, "after filtering, an email list with no valid emails should be empty")

}

func getSampleProcessorMessage() *ProcessorMessage {
	return &ProcessorMessage{
		Secret:             "asdf",
		DeskCaseID:         "1234",
		TwitchUserID:       "17394",
		DeskCustomerEmails: "test@twitch.tv, test@test2@twtich.tv, test@mods.twitch.tv",
	}
}
