"""autotag.py

Automatically tag EC2 instances, autogenerated by chasquimigration.

To run it: pip install --user boto3 && python3 autotag.py


usage: autotag.py [-h] [--dryrun] [--input INPUT]

Automatically tag EC2 instances.

optional arguments:
  -h, --help     show this help message and exit
  --dryrun       Do a dry run, checking if you have permissions without
                 actually tagging.
  --input INPUT  Path to the json generated by chasquimigration with
                 stages/hosts information.

"""

import argparse
import json
import logging

import boto3
import botocore


class EC2:
    FILTER_LIMIT = 200

    def __init__(self, region, dryrun):
        self.client = boto3.client("ec2", region)
        self.resource = boto3.resource("ec2", region)
        self.data = {}
        self.dryrun = dryrun

    def gather_addresses_info(self, addresses):
        logging.debug("Gathering info for addresses: {}".format(list(addresses)))
        missing = [address for address in addresses if address not in self.data]
        self.warm_cache_by_addresses(missing)

    def paginated_items(self, items):
        return (
            items[i : i + self.FILTER_LIMIT]
            for i in range(0, len(items), self.FILTER_LIMIT)
        )

    def warm_cache_by_addresses(self, addresses):
        self.data.update(
            {
                instance.private_ip_address: instance
                for address_group in self.paginated_items(addresses)
                for instance in (
                    self.resource.instances.filter(
                        Filters=[
                            {"Name": "private-ip-address", "Values": address_group}
                        ]
                    )
                )
            }
        )

    def get(self, address):
        return self.data.get(address)

    def untag_unexpected_instances(self, service, stage, tag_key, expected_hosts):
        for instance in self.resource.instances.filter(
            Filters=[{
                "Name": "tag-key",
                "Values": [
                    "ServiceTuple:{}:{}".format(service, stage)
                ]
            }]
        ):
            if instance.private_ip_address in expected_hosts:
                continue
            logging.warning(
                "Instance {} shouldn't be tagged as {} - {}. Untagging..."
            ).format(instance.private_ip_address, service, stage)
            try:
                instance.delete_tags(DryRun=self.dryrun, Tags=[{"Key": tag_key}])
            except botocore.exceptions.ClientError as err:
                if err.response["Error"]["Code"] == "DryRunOperation":
                    logging.info("Untagging will succeed")
                else:
                    raise

    def tag_untagged_instances(self, tag_key, hosts):
        to_tag = {
            address
            for address in hosts
            if not self.get(address)
            or (
                self.get(address).tags is not None
                and not any(
                    tag for tag in self.get(address).tags if tag["Key"] == tag_key
                )
            )
        }
        # to_tag = hosts
        if not to_tag:
            logging.info("All hosts are already correctly tagged. Nothing to do here.")
            return
        logging.info("Tagging {} in {}".format(tag_key, list(to_tag)))
        for address_group in self.paginated_items(list(to_tag)):
            try:
                self.resource.instances.filter(
                    Filters=[{"Name": "private-ip-address", "Values": address_group}]
                ).create_tags(
                    DryRun=self.dryrun, Tags=[{"Key": tag_key, "Value": "::"}]
                )
            except botocore.exceptions.ClientError as err:
                if err.response["Error"]["Code"] == "DryRunOperation":
                    logging.info("Tagging will succeed")
                else:
                    raise

    def stage_tagger(self, service, stage, hosts_in_consul):
        logging.info("Processing service={} stage={}".format(service, stage))
        tag_key = "ServiceTuple:{}:{}".format(service, stage)
        # untag nodes for this service:stage that shouldn't be tagged
        self.gather_addresses_info(hosts_in_consul)
        self.untag_unexpected_instances(service, stage, tag_key, hosts_in_consul)
        self.tag_untagged_instances(tag_key, hosts_in_consul)


def main():
    logging.basicConfig(
        format="%(name)-32s %(levelname)-8s %(message)s", level=logging.DEBUG
    )
    logging.getLogger("botocore").setLevel(logging.INFO)
    logging.getLogger("boto3").setLevel(logging.INFO)
    logging.getLogger("urllib3").setLevel(logging.INFO)

    parser = argparse.ArgumentParser(description="Automatically tag EC2 instances.")
    parser.add_argument(
        "--dryrun",
        action="store_true",
        default=False,
        help="Do a dry run, checking if you have permissions without actually tagging.",
    )
    parser.add_argument(
        "--input",
        action="store",
        default="autotag.json",
        help="Path to the json generated by chasquimigration with stages/hosts information.",
    )
    args = parser.parse_args()
    with open(args.input) as fh:
        input_json = json.load(fh)
        for service, stages in input_json.items():
            for stage, stage_data in stages.items():
                for region, hosts_in_consul in stage_data.items():
                    EC2(region, args.dryrun).stage_tagger(
                        service, stage, hosts_in_consul
                    )


if __name__ == "__main__":
    main()
