package main

import (
	"context"
	"log"

	"github.com/cactus/go-statsd-client/statsd"
	"github.com/pkg/errors"

	"code.justin.tv/common/config"
	"code.justin.tv/common/twitchhttp"
	desk "code.justin.tv/cs/desk-client"
	"code.justin.tv/cs/pong/api"
	"code.justin.tv/cs/pong/secrets"
	"code.justin.tv/cs/pong/tickets"
	"code.justin.tv/cs/pong/tickets/amazon"
	"code.justin.tv/cs/pong/tickets/processor"
	"code.justin.tv/cs/pong/tickets/twitch"
	"code.justin.tv/cs/pong/turbo"
	salesforce "code.justin.tv/cs/salesforce-client"
	"code.justin.tv/foundation/twitchclient"
	samus "code.justin.tv/samus/gateway/client"
	users "code.justin.tv/web/users-service/client/usersclient_internal"
)

func init() {
	config.Register(map[string]string{
		"secrets_dir":                  "",
		"desk_client_endpoint":         "https://help.twitch.tv",
		"users_client_endpoint":        "http://users-service.prod.us-west2.twitch.tv",
		"partnerships_client_endpoint": "http://partnerships.prod.us-west2.twitch.tv",
		"turbo_client_endpoint":        "http://prod-nitro.us-west-2.elasticbeanstalk.com",
		"samus_client_endpoint":        "http://prod-samus-gateway-env.ybgep8dqmz.us-west-2.elasticbeanstalk.com",
	})
}

func main() {
	err := config.Parse()
	if err != nil {
		log.Fatal(err)
	}

	currentSecret := "desk.payload_secret"
	deskPayloadSecret, err := secrets.PopulateSecret(currentSecret)
	if err != nil {
		log.Fatal(errors.Wrapf(err, "populating secret: %s", currentSecret))
	}

	stats := config.Statsd()

	deskClient, err := createDeskClient(config.Statsd())
	if err != nil {
		log.Fatal(err)
	}

	serviceCloudClient, err := createServicecloudClient(true)
	if err != nil {
		log.Fatal(err)
	}

	usersClient, err := createUsersClient()
	if err != nil {
		log.Fatal(err)
	}

	ticketProcessorClient, err := createTicketProcessorClient(deskClient, serviceCloudClient, usersClient, stats)
	if err != nil {
		log.Fatal(err)
	}

	turboClient, err := createTurboClient()
	if err != nil {
		log.Fatal(err)
	}

	samusClient, err := createSamusClient()
	if err != nil {
		log.Fatal(err)
	}

	twitchClient, err := createTwitchClient(deskClient, serviceCloudClient, usersClient, turboClient, samusClient, stats)
	if err != nil {
		log.Fatal(err)
	}

	amazonClient, err := createAmazonClient(twitchClient, stats)
	if err != nil {
		log.Fatal(err)
	}

	server, err := api.NewServer(usersClient, twitchClient, amazonClient, ticketProcessorClient, deskClient, deskPayloadSecret, serviceCloudClient)
	if err != nil {
		log.Fatal(err)
	}

	log.Fatal(twitchhttp.ListenAndServe(server, twitchhttp.NewConfig()))
}

func createServicecloudClient(noop bool) (salesforce.Client, error) {

	if noop {
		return nil, nil
	}

	var value string
	var err error
	var currentSecret string
	salesforceConfig := salesforce.Config{
		URL:          "",
		ClientID:     "",
		ClientSecret: "",
		Password:     "",
		Username:     "",
	}

	currentSecret = "service_cloud.instance_url"
	value, err = secrets.PopulateSecret(currentSecret)
	if err != nil {
		return nil, errors.Wrapf(err, "populating secret: %s", currentSecret)
	}
	salesforceConfig.URL = value

	currentSecret = "service_cloud.client_id"
	value, err = secrets.PopulateSecret(currentSecret)
	if err != nil {
		return nil, errors.Wrapf(err, "populating secret: %s", currentSecret)
	}
	salesforceConfig.ClientID = value

	currentSecret = "service_cloud.client_secret"
	value, err = secrets.PopulateSecret(currentSecret)
	if err != nil {
		return nil, errors.Wrapf(err, "populating secret: %s", currentSecret)
	}
	salesforceConfig.ClientSecret = value

	currentSecret = "service_cloud.username"
	value, err = secrets.PopulateSecret(currentSecret)
	if err != nil {
		return nil, errors.Wrapf(err, "populating secret: %s", currentSecret)
	}
	salesforceConfig.Username = value

	currentSecret = "service_cloud.password"
	value, err = secrets.PopulateSecret(currentSecret)
	if err != nil {
		return nil, errors.Wrapf(err, "populating secret: %s", currentSecret)
	}
	salesforceConfig.Password = value

	currentSecret = "service_cloud.security_token"
	value, err = secrets.PopulateSecret(currentSecret)
	if err != nil {
		return nil, errors.Wrapf(err, "populating secret: %s", currentSecret)
	}

	// a service cloud API password is an account password+security_security
	salesforceConfig.Password += value

	return salesforce.NewSalesforceClient(&salesforceConfig)
}

func createDeskClient(stats statsd.Statter) (desk.Client, error) {

	var value string
	var err error
	var currentSecret string
	deskConfig := desk.Config{
		Endpoint:       "", //config.Resolve("desk_client_endpoint"),
		ConsumerKey:    "",
		ConsumerSecret: "",
		AccessToken:    "",
		AccessSecret:   "",
		Stats:          stats,
	}

	currentSecret = "desk.instance_url"
	value, err = secrets.PopulateSecret(currentSecret)
	if err != nil {
		return nil, errors.Wrapf(err, "populating secret: %s", currentSecret)
	}
	deskConfig.Endpoint = value

	currentSecret = "desk.consumer_key"
	value, err = secrets.PopulateSecret(currentSecret)
	if err != nil {
		return nil, errors.Wrapf(err, "populating secret: %s", currentSecret)
	}
	deskConfig.ConsumerKey = value

	currentSecret = "desk.consumer_secret"
	value, err = secrets.PopulateSecret(currentSecret)
	if err != nil {
		return nil, errors.Wrapf(err, "populating secret: %s", currentSecret)
	}
	deskConfig.ConsumerSecret = value

	currentSecret = "desk.access_token"
	value, err = secrets.PopulateSecret(currentSecret)
	if err != nil {
		return nil, errors.Wrapf(err, "populating secret: %s", currentSecret)
	}
	deskConfig.AccessToken = value

	currentSecret = "desk.access_secret"
	value, err = secrets.PopulateSecret(currentSecret)
	if err != nil {
		return nil, errors.Wrapf(err, "populating secret: %s", currentSecret)
	}
	deskConfig.AccessSecret = value

	return desk.NewClient(deskConfig)
}

func createUsersClient() (users.InternalClient, error) {
	conf := twitchclient.ClientConf{
		Host: config.Resolve("users_client_endpoint"),
	}
	return users.NewClient(conf)
}

func createTicketProcessorClient(deskClient desk.Client, serviceCloudClient salesforce.Client, usersClient users.InternalClient, stats statsd.Statter) (processor.Client, error) {
	conf := processor.Config{
		DeskClient:         deskClient,
		ServiceCloudClient: serviceCloudClient,
		UsersClient:        usersClient,
		Stats:              stats,
	}
	return processor.NewClient(conf)
}

func createTurboClient() (turbo.Client, error) {
	conf := twitchclient.ClientConf{
		Host: config.Resolve("turbo_client_endpoint"),
	}

	return turbo.NewClient(conf)
}

func createSamusClient() (samus.Client, error) {
	conf := twitchclient.ClientConf{
		Host: config.Resolve("samus_client_endpoint"),
	}
	return samus.NewClient(conf)
}

func createTwitchClient(deskClient desk.Client, serviceCloudClient salesforce.Client, usersClient users.InternalClient, turboClient turbo.Client, samusClient samus.Client, stats statsd.Statter) (twitch.Client, error) {
	conf := twitch.Config{
		DeskClient:         deskClient,
		ServiceCloudClient: serviceCloudClient,
		UsersClient:        usersClient,
		TurboClient:        turboClient,
		SamusClient:        samusClient,
		Stats:              stats,
	}
	return twitch.NewClient(conf)
}

func createAmazonClient(twitchClient twitch.Client, stats statsd.Statter) (amazon.Client, error) {

	var value string
	var err error
	var currentSecret string

	conf := amazon.Config{
		TicketAckHandler: func(ctx context.Context, ack tickets.AmazonAck) error {
			return twitchClient.MarkTicketTransferred(ctx, ack.TwitchTicketID, ack.AmazonMediaID)
		},
		TwitchTicketHandler: func(ctx context.Context, t tickets.Twitch) error {
			return twitchClient.CreateTicket(ctx, t)
		},
		Stats: stats,
	}

	// Get the public key to encrypt what we send
	currentSecret = "amazon_client.encryption.public_key"
	value, err = secrets.PopulateSecret(currentSecret)
	if err != nil {
		return nil, errors.Wrapf(err, "populating secret: %s", currentSecret)
	}
	encryptionKey, err := secrets.ReadRSAPublicKey(value)
	if err != nil {
		return nil, errors.Wrapf(err, "parsing RSA Public Key: %s", currentSecret)
	}
	conf.EncryptionPublicKey = encryptionKey

	// Get the private key to decrypt what we recieve
	currentSecret = "amazon_client.decryption.private_key"
	value, err = secrets.PopulateSecret(currentSecret)
	if err != nil {
		return nil, errors.Wrapf(err, "populating secret: %s", currentSecret)
	}
	decryptionKey, err := secrets.ReadRSAPrivateKey(value)
	if err != nil {
		return nil, errors.Wrapf(err, "parsing RSA Private Key: %s", currentSecret)
	}
	conf.DecryptionPrivateKey = decryptionKey

	// Return the finalized client
	return amazon.NewClient(conf)
}
