package secrets

import (
	"bytes"
	"crypto/rsa"
	"crypto/x509"
	"encoding/pem"
	"fmt"
	"log"
	"os"
	"strings"
	"text/template"

	"code.justin.tv/common/config"
	"code.justin.tv/systems/sandstorm/manager"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sts"
	"github.com/pkg/errors"
)

func getSecret(sandstormManager manager.API, secretName string) (string, error) {
	secretNameTemplate := "{{.Org}}/{{.App}}/{{.Environment}}/{{.SecretName}}"
	params := struct {
		Org         string
		App         string
		Environment string
		SecretName  string
	}{
		Org:         config.App(),         // owning group (support, safety, web-rails, etc.)
		App:         "app",                // service name (pong, dmca, etc.)
		Environment: config.Environment(), // env name     (production, staging, development, etc.)
		SecretName:  secretName,           // secret name  (desk.access_token, etc.)
	}

	tmpl, err := template.New("secretNameTemplate").Parse(secretNameTemplate)
	if err != nil {
		return "", errors.Wrapf(err, "parsing secretName template for %s: ", secretName)
	}

	var key bytes.Buffer
	err = tmpl.Execute(&key, params)
	if err != nil {
		return "", errors.Wrapf(err, "executing secretName template for %s: ", secretName)
	}

	secret, err := sandstormManager.Get(key.String())
	if err != nil {
		return "", errors.Wrapf(err, "getting secret value for %s in sandstorm: ", secretName)
	}
	if secret == nil {
		blankError := fmt.Errorf("secret blank for %s from sandstorm.", secretName)
		return "", blankError
	}

	return string(secret.Plaintext), nil
}

// PopulateSecret takes a name and returns the value fetched from sandstorm,
// or an error.
func PopulateSecret(secretName string) (string, error) {

	// Early exit to support passing in via local env vars
	localEnv, exists := os.LookupEnv(config.App() + "__" + strings.Replace(secretName, ".", "__", -1))
	if exists && len(localEnv) > 0 {
		log.Println("Env secret found:", (config.App() + "." + secretName))
		log.Println("Env secret value:", localEnv)
		return localEnv, nil
	}

	serviceName := config.App() + "-" + config.Environment()
	if serviceName[len(serviceName)-1:] == "-" || string(serviceName[0]) == "-" {
		return "", fmt.Errorf("service name incomplete: %s", serviceName)
	}

	session, err := session.NewSession()
	if err != nil {
		return "", errors.Wrap(err, "creating aws session")
	}

	// Get credentials to access information on the sandstorm account
	creds := credentials.NewCredentials(&stscreds.AssumeRoleProvider{
		// This is the ARN to the IAM role retrieved from https://dashboard-v2.internal.justin.tv/sandstorm/manage-roles
		RoleARN: "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/" + serviceName,
		Client:  sts.New(session),
	})

	m := manager.New(manager.Config{
		AWSConfig:   aws.NewConfig().WithRegion("us-west-2").WithCredentials(creds),
		ServiceName: serviceName,
	})
	defer func() {
		_ = m.CleanUp()
	}()

	value, err := getSecret(m, secretName)
	if err != nil {
		return "", err
	}

	return value, nil
}

// ReadRSAPublicKey takes a string, converts it to a byte slice,
// attempts to parse that slice as a PEM encoded RSA PUBLIC key,
// and returns either that PUBLIC key or an error
func ReadRSAPublicKey(name string) (*rsa.PublicKey, error) {
	data := []byte(name)

	block, _ := pem.Decode(data)
	if block == nil {
		return nil, errors.New("data is not PEM encoded")
	}

	key, err := x509.ParsePKIXPublicKey(block.Bytes)
	if err != nil {
		return nil, err
	}

	rsaKey, ok := key.(*rsa.PublicKey)
	if !ok {
		return nil, errors.New("key is not an RSA public key")
	}
	return rsaKey, nil
}

// ReadRSAPrivateKey takes a string, converts it to a byte slice,
// attempts to parse that slice as a PEM encoded RSA PRIVATE key,
// and returns either that PRIVATE key or an error
func ReadRSAPrivateKey(name string) (*rsa.PrivateKey, error) {
	data := []byte(name)

	block, _ := pem.Decode(data)
	if block == nil {
		return nil, errors.New("data is not PEM encoded")
	}

	key, err := x509.ParsePKCS8PrivateKey(block.Bytes)
	if err != nil {
		return nil, err
	}

	rsaKey, ok := key.(*rsa.PrivateKey)
	if !ok {
		return nil, errors.New("key is not an RSA private key")
	}
	return rsaKey, nil
}
