# Canary ASG to allow experimental deploys with minimized impact.

resource "template_file" "canary_app_user_data" {
  template = "${file("${path.module}/asg_puppetizer.template")}"
  vars {
    hostname_prefix = "${var.hostname}-aws"
    domain = "${lookup(var.domain, var.environment)}"
    cluster = "${var.name}"
    environment = "${var.environment}"
    canary = "true"
  }
}

resource "aws_launch_configuration" "canary_app_conf" {
  name_prefix = "${var.name}-${var.environment}-canary-app-"
  image_id = "${var.ami}"
  instance_type = "${var.instance_type}"
  security_groups = ["${var.twitch_subnets_sg}"]
  user_data = "${template_file.canary_app_user_data.rendered}"
  iam_instance_profile = "${aws_iam_instance_profile.app.id}"

  root_block_device {
    volume_size = "${var.root_volume_size}"
  }

  lifecycle {
    create_before_destroy = true
  }
}

resource "aws_autoscaling_group" "canary_app_asg" {
  name = "${var.name}-${var.environment}-canary-app"
  vpc_zone_identifier = ["${split(",", "${var.service_subnets}")}"]
  max_size = "${var.canary_size}"
  min_size = "${var.canary_size}"
  desired_capacity = "${var.canary_size}"
  health_check_grace_period = 600
  health_check_type = "EC2"
  load_balancers = ["${aws_elb.internal_app_elb.name}", "${aws_elb.external_app_elb.name}"]
  launch_configuration = "${aws_launch_configuration.canary_app_conf.name}"

  tag {
    key = "Name"
    value = "${var.name}"
    propagate_at_launch = true
  }
  tag {
    key = "Owner"
    value = "${var.owner}"
    propagate_at_launch = true
  }
  tag {
    key = "Project"
    value = "${var.project}"
    propagate_at_launch = true
  }
}

module "canary_asg_dns" {
  source = "git::git+ssh://git@git-aws.internal.justin.tv/web/asg-r53-lambda.git//modules/notifications"
  topic_name = "${var.name}_${var.environment}_canary_topic"
  asg_name = "${aws_autoscaling_group.canary_app_asg.name}"
  lambda_arn = "${var.asg_lambda_arn}"
}
