package amazon

import (
	"context"
	"crypto/rsa"

	"code.justin.tv/cs/pong/tickets"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/pkg/errors"
)

type Client interface {
	CreateTicket(ctx context.Context, t tickets.Amazon) error
}

type Config struct {
	EncryptionPublicKey  *rsa.PublicKey
	DecryptionPrivateKey *rsa.PrivateKey
	TicketAckHandler     TicketAckHandler
	TwitchTicketHandler  TwitchTicketHandler
	Stats                statsd.Statter
}

type TicketAckHandler func(ctx context.Context, ack tickets.AmazonAck) error
type TwitchTicketHandler func(ctx context.Context, t tickets.Twitch) error
type ticketPublishHandler func(ctx context.Context, t tickets.Amazon) error

type client struct {
	ticketPublishHandler ticketPublishHandler
}

func NewClient(conf Config) (Client, error) {

	ticketPublisher, err := createTicketPublisher(conf.EncryptionPublicKey, conf.Stats)
	if err != nil {
		return nil, errors.Wrap(err, "failed to create ticket publisher")
	}

	err = createTicketAckSubscriber(conf.TicketAckHandler, conf.DecryptionPrivateKey, conf.Stats)
	if err != nil {
		return nil, errors.Wrap(err, "failed to create ticket ack subscriber")
	}

	err = createTwitchTicketSubscriber(conf.TwitchTicketHandler, conf.DecryptionPrivateKey, conf.Stats)
	if err != nil {
		return nil, errors.Wrap(err, "failed to create Twitch ticket subscriber")
	}

	return &client{
		ticketPublishHandler: ticketPublisher,
	}, nil
}
