package amazon

import (
	"crypto/aes"
	"crypto/cipher"
	"crypto/rand"
	"crypto/rsa"
	"crypto/sha256"
	"encoding/json"

	"github.com/pkg/errors"
)

const (
	aesKeySize = 32
	nonceSize  = 12
)

type encryptedPayload struct {
	EncryptedKey  []byte `json:"encrypted_secret"`
	Nonce         []byte `json:"nonce"`
	EncryptedData []byte `json:"encrypted_data"`
}

func encrypt(publicKey *rsa.PublicKey, src interface{}) ([]byte, error) {
	key := make([]byte, aesKeySize)
	if _, err := rand.Read(key); err != nil {
		return nil, errors.Wrap(err, "unable to create key")
	}

	nonce := make([]byte, nonceSize)
	if _, err := rand.Read(nonce); err != nil {
		return nil, errors.Wrap(err, "unable to create nonce")
	}

	block, err := aes.NewCipher(key)
	if err != nil {
		return nil, errors.Wrap(err, "unable to create cipher block")
	}

	gcm, err := cipher.NewGCM(block)
	if err != nil {
		return nil, errors.Wrap(err, "unable to create AES GCM cipher")
	}

	jsonedData, err := json.Marshal(src)
	if err != nil {
		return nil, errors.Wrap(err, "unable to json marshal src")
	}

	encryptedKey, err := rsa.EncryptOAEP(sha256.New(), rand.Reader, publicKey, key, nil)
	if err != nil {
		return nil, errors.Wrap(err, "unable to encrypt key")
	}

	encryptedData := gcm.Seal(nil, nonce, jsonedData, nil)

	payload := encryptedPayload{
		EncryptedKey:  encryptedKey,
		Nonce:         nonce,
		EncryptedData: encryptedData,
	}

	jsonedPayload, err := json.Marshal(payload)
	return jsonedPayload, errors.Wrap(err, "unable to json marshal encryption payload")
}

func decrypt(privateKey *rsa.PrivateKey, dst interface{}, src []byte) error {
	var payload encryptedPayload
	if err := json.Unmarshal(src, &payload); err != nil {
		return errors.Wrap(err, "unable to json unmarshal encryption payload")
	}

	key, err := rsa.DecryptOAEP(sha256.New(), rand.Reader, privateKey, payload.EncryptedKey, nil)
	if err != nil {
		return errors.Wrap(err, "unable to decrypt key")
	}

	block, err := aes.NewCipher(key)
	if err != nil {
		return errors.Wrap(err, "unable to create cipher block")
	}

	gcm, err := cipher.NewGCM(block)
	if err != nil {
		return errors.Wrap(err, "unable to create AES GCM cipher")
	}

	jsonedData, err := gcm.Open(nil, payload.Nonce, payload.EncryptedData, nil)
	if err != nil {
		return errors.Wrap(err, "unable to decrypt jsoned data")
	}

	err = json.Unmarshal(jsonedData, &dst)
	return errors.Wrap(err, "unable to json unmarshal data")
}
