package amazon

import (
	"context"
	"crypto/rsa"
	"strconv"
	"time"

	"code.justin.tv/chat/workerqueue"
	"code.justin.tv/common/config"
	"code.justin.tv/cs/pong/logger"
	"code.justin.tv/cs/pong/tickets"

	"github.com/cactus/go-statsd-client/statsd"
	"github.com/pkg/errors"
)

func init() {
	config.Register(map[string]string{
		"amazon_client_ticket_sqs_account_id": "",
		"amazon_client_ticket_sqs_region":     "",
		"amazon_client_ticket_sqs_queue_name": "",

		"amazon_client_ticket_ack_sqs_account_id":  "",
		"amazon_client_ticket_ack_sqs_region":      "",
		"amazon_client_ticket_ack_sqs_queue_name":  "",
		"amazon_client_ticket_ack_sqs_num_workers": "1",
		"amazon_client_ticket_ack_handler_timeout": "30s",

		"amazon_client_twitch_ticket_sqs_account_id":  "",
		"amazon_client_twitch_ticket_sqs_region":      "",
		"amazon_client_twitch_ticket_sqs_queue_name":  "",
		"amazon_client_twitch_ticket_sqs_num_workers": "1",
		"amazon_client_twitch_ticket_handler_timeout": "30s",
	})
}

func (c *client) CreateTicket(ctx context.Context, t tickets.Amazon) error {
	err := c.ticketPublishHandler(ctx, t)
	if err != nil {
		return errors.Wrap(err, "unable to publish ticket")
	}

	logger.Infof("created Amazon ticket from Twitch ticket %s", t.TwitchTicketID)
	return nil
}

func createTicketPublisher(encryptionKey *rsa.PublicKey, stats statsd.Statter) (ticketPublishHandler, error) {
	pubConf := workerqueue.PublisherConfig{
		AccountID: config.Resolve("amazon_client_ticket_sqs_account_id"),
		Region:    config.Resolve("amazon_client_ticket_sqs_region"),
		QueueName: config.Resolve("amazon_client_ticket_sqs_queue_name"),
		Version:   1,
	}
	pub, err := workerqueue.NewPublisher(pubConf, stats)
	if err != nil {
		return nil, errors.Wrap(err, "unable to create publisher")
	}

	publishTicket := func(t tickets.Amazon) <-chan error {
		result := make(chan error)

		go func() {
			b, err := encrypt(encryptionKey, t)
			if err != nil {
				result <- errors.Wrap(err, "unable to encrypt ticket data")
				return
			}

			err = pub.AddTask(b)
			if err != nil {
				result <- errors.Wrap(err, "unable to add message to SQS")
				return
			}

			logger.Infof("published Amazon ticket for Twitch ticket %s", t.TwitchTicketID)
			result <- nil
		}()

		return result
	}

	return func(ctx context.Context, t tickets.Amazon) error {
		select {
		case <-ctx.Done():
			return errors.New("failed to create ticket by deadline")
		case err := <-publishTicket(t):
			return err
		}
	}, nil
}

func createTicketAckSubscriber(handler TicketAckHandler, decryptionKey *rsa.PrivateKey, stats statsd.Statter) error {
	numWorkers, err := strconv.Atoi(config.Resolve("amazon_client_ticket_ack_sqs_num_workers"))
	if err != nil {
		return errors.Wrap(err, "failed to parse amazon_client_ticket_ack_sqs_num_workers")
	}

	handlerTimeout, err := time.ParseDuration(config.Resolve("amazon_client_ticket_ack_handler_timeout"))
	if err != nil {
		return errors.Wrap(err, "failed to parse amazon_client_ticket_ack_handler_timeout")
	}

	handleV1Payload := func(data []byte) error {
		var ta tickets.AmazonAck
		if err = decrypt(decryptionKey, &ta, data); err != nil {
			return errors.Wrap(err, "unable to decrypt Amazon ticket ack")
		}

		logger.Infof("received Amazon ticket ack for Twitch ticket %s", ta.TwitchTicketID)

		ctx, cancel := context.WithTimeout(context.Background(), handlerTimeout)
		defer cancel()

		if err = handler(ctx, ta); err != nil {
			return errors.Wrap(err, "failed while calling Amazon ticket ack handler")
		}

		logger.Infof("processed Amazon ticket ack for Twitch ticket %s", ta.TwitchTicketID)
		return nil
	}

	params := workerqueue.CreateWorkersParams{
		AccountID:  config.Resolve("amazon_client_ticket_ack_sqs_account_id"),
		Region:     config.Resolve("amazon_client_ticket_ack_sqs_region"),
		QueueName:  config.Resolve("amazon_client_ticket_ack_sqs_queue_name"),
		NumWorkers: numWorkers,
		Tasks: map[workerqueue.TaskVersion]workerqueue.TaskFn{
			1: handleV1Payload,
		},
	}
	_, errChan, err := workerqueue.CreateWorkers(params, nil, stats)

	go func() {
		for err := range errChan {
			logger.Errorf("error while retrieving Amazon ticket ack from SQS: %v", err)
		}
	}()

	return errors.Wrap(err, "unable to create SQS subscription for Amazon ticket ack")
}

func createTwitchTicketSubscriber(handler TwitchTicketHandler, decryptionKey *rsa.PrivateKey, stats statsd.Statter) error {
	numWorkers, err := strconv.Atoi(config.Resolve("amazon_client_twitch_ticket_sqs_num_workers"))
	if err != nil {
		return errors.Wrap(err, "failed to parse amazon_client_twitch_ticket_sqs_num_workers")
	}

	handlerTimeout, err := time.ParseDuration(config.Resolve("amazon_client_twitch_ticket_handler_timeout"))
	if err != nil {
		return errors.Wrap(err, "failed to parse amazon_client_twitch_ticket_handler_timeout")
	}

	handleV1Payload := func(data []byte) error {
		var t tickets.Twitch
		if err = decrypt(decryptionKey, &t, data); err != nil {
			return errors.Wrap(err, "unable to decrypt Twitch ticket")
		}

		logger.Infof("received Twitch ticket request for Amazon ticket %s", t.AmazonTicketID)

		ctx, cancel := context.WithTimeout(context.Background(), handlerTimeout)
		defer cancel()

		if err = handler(ctx, t); err != nil {
			return errors.Wrapf(err, "failed while calling Twitch ticket handler on %s", t.AmazonTicketID)
		}

		logger.Infof("processed Twitch ticket for Amazon ticket %s", t.AmazonTicketID)
		return nil
	}

	params := workerqueue.CreateWorkersParams{
		AccountID:  config.Resolve("amazon_client_twitch_ticket_sqs_account_id"),
		Region:     config.Resolve("amazon_client_twitch_ticket_sqs_region"),
		QueueName:  config.Resolve("amazon_client_twitch_ticket_sqs_queue_name"),
		NumWorkers: numWorkers,
		Tasks: map[workerqueue.TaskVersion]workerqueue.TaskFn{
			1: handleV1Payload,
		},
	}
	_, errChan, err := workerqueue.CreateWorkers(params, nil, stats)

	go func() {
		for err := range errChan {
			logger.Errorf("error while retrieving Twitch ticket from SQS: %v", err)
		}
	}()

	return errors.Wrap(err, "unable to create SQS subscription for Twitch ticket")
}
