package processor

import (
	"encoding/json"
	"io"
	"strings"

	"github.com/pkg/errors"
)

type SpecialUser struct {
	UserID       string `json:"user_id"`
	Language     string `json:"lang"`
	JiraID       string `json:"jira_id"`
	SirtImpacted bool
}

type SpecialUserArray []SpecialUser

type SpecialUserMap map[string]SpecialUser

//get the user out of the map
func (c *client) GetSpecialUser(userID string) *SpecialUser {
	if user, exists := c.userMap[userID]; exists {
		return &user
	}
	return nil
}

//parse the user list into the map
func buildSpecialUserMap(reader io.Reader) (*SpecialUserMap, error) {

	var data SpecialUserArray

	if err := json.NewDecoder(reader).Decode(&data); err != nil {
		return nil, errors.Wrapf(err, "couldn't open special user list")
	}

	parsedMap := make(SpecialUserMap)

	for _, user := range data {

		user.SirtImpacted = strings.HasPrefix(user.JiraID, "SIRT-")
		user.Language = strings.Replace(user.Language, "-", "_", -1) // https://support.desk.com/customer/portal/articles/1096507-supported-languages%22
		parsedMap[user.UserID] = user
	}

	return &parsedMap, nil
}

//"fetch" the user list. local blob for now.
//TODO: Move to s3
func userList() *strings.Reader {

	return strings.NewReader(`[
{"user_id":"15817895", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"14127145", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21135664", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21443963", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28506343", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"36748494", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39008778", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40947838", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45932903", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52392595", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53615339", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57495012", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60976598", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73169263", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29162427", "lang":"ja", "jira_id": "SIRT-3916"},
{"user_id":"31512281", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36398857", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38640598", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"38824732", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40237959", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40878455", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43819696", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73638936", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23759076", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29790926", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"38227452", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43724111", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44196649", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48300295", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64660307", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"68985739", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73856768", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"97938105", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"11761105", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29178367", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29952847", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"42189339", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"50879107", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52009879", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52022476", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"61476444", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"32170443", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37115112", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42184444", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42740183", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43395056", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"58209198", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"58852971", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69312862", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"70749652", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71938889", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83088809", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90286898", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22817779", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"23870177", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24640968", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26133159", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35942822", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"65306917", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"101987736", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29213889", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"29290346", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32982005", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"34928651", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44420402", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55548429", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58142743", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"59799694", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61030543", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"66625885", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"67644841", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"20541020", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24029647", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31041788", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44030871", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46749745", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51107056", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54223951", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57741760", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"64277881", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66745380", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79535097", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"134333858", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"10422316", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36958759", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39332835", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44478654", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49098666", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49460699", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49936959", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57319562", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61668859", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66624012", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67209987", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39388311", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39564305", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40392752", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43813755", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68250362", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"84347151", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"23017540", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"23567348", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29194763", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31693679", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36486819", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46860368", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66958136", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"68734238", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"45375", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42350848", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42706613", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42976063", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44854818", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47455062", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49564024", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50892345", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52171592", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"61238560", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66851196", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89270330", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"134458926", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"27392163", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"37027412", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"38009707", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"45963939", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46548891", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49350770", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52888325", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"53579931", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97932220", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"103445191", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"108068529", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"23267288", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"60228289", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"73201819", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"146316878", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"24276763", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"30252482", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38590210", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43225913", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"44434203", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45715401", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60636311", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63717240", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65455606", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80113786", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88473769", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"144783070", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"26259929", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27617793", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"28086500", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29036637", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36619478", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42374674", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47851450", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51114927", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54278085", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69314638", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"101013574", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"101796507", "lang":"sk", "jira_id": "SIRT-3916"},
{"user_id":"29060686", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"30095062", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31766914", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42612865", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"46523081", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"52293294", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58137905", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"73187665", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"79387067", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79767158", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"145985397", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"10061206", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"25458140", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28382367", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29183994", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41040539", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42426959", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43866546", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44406059", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60344826", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70012134", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"85841291", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23209372", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32526145", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38241108", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48635505", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52201420", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54393118", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"55566505", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"58808589", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"61436224", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"61782991", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73766828", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78219369", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"81664195", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83325228", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"109566324", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"118344687", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"30879045", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38809831", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"41931600", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"45240415", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49391167", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54217506", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64554554", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"64715929", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"31456831", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39758103", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46240121", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64008800", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80480366", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"93430814", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28625901", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"32016599", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37616793", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"40554158", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57777506", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"62134753", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65909640", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"107534469", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"161462049", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41416494", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"86420260", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89166431", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"111520945", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28294141", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"30227322", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38425156", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40796986", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"57711297", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"59440651", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"64120215", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67544406", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68201909", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72874087", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"86012720", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26199825", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58475646", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69183252", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76240440", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36499364", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"41937032", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"51741279", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57277716", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75837408", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"113482469", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"119665015", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"137530734", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26446222", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28335575", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30192395", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55647342", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59640880", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"68298191", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"82189596", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38276377", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43291516", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"46198940", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46361506", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48936121", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64389093", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"65478158", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82906925", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"136687676", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41193439", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53215881", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54972568", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57025612", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"58496627", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64313725", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64792852", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65504874", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74943802", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75599532", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"127010169", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"152876960", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"41819701", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50879454", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"58880101", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"74002320", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"87769887", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"98014883", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"100611795", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"121506758", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"147302854", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"27050900", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27928643", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42602808", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45893174", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46945963", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47782502", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"55643301", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"142304434", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"29779113", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"31504592", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37191091", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45828172", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86459011", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"106592182", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"134398279", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23516387", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"31845921", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43669971", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"51983188", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52273755", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61217891", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67813591", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71775408", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88898062", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"12843756", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"28194056", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"35826403", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41383939", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"65908939", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74366371", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86086029", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"101324710", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"113615388", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"25853619", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"27234027", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"31506856", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32142917", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41222584", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"61477331", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64812917", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"69110358", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84340898", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93237949", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22738735", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"31316083", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40238240", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41792596", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52698267", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96673858", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"108513678", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"133936121", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31429231", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"32051686", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35714077", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"41124069", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43527461", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65396505", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77595133", "lang":"hu", "jira_id": "SIRT-3916"},
{"user_id":"114214297", "lang":"es-mx", "jira_id": "SIRT-3916"},
{"user_id":"152483936", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"35620614", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42668673", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46946376", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"48826827", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"55056956", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57997483", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62960437", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81244322", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"96650519", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"128337179", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"11897156", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20802201", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31832636", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"37871341", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38630582", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62163416", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64313247", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66980356", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70313509", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71155920", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"31172842", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31651424", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"33340101", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36667927", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"53322977", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"56825150", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"89514346", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36709773", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"40035360", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"42300569", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46433273", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49822592", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"63984537", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"76215683", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90121429", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"25118562", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42333456", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"53353065", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"55295881", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75190038", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79721302", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90064525", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"43286351", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"57117529", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61647672", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75359229", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77026041", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79590671", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88097704", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91156654", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"166578941", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"36612285", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41429319", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47316388", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47385083", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"63024851", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"73963231", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74402806", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"106582008", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28732752", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40397064", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47618749", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84428483", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"23794451", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"28180693", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30943316", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37465841", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"52008838", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56326582", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59300753", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62643574", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"68945519", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"156974114", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"10990089", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36138549", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37974282", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"45540872", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57657258", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58142651", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"64756920", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"83105204", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55858264", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70014027", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"71401727", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71423581", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"76148071", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"8277072", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"71365913", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78988391", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"12252216", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25877830", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36981900", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37301213", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38147024", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38257638", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"45584929", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48135074", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57183593", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61226347", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67570227", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39557135", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44706436", "lang":"sk", "jira_id": "SIRT-3916"},
{"user_id":"45086946", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49855328", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54390262", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"56310934", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59341876", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64808322", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"86102570", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"122518385", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"28019688", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38763300", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44713295", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58830820", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"61114614", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"65903429", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93688716", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"114402336", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37222065", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41298169", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"44159871", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44261733", "lang":"ja", "jira_id": "SIRT-3916"},
{"user_id":"48322426", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"60868937", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"62340124", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24178649", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28093133", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39235821", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"53424410", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76440263", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78048951", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97263957", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"135384957", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"137227746", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"32735112", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36964374", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38769337", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40102698", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49744614", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54322235", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54354897", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95013961", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31181093", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50543883", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53003065", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65833375", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72411475", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"72764112", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90852676", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"18785568", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37044728", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43242167", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49255969", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52272566", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57174543", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"61932807", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"62565931", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63280420", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65756938", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67654107", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"81283659", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"23880906", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30022562", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"45066186", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51005597", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"55688296", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"62016802", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68663561", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"122371812", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"42335288", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49938258", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"53535848", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"69874852", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"132673025", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"10514143", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"26666505", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"36061552", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38461708", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"45076097", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52887453", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54257875", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58553181", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62190346", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70115460", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"42812658", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43943812", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50832483", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51207857", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60704775", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"62328204", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"69244542", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"46335308", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49461375", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54013470", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60775367", "lang":"hu", "jira_id": "SIRT-3916"},
{"user_id":"68601712", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69352999", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23216314", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"23810532", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"26830586", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"30846079", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"30916944", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"43757920", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"47792755", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48390960", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53768650", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56474852", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57986244", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58711538", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"126024125", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"153752816", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52612318", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53850892", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66986885", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"12868886", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27078549", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"31512328", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"37270770", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44517153", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"44618372", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44885559", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48460920", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"52532209", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61509253", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68762114", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"85954323", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"90442948", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"114930715", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28121586", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47081378", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51255426", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70796829", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71290978", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"83930298", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"87751996", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"88263141", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23166736", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23667441", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24236958", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"25904408", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"36550506", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48366188", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51595504", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52983355", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"61809943", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69913348", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76273969", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"81609640", "lang":"hu", "jira_id": "SIRT-3916"},
{"user_id":"82876486", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"5475943", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29394551", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42396540", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59507798", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62476481", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"22139700", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31655301", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"31738021", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"43589204", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51573196", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61793292", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"65621980", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77829817", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79563311", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"23443294", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26217505", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32397403", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37130715", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"42001338", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"52697217", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52893359", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63690732", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"67529429", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"77345489", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91952266", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"14704660", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"15941176", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24728917", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"29866529", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"31844968", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40932166", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47782882", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"48925462", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53901238", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58901400", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"15195066", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30642304", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31430402", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32248160", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40251283", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44615909", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48229203", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73235141", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32492713", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37952090", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39133247", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39242739", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54447016", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59294969", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72058608", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92982522", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"17324860", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27157144", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28894255", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"40408937", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"49069581", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"51548975", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"72849525", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"110039706", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22129117", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30569371", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"42228727", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44637677", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"56100587", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60858899", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"75384889", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41348326", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60532082", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"66969365", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73639442", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"8711010", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32453507", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52254535", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54363504", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63410551", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82242970", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"24868086", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26109918", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27411295", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36571205", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46753319", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50795318", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"57870979", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64694483", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"137143191", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40741072", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41038043", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41576475", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44391243", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50225071", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"55050619", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56759361", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"70995785", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"115384464", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"8747191", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26181477", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42963682", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43830356", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"45598402", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46930894", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23552702", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29736055", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31010214", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35944643", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36091835", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40422473", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41489511", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56568209", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70470751", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"84351168", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"133231893", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"24202282", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29267541", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36930940", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42318017", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"43505668", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"43583604", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45232009", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"51395362", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"52749788", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55013873", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57438652", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58856096", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64361581", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"82969915", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"132951588", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22960911", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36184209", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44298270", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71140638", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"72271449", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90738866", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"133871379", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"21840616", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32527328", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42504057", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"42555342", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50498667", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"56470225", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"62563432", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"80234450", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82816994", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44770513", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61355652", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"113798942", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"9012871", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30538588", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"32253071", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"37274906", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39645368", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48833232", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"52155950", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"58613449", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97347405", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97657130", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"121659420", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29157717", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29375868", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"32739663", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43483801", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52188968", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53372754", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"59609487", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92234547", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"105956523", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"9235733", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36345930", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39754729", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41497306", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56498464", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58910723", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61409336", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"65674132", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"67233840", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80283338", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95509807", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31488488", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41304483", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"46448736", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49750299", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50340392", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54674795", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56776719", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59419075", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64692791", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"67315927", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"89756518", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"91462503", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96441885", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20846357", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20890876", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22883345", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28331953", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40366097", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42904215", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73792882", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79376196", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22440311", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27217751", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"33214528", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40129319", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40796625", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41696199", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45392871", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"54852607", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57236495", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59763335", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65636435", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67806085", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71006036", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"128479231", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"155210284", "lang":"ja", "jira_id": "SIRT-3916"},
{"user_id":"36038858", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50289747", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56806100", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"117099874", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30672745", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31372363", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38813263", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42326102", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46576874", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46701309", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"48151733", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"50783449", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54604135", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60425567", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62706135", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84468899", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"89716570", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"119427464", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31548874", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35087023", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"40583201", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46028489", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60775262", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67129038", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"73992711", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"111636197", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"112004208", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"22401358", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25709348", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"35889585", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39714694", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43413779", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70850699", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87901559", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27521249", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51664457", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"60900813", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65717574", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78409670", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21429840", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22200081", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"26587429", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37110897", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40208574", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"43217917", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"49057209", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"53248084", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58549373", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"152942476", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"157118957", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"4539552", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36353752", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"39141793", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43318263", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45662243", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52642204", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"57934918", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"73664340", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29411668", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"41778110", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60466912", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46243718", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46311758", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48605838", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"49541705", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55119705", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69255151", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"80195475", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83073643", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88045177", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"98598887", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38717005", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40235518", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41783223", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55642458", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64223970", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"85568605", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96758206", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"125996398", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"20773527", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24378421", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28682277", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30481631", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32104271", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"37331117", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45378439", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48964992", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49461453", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49831276", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65249445", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65397321", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83111736", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"110533380", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"138633370", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"30250296", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31830073", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"39103542", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40051504", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42907519", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46244455", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"50284398", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50705028", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"51857867", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66059157", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"45754258", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86082801", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"22844532", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55468159", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63063627", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71550276", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74630410", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38209220", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38950649", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"39647543", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41792401", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"44257339", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57115216", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61163757", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64579773", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66792732", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77354517", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80712117", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"117355104", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"5133306", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21484693", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31181182", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"32878371", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38421618", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39393036", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48441961", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51264337", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58634713", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60710809", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62394718", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62954358", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83941782", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87618021", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"92507839", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"101275512", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"118353154", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"187853932", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"24000429", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"42701014", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"47788640", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"51230613", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52865689", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58282310", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60155654", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68504452", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"68640084", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"73512428", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74979404", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94672635", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21910436", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37157512", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41173974", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65678033", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79085637", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"137484643", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36684775", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37185656", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40124154", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51731876", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"53951294", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55137465", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57324257", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"108722926", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"19554808", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"25824778", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"27237812", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32131330", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"35724477", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46996751", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"55869047", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"58335168", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"62225926", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"69540289", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"113699593", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"48303021", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"58792674", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"59805102", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80796451", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86001793", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36093365", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41967231", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"43123619", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44763426", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"49310906", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50646467", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52680906", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53399381", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54189988", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54646233", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"55060347", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"63420784", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66556726", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"69641190", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83742991", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"31264802", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"35928006", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"36166287", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"53823363", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59274517", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66822237", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67954525", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39087257", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"46046149", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53112548", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54358970", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68902954", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21733581", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49359702", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62092090", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62548878", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69873371", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"7804948", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30187079", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"38088808", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"42771500", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43536199", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"50525022", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56163279", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60098026", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62184815", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63474399", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"117597868", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31616829", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41931842", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"45492896", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45854505", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"61285519", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"64867027", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70973038", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94370496", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"153045451", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"15322358", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23438972", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27563615", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39096952", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40130143", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"48865821", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"50710531", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62153526", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"67539021", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67877057", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77777087", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"157921923", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"24208138", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41412678", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53543964", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"56697203", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58886863", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"62536708", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67552751", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76310863", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89694243", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"13552832", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32102526", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"38783109", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"46450059", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50737403", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"55318639", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64837984", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93901687", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"142542553", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"143998254", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"12826", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45211789", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45328785", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54095004", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67068504", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"90165610", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"357923", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"4996923", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27079765", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"28209837", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36785551", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"40734017", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"52411606", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55931136", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60360708", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63777499", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64088694", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86547084", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"18728210", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25365731", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"40209323", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42152196", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52026993", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52669958", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66739165", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"32346347", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36347732", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41960285", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43631296", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50613101", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54963132", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"55522810", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64736928", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"124281648", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"144410694", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"22414847", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23962090", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44864970", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59762118", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81137370", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"87651837", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"44307737", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44426514", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"50578234", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50701928", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51232715", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"51837942", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54479665", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61340774", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61391465", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67289124", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78555139", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78994679", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26504354", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35636464", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39934557", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42091142", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42753214", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49483836", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62234895", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62773165", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"78327181", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92149810", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"24311474", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37029790", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39607327", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42800169", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44563074", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"52025878", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55013750", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59632944", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65863054", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"232060", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37665359", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39752047", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"43786483", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61497384", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64121012", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66065296", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"88058967", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25567999", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26557012", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"26931005", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32620389", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70256441", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"70856607", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"77499845", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"131780396", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"135594567", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26375049", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38747463", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39535322", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40167739", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"43518422", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46249909", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52606669", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63199306", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"142662928", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26982476", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32422416", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42711314", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44974026", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"45229856", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49383002", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"61831475", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"144422493", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"22816945", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27725658", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37196048", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47167553", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49085022", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49761444", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57910251", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67207162", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67955226", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97493485", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"28224454", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28508595", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37286578", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45271474", "lang":"hu", "jira_id": "SIRT-3916"},
{"user_id":"47622446", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"50054288", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55784296", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57431498", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76305264", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85594920", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"109859324", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57929181", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58949952", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60099281", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44388675", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48026005", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56052144", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96814951", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"153727574", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"24848220", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"28483161", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41013355", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54186240", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55252411", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"95162712", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"22985931", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43603419", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"45963757", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53119391", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"55993782", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58385864", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63479458", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26777739", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"31533784", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37104827", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50066666", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51753176", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52926395", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56812841", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58999156", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59320733", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"64475794", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75885541", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78517916", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82907708", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29766999", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45685832", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48864920", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60726465", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"96664629", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101794047", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"108243919", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"116328241", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"7920047", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23429745", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"29774436", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"38808867", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46356653", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"47522476", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50574053", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"61387579", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"63825731", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"70496679", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91072211", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"10840477", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"17936068", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22481989", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28756240", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37588904", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39417718", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45908493", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46049489", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52180233", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57864278", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76923790", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86258195", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"14306193", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45759110", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"86144944", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"133027761", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28459225", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30149171", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31528282", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31646683", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38578647", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"49961496", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59432906", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78594549", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"111328020", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"115632031", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"7254680", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31112870", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36187471", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"39011751", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"43699536", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"44838026", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"49434485", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"79919885", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88365100", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21000125", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28833678", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"31357360", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"35095114", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37646785", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49475143", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"54611502", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"65880537", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68537798", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72765078", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"78707282", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"133586537", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26146079", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"28148742", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36920747", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38616262", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40103960", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"41547757", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48378884", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50185928", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50476113", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58180007", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"58263930", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"60625230", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60855930", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73181729", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"31745816", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40749646", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48851442", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"49850665", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54690217", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58507658", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"68650091", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"74870250", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76068606", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"78598731", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"110295366", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"112090790", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"23601307", "lang":"zh-cn", "jira_id": "SIRT-3916"},
{"user_id":"26963025", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29315721", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"62809668", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66868101", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"70340423", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74956901", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"122729925", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40990503", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51570210", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89865176", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25158382", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29157943", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"29790181", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46635280", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46909291", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55215092", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"55832766", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57658019", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71027780", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80375329", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"137259043", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"27787394", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39512040", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"39591328", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"46162423", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53381601", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60707017", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72026242", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"82555017", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"23515569", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29219402", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36428805", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42091600", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43624425", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43922214", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59211413", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62051532", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66310735", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"102514170", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"109750636", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"145852492", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"37011924", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"37359333", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"57381880", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27971774", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31787311", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40019211", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55782807", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62801219", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93335144", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"11847181", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"44806759", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"48407145", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48897313", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49487750", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50293994", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50802675", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58713099", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"38448360", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39371011", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"42857777", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43231372", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51002736", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"60634216", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94694063", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"23969535", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45720749", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54440328", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54667485", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"59266424", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78776381", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"87517468", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"42670575", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46642581", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"135801009", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24505828", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41260104", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"45811858", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54572160", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"59717810", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"62585849", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64365264", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"130808035", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"137574839", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"145742099", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"24233423", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"36762630", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"37137439", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"38693833", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42105045", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43612565", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"47673034", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48848790", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61294132", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66365609", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68544729", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72989938", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77711065", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38565008", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47221069", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68766846", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71099013", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76516457", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"93064696", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"136763481", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24631685", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"31429178", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60495043", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86978736", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91102840", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"132948142", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23540407", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26177401", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36783863", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"38399442", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"39762274", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45018740", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47922996", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51849994", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55473737", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"59389227", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79613118", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81880033", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90611753", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22380576", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30246279", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31118356", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36466722", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"38321903", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45942213", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"46408485", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50718436", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55503111", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60888930", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72283900", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89667896", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23466988", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25224335", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"38610904", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39423139", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52879908", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"54987307", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60992315", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61994519", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67537281", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"72215238", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39402065", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47978946", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"49903482", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"50724728", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51453423", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52505575", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55697631", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73214971", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90357721", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38036027", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47359237", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51322519", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58430073", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69676804", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70731701", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"82239298", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91865044", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"25984446", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38449299", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41736804", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"49578287", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"49660078", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"53656933", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54273030", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60633015", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"71910941", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"73004952", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"84460698", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"133630087", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"28340135", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42752239", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44316260", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"59384570", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69589275", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"88679997", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"6060618", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26991613", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38013718", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39195490", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47694522", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48013324", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"49952382", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58506546", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65941511", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81632249", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"122618071", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"29514075", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"31174693", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37545131", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48842660", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"55859980", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56967214", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58079303", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58155864", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"83015275", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23716436", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35856728", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"48052688", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48531554", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52182636", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"58235841", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60066059", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95869154", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26050929", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37747712", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45288002", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49329020", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"100163533", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"23491621", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25904902", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36714399", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"43169973", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49406741", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"61119434", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61332190", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72697673", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27842782", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32529879", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32575412", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40444012", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"45229054", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"47718357", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51466249", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57379710", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60233256", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"60650421", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62651783", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65582161", "lang":"en-us", "jira_id": "SIRT-3916"},
{"user_id":"78090479", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"147292685", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"28395944", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44556651", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46981304", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50947422", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52395778", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60238572", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"73683727", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"77079185", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"84061558", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96857425", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"119753524", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32009461", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41512206", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"45417511", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46038957", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55755273", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71148672", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26962893", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30707866", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37860445", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59085834", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60416891", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61818490", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"103903291", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"29005425", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30111039", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"30518952", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39526540", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"46842631", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47701204", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51048011", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80509828", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"98352255", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51583614", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"62863155", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"66281558", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68274024", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71401677", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"90966791", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"123878118", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36005081", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"39303206", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40278494", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"42754683", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57344987", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"58206532", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63810019", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"66996436", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75204547", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76386086", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79815461", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93391973", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"145746374", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30229102", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40125956", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41672528", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"45477920", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48978763", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53798404", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74161756", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76549029", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36039811", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40092752", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"43119976", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44081390", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77193263", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"80301304", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"85389366", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96606976", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51386047", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56235661", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58776321", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94749194", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101851557", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"14717297", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27160592", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47107205", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51600155", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55662349", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59684589", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64864461", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"81733969", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"155471112", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"9072112", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"19222647", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32279477", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36026894", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36314911", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37881206", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"41440186", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42053065", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45031725", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"48264453", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"27138097", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"38018993", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"40281432", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43659372", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"46702585", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59123049", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75040172", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28289412", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32965962", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37025431", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38794107", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"46660439", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49967326", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"52113532", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"52398104", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64345172", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"68929783", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73518861", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26010051", "lang":"sk", "jira_id": "SIRT-3916"},
{"user_id":"26570044", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42243026", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"48592795", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49961064", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65790871", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77707729", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"104651247", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"168699923", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"26542483", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28153670", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37919574", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71509792", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"78410431", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"181668748", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"30244276", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"39922024", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"41782916", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43821087", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48299918", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55275330", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"61326077", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"61586541", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"68605766", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"69353170", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78358978", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"103495546", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37159546", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"42080066", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45261005", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49591838", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"51097446", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"52597516", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"53401397", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"54124254", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60634550", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"97289243", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"133220173", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"26893862", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35031151", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37121843", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50580593", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55690892", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80599379", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"89116611", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"26226671", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35787496", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38189243", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39508073", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39776538", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"46677649", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"51211652", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57983322", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"61010835", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"7057599", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"28555885", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"43492114", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"56547060", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"58838913", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62712558", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"63356519", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"64521931", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81540006", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"28369357", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43448813", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44969340", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48625983", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"49895068", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50331502", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60443069", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62233162", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65329602", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79081723", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26181172", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"31664684", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45338472", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54286214", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"75441973", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82174559", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"85439485", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91393254", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31330289", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35736723", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"49749650", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"52004242", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53933231", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59662401", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85627721", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25274574", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29811655", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"39100125", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"61120504", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31989012", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40034021", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54898068", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55243635", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57174896", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"72332594", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76775834", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"131891602", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"145947814", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"13394421", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"33274250", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"41215732", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62421100", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"142350397", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"172306990", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"31945897", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32693207", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"36358929", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"51226070", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56476139", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62510803", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71799716", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"75080239", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82186680", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"113431061", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"114674688", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28174253", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"30820346", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"47167223", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47669812", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"49509859", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56608490", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57466576", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"58962208", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"83181528", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"9846758", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25046589", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31362337", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38624947", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41925308", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43026884", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44566682", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49898633", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"56688866", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58692519", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59404159", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67373181", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83258530", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"116031590", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"132310923", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"147404595", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"26968568", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"27022195", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"36333949", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"40739393", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41232055", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"42930843", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47263526", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"49220160", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52555426", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62196265", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69742525", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29051870", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50791633", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50978799", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53186474", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55529949", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62706124", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70164060", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22092298", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"32107639", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"32673309", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"37885960", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"37972862", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39469769", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40134836", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"46984586", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"48174677", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55871191", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57431352", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"63072219", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"72876177", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"103281857", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"25722748", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73182708", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87894008", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96145274", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97772818", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"99957759", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"103902449", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"125147757", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"131919193", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"38632927", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79353708", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"87413997", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97843962", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"145026269", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23446870", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24355460", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27339649", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"31197091", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36322682", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"49867632", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"59839953", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"64215353", "lang":"ja", "jira_id": "SIRT-3916"},
{"user_id":"91440996", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"38746347", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55001264", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"69096292", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92598562", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"106384651", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24300737", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29878802", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59797258", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62869384", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"164569572", "lang":"th", "jira_id": "SIRT-3916"},
{"user_id":"175545756", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37906665", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"46583254", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48328389", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"52297540", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"55856055", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"59069292", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61283352", "lang":"sk", "jira_id": "SIRT-3916"},
{"user_id":"62022301", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70310122", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"87912554", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37204193", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37723372", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40073536", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"48362281", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55058199", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68222156", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"146962133", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"14126697", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30890027", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37913946", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42120446", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92971842", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21517478", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36249312", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38652101", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45578686", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49283782", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"60880337", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29923478", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36502556", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38048649", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40989814", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54681827", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"56716886", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"61611256", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66949152", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76829798", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"116330895", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"63634162", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64652713", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30178079", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39939458", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43098221", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44715536", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69756118", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92512512", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"142498792", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"29373000", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32083620", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40561103", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46542980", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47510096", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48073449", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50862920", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53185259", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68716428", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"89068364", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36185116", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37657313", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55910223", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69553846", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70462905", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"15310631", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"28700995", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32004950", "lang":"ar", "jira_id": "SIRT-3916"},
{"user_id":"43757032", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"54212743", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55724501", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"55754731", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55796432", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"55813525", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58144405", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78554638", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"102388206", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"27533461", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28334733", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31786804", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37833666", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58905589", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59673017", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"61650152", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73179613", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"77128954", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"22370164", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"29033993", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46875169", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55388093", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79092069", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"29767678", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31619656", "lang":"en-us", "jira_id": "SIRT-3916"},
{"user_id":"37882994", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41062957", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42038591", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45615359", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"46992845", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52713349", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53191330", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28549232", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30600261", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"37573267", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39766793", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46907405", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47715970", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"69333699", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89673062", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"91234127", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"125271876", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"153066324", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"27972683", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32066570", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"56733115", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58928131", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60864760", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70814573", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"29545940", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31092734", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"32504896", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41860534", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49455794", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54804025", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"55844667", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61183571", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65814795", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74956114", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"118380209", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"133138375", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"41573614", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63859639", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"63956850", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67710176", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"82319981", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92727041", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"11802382", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"14666168", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23070659", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38942371", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"60132788", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"72289707", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"157117599", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38319663", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41708502", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"44021622", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45714095", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45845839", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"50467529", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52496535", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"53211865", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55904380", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61451833", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76446919", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84291930", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"91262507", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"96693137", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"99563223", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101947068", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"109295688", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"16230322", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"19629732", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"22783976", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25057075", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"30195394", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42576186", "lang":"ja", "jira_id": "SIRT-3916"},
{"user_id":"48728429", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55931047", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"60962190", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64773139", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79829379", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91863957", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"105831232", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"147921123", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22933162", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28193870", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31482482", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36700799", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41399072", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"46916173", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56856966", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"62410464", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67910405", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"119571137", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26010918", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27036511", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37828895", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43281658", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49150415", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"76858891", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20816892", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39882134", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44733350", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"45754170", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54097642", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54551730", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"56890127", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57854898", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61958195", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94474289", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"125586437", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32573569", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36745456", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"37616824", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53371828", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54470404", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"70289520", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74849068", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"97307866", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"36007950", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49134802", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"58581569", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67938029", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74535022", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"83325354", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"18074328", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31444564", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"35083469", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43104896", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43483483", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44310969", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48416651", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56048772", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64352895", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"68267249", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"76862248", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23071507", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42610280", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"74123383", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"90596771", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"90628600", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32635075", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"37672588", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43497346", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"51065118", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59687932", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61017510", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"92864845", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35700542", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39024957", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42924677", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"43413099", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55293093", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63534382", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66092143", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"99780322", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"100626595", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"146059590", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"153650001", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"30566314", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30810363", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44481459", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"46309353", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59174771", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60492460", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70832502", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83168135", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89845448", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"25543694", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28663388", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29001430", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"38295171", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"53886921", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85118447", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"87415479", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24967603", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61494055", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67708819", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68538217", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"126186305", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25712785", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38617117", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"45210505", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54660420", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"55652315", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"59586636", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79456050", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"116579950", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"120167101", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"121950216", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25403162", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26969976", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29235003", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30219129", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"32151842", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37789221", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"37795424", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40734526", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54452063", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70443386", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"71364655", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77096573", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"132962273", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21053217", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27755292", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29278831", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36974909", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51744900", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66882281", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77730446", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"80429907", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85397463", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"132349194", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"26220928", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"32808872", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35783804", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"44929725", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51569382", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52038086", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58200581", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63481253", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71630790", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"104535973", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"117269269", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"132778462", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"30552109", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"32624117", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48931116", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49822147", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55782780", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57321841", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"25096613", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25220444", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26550734", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29648206", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35946245", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52956111", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53538052", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"159511430", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"3296592", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37049860", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37815736", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"44661067", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53554794", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57055295", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"66437441", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73403665", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"108496571", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"124507740", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"25625964", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"35615986", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36439315", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"43692013", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53015658", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53789480", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54626690", "lang":"zh-cn", "jira_id": "SIRT-3916"},
{"user_id":"59262079", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68772002", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"72155752", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"73930433", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"84071358", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26085270", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36084768", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38097404", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"38961852", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41681871", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42671231", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"47233211", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59062561", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"60081491", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"63249841", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"64728980", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87407516", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"23283643", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31301032", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32910987", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35787784", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"37898036", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42455846", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51107831", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52829158", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58807733", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59717503", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68795164", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"70607200", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72186939", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72921237", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"108212009", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21488359", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35065542", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39849225", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"47603787", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51539288", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"53108465", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64507648", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65570369", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35040236", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51147833", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51223540", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95604679", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"130785533", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"23278374", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28039497", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"30080865", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31035129", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31066504", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31459678", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40867503", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41896726", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"64876993", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81199940", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82727970", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"84758445", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"105513574", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27621943", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"39912179", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40970284", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47277813", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59901910", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71121389", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"106066397", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37052605", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43313963", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45775956", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"46157102", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49485944", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"55416530", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"55665059", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56262315", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69158021", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43486684", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"59468562", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62064623", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"76639347", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76751451", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"32601694", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35911762", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38088223", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39776959", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"58902697", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59986291", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60353001", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70481491", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"73426800", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"77497071", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"26486934", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32880092", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"32980164", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"34501511", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64334536", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77096344", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91470973", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"105818187", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29089588", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29762531", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"31763448", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"38954179", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"50610318", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58141105", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61958436", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71921872", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95810511", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"118226075", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"143782485", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"33162951", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36180295", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"36997887", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41904021", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"43933405", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48392794", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"57862808", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59951909", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"73162523", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91137296", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92039146", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"114023144", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22251569", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36518871", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"51373064", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57581687", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72809503", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"84534968", "lang":"zh-cn", "jira_id": "SIRT-3916"},
{"user_id":"112718738", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27293001", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37316366", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39898478", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40491265", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"41245072", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51932212", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"57219173", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"73890632", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89657847", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"103629660", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30372150", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36378848", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"37385429", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39105027", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46231706", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"55433572", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76433625", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79604378", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91096909", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"6661046", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30903394", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35816092", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"50200590", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"64593212", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69044749", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"30143623", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37913455", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"46565109", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"47883461", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73368130", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"77415924", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"83904447", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"27299309", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38553701", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"39459577", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39543557", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45278859", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47479793", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49743099", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"53058059", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62823960", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70851678", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"74096411", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74564635", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77003113", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"84427021", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"153781215", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31376407", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36583193", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47156914", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53127681", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56463322", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92469252", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"17240511", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"30587451", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41463454", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47903025", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"56399396", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58624110", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"68162370", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27602858", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"36848494", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"49162221", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55254680", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"61973152", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"69516643", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71613913", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85541305", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"133753338", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"36102146", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39208132", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"43320638", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43546189", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"54642612", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56351581", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"61526069", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"63711079", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36365617", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42482279", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"48644395", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50365062", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"106076804", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"153845958", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"9253220", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20173781", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"39124769", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"43371305", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45277187", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48950662", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50793917", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"54313692", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66730599", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87369781", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20486408", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26042225", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29033557", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44402552", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51697414", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53521245", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54998332", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56181875", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59205671", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74809898", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28538842", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36984176", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"45133295", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"51620841", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"57655772", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57825222", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72103107", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"86254599", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88919407", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"94132144", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27515816", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44671214", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"45015602", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66180130", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75198291", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82153193", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26379687", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28491436", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37299076", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"42287944", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43038955", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"50434055", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51314188", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"53441362", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55545960", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55702850", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61628441", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71022181", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40929724", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"45056562", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49543022", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58278070", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"13470738", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26414001", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27814025", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44346467", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56849855", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88171049", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"116865292", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"121532982", "lang":"th", "jira_id": "SIRT-3916"},
{"user_id":"37665308", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41032185", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43988090", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44041565", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47017709", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49225169", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58885835", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61572435", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69930659", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"135626769", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"4811154", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"18403666", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"30542132", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36114207", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43705713", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46305884", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"50682384", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"56775418", "lang":"en-us", "jira_id": "SIRT-3916"},
{"user_id":"59336873", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"66557755", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79514545", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92653854", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"135397032", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"29913305", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37361120", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41124427", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41922637", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"56343413", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"69095547", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"71168126", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"155211042", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"20600021", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"22892906", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"26854672", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30468857", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"38178386", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"38900241", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50466411", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55163651", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56543814", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69464459", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"69716367", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83178715", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95901699", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"148173057", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30319975", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30977827", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31284401", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32680465", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49675137", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"54509569", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"60877776", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89927922", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97734615", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"133372065", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"135981615", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"12519", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27392269", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40102060", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47414100", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"52699423", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"65474915", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69317448", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"105697698", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"27913876", "lang":"ko", "jira_id": "SIRT-3916"},
{"user_id":"30408161", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47348321", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52047799", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53524279", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62585308", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"62828439", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64233300", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"64884885", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"103226605", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"9232896", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32534750", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"37641126", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"44385810", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"47657082", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"48752377", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"51380124", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51982541", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52229704", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71795798", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101553748", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"26926258", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"37585241", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46089180", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50381005", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"51026659", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61375784", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"42235086", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50515870", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"52004146", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56450362", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73188950", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"114083812", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38360462", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"41239068", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"51993509", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53903175", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61783731", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"67452381", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80917493", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"24421779", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"30264374", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"54772040", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61162012", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66482009", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"136213037", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21293733", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35830152", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37738057", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53766534", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56007320", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"57912607", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59160360", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"59555154", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62669498", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78702118", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"112637361", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"32294370", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"45838137", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47629322", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48049350", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"48751506", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"50707336", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52895452", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61053163", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"63815706", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74677006", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"75676025", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85478690", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87414811", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"13258802", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21287861", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44257248", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53906013", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58125772", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88991216", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"43389020", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43623309", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"44224859", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"47593251", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"49802645", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"53513430", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74222425", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"78893940", "lang":"hu", "jira_id": "SIRT-3916"},
{"user_id":"80704776", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87567861", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"136751316", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"126498", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"29936829", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43234527", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48961282", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"52006088", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52179758", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54515603", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57418598", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61944321", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62281602", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80280462", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"84264354", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86538014", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"93256766", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"145581448", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"146840624", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"26510881", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29887916", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"52789858", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53913494", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63075435", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"77491467", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81598302", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"17568595", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26071681", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37585168", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"41635412", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"44045879", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47863157", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48518094", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"55436491", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56006088", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"69089646", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80530923", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"112205624", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36711371", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"46034825", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54420020", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"57628818", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88739603", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"112717103", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"25892566", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37091024", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51675737", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76554128", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91940838", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"135119639", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"27085535", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35089770", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"73944216", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79647026", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"30045615", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36499142", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37346765", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"42137690", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49909719", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52840570", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56441840", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57812258", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61958469", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62506006", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"62706120", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63279239", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80713305", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"27695721", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45176328", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69310622", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"27794385", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"51185466", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52014902", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61377020", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75646199", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86756426", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"123555253", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28633374", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"34382677", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"37229083", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41508989", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41780983", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"49984692", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58622981", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61924654", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"24578437", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26274979", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29514354", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38646632", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38818301", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52372936", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52972913", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53393264", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58099819", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62216909", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"65651269", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"68122209", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"77530696", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81757115", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31528930", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"32040183", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36208257", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38353316", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44137690", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"47394922", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"54535504", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"57243145", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"62068602", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"70009971", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42871035", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44802161", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46535860", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47805056", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48201987", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"52313795", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53644740", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53794737", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55079674", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60752086", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61800143", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"71400832", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"91015171", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"21096366", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"29425855", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"31059367", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32388710", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62480149", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"66706190", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88126980", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"24681472", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35824762", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40826049", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"40913816", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41830833", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43204445", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44367510", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46020837", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"46642250", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"47349529", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"51237436", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51250530", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51505153", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66414869", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"74779470", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96393511", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"106049756", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23932568", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26158435", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38402589", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"78345414", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"120619103", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25692895", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30998628", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31591609", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40940366", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"41377798", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54688547", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57474335", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70273885", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"74036003", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91513161", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"131388646", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26558657", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49592467", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49998625", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67493947", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80993076", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"81031310", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97555045", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"128101185", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"137239891", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"8062492", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52743300", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"62954145", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"144858560", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"31947757", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"32519119", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44124368", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46302503", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"109828380", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"9460020", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"20624842", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31937659", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39086550", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"47585485", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"54972156", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58708675", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74026534", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"99281400", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"131554374", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"133709510", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"21137134", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32877280", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36695687", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42368280", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45033437", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51664836", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56593664", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"59215522", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65608372", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79851776", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"107725934", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37830102", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41060918", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41145774", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45019971", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48438935", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50064414", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"54153665", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54654948", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62352234", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"63455054", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"92458882", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"124367830", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25223146", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"37490536", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44042221", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"47288140", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48876942", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51827558", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53596756", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"64912026", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68086005", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83498644", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"13582930", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23862428", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26089171", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29797074", "lang":"ar", "jira_id": "SIRT-3916"},
{"user_id":"49713369", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"50458425", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"56069206", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56523479", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"60532143", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62698239", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"95498185", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"24031882", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"28037936", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"31044804", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"45676811", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49314008", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52129490", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"55332912", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57475538", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"66222498", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76967800", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"79613228", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87790259", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"100412248", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"103578654", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23586179", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"23612426", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24117230", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32230551", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"32682479", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51992020", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54223274", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68981575", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89377486", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"21288289", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30231466", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31228881", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"32291315", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32563013", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"46323534", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"53095537", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"57691265", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58556071", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61039949", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"61298325", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78162929", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"81879785", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"39556048", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"42482033", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58680707", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"132431999", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41513996", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"44625749", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51279524", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"51611292", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54103836", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55677400", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"61163858", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64984768", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78185643", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81032283", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30666848", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47111201", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"56829368", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59164074", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63876973", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"98408099", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22859264", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28090370", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28218792", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"30189508", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30681927", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"37132570", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38339268", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"40592258", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"45084412", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45934994", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"53399402", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"61394564", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"62152567", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88606617", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93573431", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"98916900", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"151382569", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"41915583", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55143672", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"57413171", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47011948", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51249347", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51843573", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"54479039", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60405193", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61852178", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62995544", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71761320", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"23152052", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27200351", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30835749", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40752020", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40876779", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47866437", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53885607", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54280588", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55646203", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"58670004", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23610891", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27847836", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32155119", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"36962946", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43316658", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"50908436", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59518711", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65714358", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73547054", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73835650", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"136192730", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22926665", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30180492", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36300554", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"43833373", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"46574335", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"52640159", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"57624624", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64654518", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81186755", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86268118", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90741685", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"28236713", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"28569556", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"37356376", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"42067406", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73481366", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"29423090", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30195602", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36380853", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41788637", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52141511", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62354392", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71690774", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"109628196", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"177585069", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40541443", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50877133", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55914354", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89111512", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"143466531", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"45978534", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"50432456", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72874737", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"79126909", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35058452", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40232608", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46269436", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"52804241", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55708864", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69726459", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"71121649", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83727381", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43093521", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45597590", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71359677", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82126679", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89899744", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"25181339", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36656770", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"46012701", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"55453781", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"67177997", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68132787", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71404450", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71434768", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"153737528", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"26860203", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38976288", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"43559750", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44803577", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47802833", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"58900245", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61189748", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66790304", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77722449", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37740778", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"41043364", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54312680", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55665493", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58484334", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"63277753", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73890219", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"75945004", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85407235", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"9056377", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36792850", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38023415", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39604685", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71968011", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"103198630", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"159122184", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41072737", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"45032003", "lang":"en-us", "jira_id": "SIRT-3916"},
{"user_id":"48577913", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"53902789", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60492223", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75286093", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83975409", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89765298", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92836426", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30187139", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39588319", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67858600", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"158423318", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"23681447", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"29534236", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"42930491", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44747133", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"44936055", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51541723", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51887369", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53519464", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68826155", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"16151006", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26217683", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"55437271", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57999722", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58831720", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68548650", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73182389", "lang":"sk", "jira_id": "SIRT-3916"},
{"user_id":"79339330", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"81480276", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"114598583", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31361952", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32745810", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44088402", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45772186", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46382201", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66543617", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"36345756", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39038760", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43658593", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44548745", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"54520566", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55602629", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62653730", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"72441295", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76074803", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"144950333", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"7244895", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40607434", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"50264309", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54367740", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62826884", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69857357", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"123391762", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23274001", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25268375", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"28227641", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31091009", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"40458518", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"61322193", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"14585676", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"25854532", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26453017", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"28104666", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28873042", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31542382", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38578697", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"38740804", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"48818161", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50078454", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"54334650", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"55855132", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81429096", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"91970788", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"144342000", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"26666254", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"43755270", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46521425", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51886365", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61542569", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"22068009", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27562026", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29295184", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29661532", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"39461389", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47907939", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65846467", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77931213", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"10789692", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"23650933", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27681398", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29326106", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32280200", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35826213", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38422160", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49670977", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53180232", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77125767", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78335816", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23372744", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24170228", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"45626262", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"47835890", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49227638", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"49570233", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"55431862", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"69504798", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"7556713", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"29708883", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"45928434", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50115200", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78752556", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"152637013", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"155375643", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"165510000", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"26278531", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26929369", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"29225126", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30204485", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35790122", "lang":"es-mx", "jira_id": "SIRT-3916"},
{"user_id":"40113670", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"41064011", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62152327", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65915200", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21920186", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32490775", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"36119890", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43114700", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"44019612", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45472828", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"45491205", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52316256", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54201353", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55610987", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63758697", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"99902945", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"106213211", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"1689320", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"7574178", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43160883", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46919016", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71683028", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75069633", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"102706510", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"147399702", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28571556", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"43558214", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51004660", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64578510", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74964368", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77231908", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"143650738", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40122228", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45802597", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"47573924", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47807267", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"51577911", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52220573", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52273830", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57965871", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61758808", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65591510", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78998500", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"81811115", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87549934", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"25466330", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"32474628", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"37419723", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44291866", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"48257930", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50904154", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"56746031", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60197543", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70806022", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"74106102", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"111358662", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"28477941", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"29289159", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32707753", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35630749", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46670331", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57979782", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"62482023", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70776162", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82392063", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"23076857", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32289612", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"39051614", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39442828", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"42016339", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51055605", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54385892", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57333795", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58987955", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"112164745", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"24133726", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"26803435", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"30787603", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39444990", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45039528", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58687199", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59997617", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63547459", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84110474", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26536341", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"36646827", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39752570", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"44797288", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49385417", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53322111", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55383072", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57129719", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66302278", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"3696218", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23728458", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25899773", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"39946481", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42021566", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60393397", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70286890", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"70883802", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"96054201", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"14477217", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30920108", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"41490898", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"48380638", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"75812676", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"38968266", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"51286274", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"98870906", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"99821023", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26428470", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30067765", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"40859268", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"49416788", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54864155", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65616047", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72127679", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"97265868", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25829399", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25976285", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"33038117", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"37444122", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"50688762", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"79089285", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"26496780", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69188975", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69481930", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"77500214", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88690521", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"26986505", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36237310", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39154683", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"46498986", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57462072", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"66172042", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79407614", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"135780788", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"159083924", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"51148404", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"64064103", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65620880", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66377574", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73428218", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"74698958", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"77931958", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"144391457", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"25459527", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31298052", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43578686", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45975465", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54129357", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54928213", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56235542", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"62045808", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"95370209", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21103242", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23862901", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36011564", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37787230", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"38222248", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43356831", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"93586685", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"144725637", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24157652", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24695621", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27092419", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42931783", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"47206295", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52125159", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53746622", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57507228", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62866293", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69302466", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71253495", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73300096", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"107319793", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35998826", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"36645683", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41163559", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44292530", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55665475", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"60733698", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62242387", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"91526781", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"7169790", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41505109", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45353633", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56478060", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76498309", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"98412518", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"10691059", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28335668", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32275215", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35887425", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37433807", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41492493", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42251379", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"85979264", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92045329", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24095171", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35286215", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55421129", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94148637", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95184701", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31842504", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"32287935", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"40169192", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44600015", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49596309", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"50247650", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51407747", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64132845", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69055937", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"18410414", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31106989", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31304772", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32110857", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40362025", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49343540", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49461731", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50775369", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"55834139", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"57305681", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"60024743", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64502877", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68273346", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88101979", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"103934945", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"118467854", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"136608871", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37667314", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39910780", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40478901", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"41548665", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44826016", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48725057", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"49392546", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53599460", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56798892", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"66383582", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"68095479", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27918866", "lang":"hu", "jira_id": "SIRT-3916"},
{"user_id":"28026181", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52989732", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56124237", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59358803", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70280350", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"72423408", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81295632", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88622410", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"88634213", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"14242417", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25809459", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"25850917", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31115561", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"36265855", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37661768", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39182464", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"48851006", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54318721", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55255371", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"57463198", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61314981", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64192133", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66947941", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"38466109", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39472532", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41684600", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42326301", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45179865", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51107582", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53174273", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65600945", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67851249", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70700103", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"9099674", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27654643", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43654371", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66499857", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68215966", "lang":"en-us", "jira_id": "SIRT-3916"},
{"user_id":"77493780", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39355141", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45926684", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"66337290", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67647385", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"74537772", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"76152094", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87029005", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94280016", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"136219683", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"46822852", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"57872083", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58788744", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60144810", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66235537", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68434176", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35738955", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"36433874", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38377014", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42535718", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"45099687", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"47123988", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"47651615", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"49478527", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65793834", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"66862805", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66972600", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"99348633", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"25860868", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27793749", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39716911", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44463791", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49840921", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62885470", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"63308245", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"145523457", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"25819996", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25878742", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29531744", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37509509", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"40463684", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"45485349", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49556456", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55899392", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60654186", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65260490", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"81010034", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89665293", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39188299", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"40657804", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44427033", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47770406", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"54300860", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"57583554", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24835361", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32583422", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"37077734", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"38212644", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38841327", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39496150", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"41298579", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"48004856", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54921232", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55337455", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56430775", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69778549", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77306309", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92431531", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21247607", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35597854", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69683213", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73995437", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"28043827", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48221594", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53499644", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55867212", "lang":"zh-cn", "jira_id": "SIRT-3916"},
{"user_id":"117128489", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"46384178", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47859966", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53959051", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57674634", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"59420086", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59943437", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"61407119", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77054929", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"86354399", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90597924", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"112672061", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"126598806", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21101858", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29542120", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37005002", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"39055364", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"41660307", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67572965", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"73999780", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74750292", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79304247", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97731911", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26031006", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36134089", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"48851855", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"64546672", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66228243", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67565242", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88112692", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"20791838", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53667351", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58966049", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"65723137", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"70588078", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"76220491", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25381226", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27528337", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"34729950", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44231076", "lang":"sk", "jira_id": "SIRT-3916"},
{"user_id":"58327786", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63256070", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"87150571", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"136263309", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23826833", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35806568", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"37454505", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43059237", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"88274324", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"35736749", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"76364087", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"31195996", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40631407", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40806299", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47249963", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60318837", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91047885", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"41494947", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"44370488", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"53655365", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63751583", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73349449", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76287107", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"77426237", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24501886", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40067818", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40379516", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"54461771", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55188203", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"60767753", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72874403", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"27237996", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"29453565", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37468009", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"37926835", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40263457", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"48856305", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53917642", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"62681815", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"105577672", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20423477", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29183721", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"29954707", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"32525635", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44575966", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48866885", "lang":"sk", "jira_id": "SIRT-3916"},
{"user_id":"52237936", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70843636", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"77466543", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"89743875", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"5959722", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"10050776", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41051737", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"41102797", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"50419845", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57056627", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83748520", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87934058", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"9486319", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"30173897", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38470744", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"49847641", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55061820", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60126217", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72529629", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76663761", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"27385604", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"36191135", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"42651314", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43848497", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47986892", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51399316", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"53252744", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"69739450", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79586411", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"141069173", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"9264713", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26054533", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26578241", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37664793", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"38406093", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39455713", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40441476", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40888118", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41479797", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43996656", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45202045", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"54151041", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55057450", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"61312869", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69700953", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"78549967", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83528030", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"135117584", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"32312957", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41819837", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48127347", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54443683", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"56912216", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62519496", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68817678", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"73938088", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79195174", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94730439", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"105725608", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"134223342", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24524459", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24655634", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37536399", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50258584", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62913586", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69889976", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72819175", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"79771422", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88825113", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"110967734", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"147162616", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31654628", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37243476", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"38271931", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45500254", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46785538", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54698907", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74785305", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25325349", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"37054660", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51033152", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54818023", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"71344810", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"23308993", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"36761702", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46325349", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"51525809", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"52395267", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"60999790", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81253051", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"39849666", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49228965", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76149825", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94788148", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"144465605", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"725171", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28148086", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29124020", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29803006", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30603223", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46011356", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49483146", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86735549", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26903221", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29882464", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37564424", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"41265658", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"48568028", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48856073", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"57042467", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71155905", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"71231165", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74352224", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76569551", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36140045", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"40378147", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42061392", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"55036439", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55757147", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66770589", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81026367", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24513755", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30239395", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40904302", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"45825334", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46056593", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53582969", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64581694", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69108896", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"76730673", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78105010", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"40494676", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42045347", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42217765", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"54828335", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"57223047", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62412877", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74330076", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"135058800", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24235206", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31512492", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37330433", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44955024", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45004438", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50320192", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64175030", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"69071230", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22919041", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54688364", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"65073247", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69624889", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"74951652", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39528268", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41519858", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45584280", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"56700112", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64415426", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74563356", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81121665", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30850161", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32361774", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46499359", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54057663", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55794782", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"121089866", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"24232964", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36083619", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38230903", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47732328", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54180216", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55503254", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76678954", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"86169202", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"98755727", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38723442", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72270559", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"91692453", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23034523", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30520215", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40025460", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44435035", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52818481", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52994315", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81149611", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"133674744", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"23930573", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41327842", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44980043", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66309073", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"67116772", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68301325", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95946223", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"27872983", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28523298", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50377041", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57187213", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62312036", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"76328107", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"108776045", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"144481063", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"21103807", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25249376", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"27507887", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"37922497", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44206284", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48861184", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61804464", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"75526332", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89904274", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"159097239", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"43358975", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51688395", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58485135", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"62347878", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"29998874", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35963985", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40870338", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42638275", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44335492", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44513483", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"49485062", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"52121860", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"62289400", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63514790", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63572351", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64438638", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80847340", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"25857062", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27180690", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31504433", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"66513919", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"68081359", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"91375820", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"102005142", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"32482546", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42966714", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69009031", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69658640", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81013683", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23778340", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28402090", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39709562", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48972338", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"53343859", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"54242076", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55502218", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66053224", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"75397465", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"17083830", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27657619", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28619197", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29073397", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29911652", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"54104711", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59339561", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29959985", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37126932", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37519254", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43533540", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44747983", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"56424023", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78516574", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"100825868", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"23749073", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29431572", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40343974", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42533974", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43806787", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"50990079", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54017321", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"66813080", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68151253", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81182255", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"23300728", "lang":"en-us", "jira_id": "SIRT-3916"},
{"user_id":"23301557", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23528098", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37320509", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"62874935", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84592065", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92506086", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"31775194", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38195474", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"49836658", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59168162", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"69583988", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"107876609", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"130208490", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"143918865", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"2305413", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26539540", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32110776", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36130862", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"45676362", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"55057565", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57105841", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58277865", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63780980", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66404880", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73779820", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"80608931", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"99307955", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"16588202", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43566092", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"44511311", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51096110", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"53083760", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61869390", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"62830843", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66086742", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"93640890", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37831502", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38365763", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41361321", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"41553979", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42879867", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"44812988", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48395246", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50258784", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51375176", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58617060", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"66374994", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70234681", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70736984", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84982039", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96050205", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97786339", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36124613", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45640286", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50919716", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"61733249", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67001764", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"77017232", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"91544173", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37475344", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42537396", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44861484", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71233782", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"88173484", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"91480340", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"146587624", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"26454341", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"29737074", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31085343", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49988292", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50883100", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51911971", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52845191", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55864392", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57659145", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59190104", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65666266", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"67091159", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84350830", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"23497107", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24004442", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26240084", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"29241738", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"37448037", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44436763", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50296579", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53771040", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55236635", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58733408", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"59704976", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"67025097", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68916664", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70878755", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83227509", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"111991165", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"142770778", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49361161", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"53315510", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55003143", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"60517101", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75184724", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"137110030", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"28993930", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40912963", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"43250211", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51399878", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65169141", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66376334", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"67039182", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78198584", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"147234368", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27832956", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"27980608", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35880294", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37192860", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"37238832", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51184399", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54927708", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66206330", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77575682", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"21549159", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25838082", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"32112134", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"32333157", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"36193108", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45751295", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"46070354", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78085141", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90715088", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22911188", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28673774", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"29491628", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"38668944", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"46552143", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65068498", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"1607353", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23473617", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36910860", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41816871", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"471093", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23401679", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27274265", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"28241419", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"39503845", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23796243", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44688582", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51512518", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"53879849", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56007331", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"66633129", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82689063", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"82912488", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24143984", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31927780", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"47009575", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51946887", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62286677", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"65976610", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69634325", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90454257", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25999472", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30739378", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50455441", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"51009880", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53979817", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64214409", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74350344", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"99928942", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"37606571", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"37607714", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37822204", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41896885", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45202194", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"46443829", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51307133", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85474500", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"134073920", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"18025465", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31263382", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"36657912", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38962435", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41879896", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44754989", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"55450325", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"71871890", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75546640", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"114590331", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"117407128", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28791302", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"46009933", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46577542", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49495142", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"64436778", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"111188265", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24593981", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29536239", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43200545", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50802087", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"77863737", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97502848", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"13220401", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29518174", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35750532", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"36851764", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41142284", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42250944", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42549987", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"43495735", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"45466696", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51702365", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52432691", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53576330", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55474396", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55998694", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96683577", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32039399", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36430298", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36973687", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41542056", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55540558", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65305038", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"80579467", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80928898", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90439249", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47076245", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48554102", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59947495", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"64227946", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65571105", "lang":"en-us", "jira_id": "SIRT-3916"},
{"user_id":"86988488", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87526882", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"27185492", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"31910139", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74088342", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"154997522", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"24963736", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"35049749", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45835319", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54087826", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58309086", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59688627", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69399440", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73522450", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"76709196", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"109231432", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32556711", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36447248", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42719239", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"43917030", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"44201586", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53384865", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54380912", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"112975219", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"147383847", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"12573349", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36707447", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38714081", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39073947", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39209213", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"44765413", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45872050", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48539733", "lang":"sk", "jira_id": "SIRT-3916"},
{"user_id":"75312192", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"152940072", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"6647968", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23449702", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27966705", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36381977", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40051024", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"41595139", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"44294222", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"53165046", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"32453820", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38239635", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39972147", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41257196", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41886194", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"42509165", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64554194", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65182274", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72784808", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"6854150", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"25502347", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"33031412", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37123124", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"41604928", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46439045", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67093099", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28982957", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37718802", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38754034", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61710834", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66479381", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"66592875", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"73486167", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"75484969", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77103262", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87118515", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26846681", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40304486", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55263674", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59417760", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75046309", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"115755767", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"183446029", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"30344540", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"31583108", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38428364", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44942700", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49202991", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51761352", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22396180", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29884841", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40843174", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49092599", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"55285386", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59489555", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25483691", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28739486", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29703859", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"30321292", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32519219", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41691741", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46259290", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47245022", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63510625", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"63525200", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69323971", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78504341", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"104247710", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"32346160", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"38425403", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40967766", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44074671", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48759766", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61775790", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"70852940", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"74000988", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"152841312", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"29477144", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38368629", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41515470", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42005868", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43406121", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"56206164", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59390180", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63588674", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64252575", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65200346", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"69087729", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"157211904", "lang":"ja", "jira_id": "SIRT-3916"},
{"user_id":"25802451", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30821801", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31992399", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"36451482", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61636595", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63664848", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67267848", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78415588", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83552872", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83649661", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"22085411", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30638200", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40462231", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"42393924", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43106497", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"45709299", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48801825", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"49516448", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"60402106", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67388477", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71842138", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101549681", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22187853", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31909372", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"47055070", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56634410", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58651825", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90337900", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"102198181", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"134536904", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"140998710", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"17370082", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"27899414", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"32043594", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36102764", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45431764", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"47342577", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56769181", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58135665", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"75406224", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37538105", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41722861", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43379165", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47077608", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"48867322", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50614184", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52655574", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"53969911", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59507623", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61358071", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61957488", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74987743", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85297581", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"89725882", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90551029", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95544779", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"137266453", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31246087", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41783645", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45260144", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77629721", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90344620", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"132638502", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"133774413", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35833183", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39745544", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"40218444", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"40382257", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"54776655", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56721306", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59174931", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"60633133", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74691249", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"90826679", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"15700014", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27076375", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47321115", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58193478", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68561049", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76930398", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"82345608", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"112632656", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"50663955", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64747974", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74429555", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"172761684", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28873501", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32362931", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35634043", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39680019", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"40021554", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40119394", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44326849", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"49210140", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52310740", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"58183945", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65470279", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66799454", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71973641", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73471277", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"117417849", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"3240233", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"19107062", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24086536", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27885601", "lang":"hu", "jira_id": "SIRT-3916"},
{"user_id":"28500519", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29375171", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41930562", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49508682", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55187914", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"89712433", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101266520", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"110848106", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"114092735", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23230541", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25927439", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"32740854", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40087727", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42562873", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"53288120", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55221603", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"100833077", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"144545563", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"42786282", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48726888", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56254982", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58037606", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"63203933", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64584597", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71937158", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97666279", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26434130", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30901424", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"37524615", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40475156", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41824827", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47207941", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"60956796", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68485473", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"138206941", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"31247863", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64924846", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67416642", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70545606", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73855044", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"77738787", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"99409633", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"105079037", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"6768122", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21269193", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32724940", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44838318", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"59837384", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71634664", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"13929939", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23557354", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"27351403", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"32272027", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"49210344", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"51073889", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59454768", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68358051", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78635237", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"98902092", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"25320495", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28635446", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30682998", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"32273354", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41932620", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"44524263", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47334663", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51146258", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53020160", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55582979", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55822931", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63198649", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74520334", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"11347785", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38759479", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45826087", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48296388", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"50796290", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50814812", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80028297", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"142447245", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"142762743", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"19660783", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25584648", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29697740", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42827756", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43621314", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"46773113", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66196625", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29616594", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31023038", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"40438183", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40554859", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48821581", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49324617", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51815229", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67749058", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69657814", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"70576818", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"147342125", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"41474758", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"42374001", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51223794", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"52559664", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88109263", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"88451829", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38748783", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45019345", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"47814276", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"59608555", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"18796879", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"21584260", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23611364", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39308348", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41204167", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"50908128", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57467729", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57850214", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62682388", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80902276", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"83556655", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27910349", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30479752", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32975263", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37741792", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40922232", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43806296", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"44807514", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52263970", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54977167", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57670494", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72139076", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36824536", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38220807", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45486624", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"60679399", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62343088", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"6997066", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25666760", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"26571766", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26589036", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44453483", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51542518", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64874824", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67865985", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70972928", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88679980", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37509829", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"41980881", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61261334", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62667864", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83796463", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"39642857", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45514070", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50472812", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51433861", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"55538528", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56667723", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"57169905", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68257485", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21534558", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31674299", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31729606", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"31934568", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37919327", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39274492", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"43513896", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"43647829", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52311633", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59175023", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"86549211", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93672312", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94703879", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"37690585", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38162290", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42070874", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"51266046", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56117960", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58109027", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63023811", "lang":"lt", "jira_id": "SIRT-3916"},
{"user_id":"76476853", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26827484", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"29819551", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36640536", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45676904", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"48121150", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"59483406", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"61376455", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64558026", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65771420", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84042516", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"131890780", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20796680", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40484103", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49061410", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55150222", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56576882", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57146845", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78340100", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"113967528", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"29187067", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32161486", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"77755107", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"78915503", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"10332629", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"19816861", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"28817745", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36417289", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46816538", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47981065", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26174064", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29512196", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"40186921", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42601631", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"65539427", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"100866553", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"102991327", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"184795850", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"41502633", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"47496894", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48492061", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50799622", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"52211645", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"9909498", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31574598", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54155443", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61764632", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"98395120", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"154276582", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25826416", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"46148032", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"52150784", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66713288", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"77898784", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80932516", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26851810", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29361162", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29780864", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"32665567", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"38572199", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88867884", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"134296514", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"4536816", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"13615847", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"36223528", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40672083", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43062569", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49881825", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"56961426", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57271987", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59840387", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"127498702", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"4494489", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"35610870", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"47872104", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48827570", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61781541", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"72080135", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26826183", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"31473473", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37445838", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40317533", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"50094733", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50899491", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"55908857", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58740402", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63086057", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76365302", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"33159949", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37327132", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39783696", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44175438", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46939291", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"55287869", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63570119", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"72151099", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"76315737", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"135774894", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"12745168", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29059657", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31008790", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58247673", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60144436", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60525975", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"71387363", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77868175", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"27003963", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29471752", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"43776727", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"53502916", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"85866235", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"87414084", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"127097268", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"31085987", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36380859", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42544319", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44198879", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50657458", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52001617", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58530746", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74501680", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39440266", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43352121", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"60768881", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67005700", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72046364", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"84299820", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"141650350", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"24180991", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46524143", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50190820", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51010013", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62442293", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64697558", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73519249", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"90166261", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23442333", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32903647", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39132290", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41800281", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43559082", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46579007", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47405780", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50166711", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"52757812", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57151952", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59764514", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61250956", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64766376", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79956191", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80356404", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42668239", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"44373017", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55174145", "lang":"en-us", "jira_id": "SIRT-3916"},
{"user_id":"55972952", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77508045", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23092786", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30973317", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44746276", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53390583", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"103714765", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"118778406", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28860530", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"35993877", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52351531", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"55721137", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86030486", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"16433726", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28102827", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46879900", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51740351", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59947427", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"79875971", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"106224157", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31880764", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42970925", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"45776860", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49371543", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49404082", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53009913", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68604443", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"70778702", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27693600", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28606100", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"38070857", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40123021", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41990695", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57540683", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"68202275", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"111412572", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"27012798", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"38317863", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39589069", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51713235", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51832704", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51951097", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53512142", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55835390", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"60138280", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81097939", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"168176509", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24428374", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"44682221", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44855831", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49643543", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"50007656", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55650328", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68408727", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37480143", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37987507", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41459182", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"44360191", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62503392", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90273023", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"143656474", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"45427009", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"67524842", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"106707580", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"485641", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24591573", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35708945", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40602118", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"41214637", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"61094994", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70123626", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39399353", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"40729937", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45951698", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51908696", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"54951185", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64264214", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73579405", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"99151559", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26837608", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48850939", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50218051", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61352732", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"100012847", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"121064499", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"126878340", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27120083", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"31136052", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44603135", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"47890041", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"51983815", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"126758093", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24821071", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26968368", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"29267852", "lang":"ko", "jira_id": "SIRT-3916"},
{"user_id":"36102990", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36955647", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37857695", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"68377419", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"89869936", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92093412", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23097131", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26619144", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26919683", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"35957384", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37282377", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44753869", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49393454", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49742007", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57401276", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59309099", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66987287", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68965270", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"71140580", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"71503960", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75040644", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24193502", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28626246", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37626200", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54266751", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55295040", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64253421", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69841709", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"113704047", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"144732934", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26056552", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31247176", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31734489", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"31821292", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32372814", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35049423", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"41222040", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51012760", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51674118", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"56684525", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70145710", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"83474371", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"102928218", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"123031290", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"137597238", "lang":"ja", "jira_id": "SIRT-3916"},
{"user_id":"160736025", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"3398302", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29837150", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39564687", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40836405", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"60952050", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64836843", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65520766", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"137399053", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30523901", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36029930", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"36211284", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41939266", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45164012", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55547782", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"66792061", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36820651", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57636027", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75170164", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93156394", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32560951", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44817616", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51060554", "lang":"sk", "jira_id": "SIRT-3916"},
{"user_id":"60131937", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"121590212", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26942614", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45777850", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59575898", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59972392", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"78621918", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89418811", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"110871184", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21201618", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40089372", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45414602", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50841744", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"56095268", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"58507438", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58528644", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72603468", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72982525", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73523929", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76154817", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85141020", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96151353", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"133435710", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28419136", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"30200196", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"37926407", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"39278161", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"41187110", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"47955870", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"48783313", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"50670111", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54720919", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58535654", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61730971", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65899619", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"68256282", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69140776", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27592406", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44688885", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45206048", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53863987", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"63398390", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"65402287", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80759622", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"85972093", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"121202089", "lang":"ar", "jira_id": "SIRT-3916"},
{"user_id":"142855666", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"27994248", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30010993", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38193658", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43856691", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49307552", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"67778011", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"72664858", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76608468", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95154864", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26819117", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"27342527", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30526712", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"37275133", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37365630", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38865192", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40584032", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41748291", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43627279", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63246261", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"90330132", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25476448", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50838240", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55655303", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39940762", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43813265", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"44494489", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52124293", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52297495", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"54295804", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59026533", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74795198", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75298490", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"81124528", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84607062", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"23179441", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31394771", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37930199", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42228578", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56895730", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59494134", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62596480", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64724656", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65651385", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74226011", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89914092", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"95851753", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"108096526", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22320542", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22956512", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"38235467", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44297377", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"45581488", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48274352", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50821515", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54415346", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54942545", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54946802", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58877892", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"62515080", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"69012754", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81737492", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95265302", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"37809425", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"45180385", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46264561", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50166863", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60655578", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"64374989", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"146399952", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"15049052", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25216312", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27361204", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35619442", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36587837", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39566505", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42602368", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"45037675", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52003410", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52839044", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69342089", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"77121616", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101517676", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"10514679", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41985410", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"46604229", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"51399606", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59992631", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64873515", "lang":"en-us", "jira_id": "SIRT-3916"},
{"user_id":"64968967", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70092233", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88477671", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"31995671", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39450700", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40783441", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48024173", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49554545", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"69043746", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94886796", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"134866023", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27979956", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"37973887", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"38067972", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"44151352", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59712498", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85652756", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"24754173", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"31500167", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37210432", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48939165", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"50035140", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"55366087", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63734884", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68512995", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90526905", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32056041", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35682848", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36034996", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48553474", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53293834", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54361543", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55584299", "lang":"sk", "jira_id": "SIRT-3916"},
{"user_id":"63229523", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64178774", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"112908378", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"8366610", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30063312", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35801016", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38003813", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"45785377", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47237770", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"51426512", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51947480", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53336734", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61272343", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64089744", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65681522", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67065701", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68443380", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74257368", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79932836", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86252879", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"87359660", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92010789", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"101088792", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"118021832", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"155430963", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38946095", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45243849", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"48760586", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51315945", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57611720", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60136046", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"66259878", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"74517082", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37579111", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38988763", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"59159221", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71852694", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78756636", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"102070446", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"120325135", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"132238549", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"51253519", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54556076", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67721093", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"82436603", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"97514004", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"21803457", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31829101", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"36500242", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"40516182", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42691253", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44211090", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"53003967", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54578868", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"60209824", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60920516", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62546475", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27926382", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"36205371", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"38958327", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45178674", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51234669", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54765343", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66042818", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71864070", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"73403372", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"152946809", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38956454", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52335949", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73773121", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"155630606", "lang":"th", "jira_id": "SIRT-3916"},
{"user_id":"22003760", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35840303", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36357155", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37025236", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"45405087", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69543635", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"77629839", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30188493", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41447088", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"46244087", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50036522", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56805343", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61462209", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32701922", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"43334795", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78044889", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92461912", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"108151038", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"9819226", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41593624", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42030709", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"49191539", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49984795", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59530540", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62334719", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63757257", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92504660", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"12944658", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"17374834", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"31783824", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37427615", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"43969092", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47641844", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"50209055", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"52400292", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72543561", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73053209", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86439663", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45233146", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"57526540", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"58344038", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60837838", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"69654411", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82850971", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"147853985", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"33149972", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45240843", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"48314655", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79865177", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93568294", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"118869352", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"29233322", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41730712", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"46863449", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"53320200", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"61023792", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71304718", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"4034015", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32353179", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45391771", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"73212891", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"88436114", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23191903", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36161862", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38592085", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44898099", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45459124", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48473981", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"56841011", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"100964328", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"454197", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37135452", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51164106", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"136370021", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32166409", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41484951", "lang":"hu", "jira_id": "SIRT-3916"},
{"user_id":"41766330", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50599610", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54640179", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"55978127", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56682139", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40019004", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45982340", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52593549", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54601197", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55350603", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"55387513", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58850839", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59879910", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"60219854", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65791746", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69759951", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"30364645", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32490234", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"33341600", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36849645", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"36866127", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37605987", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41036089", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43025716", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46289050", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"50102431", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54381683", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64371295", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65806812", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81393528", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87442177", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38790512", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44483585", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"51353391", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54788532", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61633105", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68628993", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"74993601", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"86373873", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"106751780", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"111181680", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"18234235", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25583652", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30080721", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48210323", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52028167", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"52191690", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86260796", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20851137", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36173090", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"40628063", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42141352", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45131316", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21500742", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35827596", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41113529", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"48937475", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51848080", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55463706", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58132258", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"60189291", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"61251060", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"176044469", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"27836784", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"31117172", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36348350", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42398052", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46151157", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47304688", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53253302", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61191882", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66416759", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70331013", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"13749718", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25578276", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26675599", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39972674", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49160492", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73593227", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"87830389", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"155848789", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26762060", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37164663", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"40287470", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41189912", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48720755", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54151664", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54608708", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60712137", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64758982", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70654570", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21783277", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"21814559", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60872898", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62416715", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75236000", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"101050242", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"120572773", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"135093685", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27378036", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44645402", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"55400895", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58442204", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61106431", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68255294", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35804576", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"38082652", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"41934181", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"51402674", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61004899", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63880165", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65567018", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66010281", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"71539373", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72997950", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87793408", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"115015025", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"307568", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"13818380", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"18466131", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"29033289", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37085517", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"44307706", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"45369540", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52281921", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59042144", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"60709992", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"63392092", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"89735593", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"96119853", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"159138789", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31621745", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50262943", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60284549", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61820132", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"77583586", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"87360412", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"95742141", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"29054712", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30612194", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37577661", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37596086", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42320153", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64646586", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41839788", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47146969", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55221727", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67800414", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70892739", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82429217", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"23417509", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32438898", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36840357", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"43407123", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47702656", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61626579", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69026664", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86710944", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"109555035", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"120795626", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22016393", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29369118", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30039402", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30396958", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"47496004", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84346727", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"99263105", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"122002874", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"128947907", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"16348249", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"17125458", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"20413219", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20945695", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26282765", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29203666", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"38706372", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"40252840", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41937464", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"49140694", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"59704647", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60300307", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61788646", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62026847", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23294488", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36675951", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37035044", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37741562", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41942805", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42183545", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"44133555", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44368156", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44010645", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"70622177", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"86861565", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"23581546", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"32508229", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"33022386", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44980407", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47654655", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48468306", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49655787", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50285695", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51051347", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60535280", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"61230210", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"65405303", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69041930", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"82384968", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"8335936", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23572698", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25077713", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36132220", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"36211495", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43803688", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"53507042", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53915011", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68774235", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"69280046", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90906051", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"3240079", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"18606153", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"19009254", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21124554", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37195924", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37209871", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48116252", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"50676184", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"53401180", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"59234593", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59410100", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43632015", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"49506535", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62207699", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"70558611", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73971228", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"90426592", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"100811319", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"130229714", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"45674785", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"59063629", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"65495224", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"135195878", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25987904", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29764517", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36822978", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"43359171", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"45248193", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45735398", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"46051805", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46825956", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56466617", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76058889", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"107078722", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"129636522", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26681400", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"35099669", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45800026", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"51925223", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73896101", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38226886", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"41563103", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41953030", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"50500408", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58975620", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61681755", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71449108", "lang":"zh-cn", "jira_id": "SIRT-3916"},
{"user_id":"113142175", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41487881", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47614509", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61050761", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"62889409", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66669208", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72880639", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"76058056", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82889687", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"88510702", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"91511459", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91965147", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95819104", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101465198", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24308506", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37832098", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51401800", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"55822652", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61776603", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62513392", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"68185031", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"75804982", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"103890299", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21623190", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"42280615", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"45087819", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46684422", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86247085", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"111631012", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32091243", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38818196", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46152483", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50146516", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53445384", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"58213463", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70977032", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"108824906", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41381217", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"45602343", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46929610", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70184851", "lang":"sk", "jira_id": "SIRT-3916"},
{"user_id":"71613832", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75424262", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75834833", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77171157", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83339381", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"142700756", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"30281401", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32532341", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"36707837", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"38879412", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39758744", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43348153", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44735372", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45261101", "lang":"ar", "jira_id": "SIRT-3916"},
{"user_id":"46388136", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47293136", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53134703", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58706053", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59086726", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59403745", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61498956", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61864798", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66887631", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84076430", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32861505", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45046588", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63594768", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"96931779", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42635977", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43309578", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45527978", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"58162240", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68587233", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68734107", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"77367471", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"9339262", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36394128", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"42312620", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58024924", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60140314", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67041222", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"68418076", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79425530", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"94378825", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32050282", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50234817", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"60550977", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61408578", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"67029440", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71023342", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"154239486", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"51886387", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58006033", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"59355483", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64308102", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66303743", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"67142020", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70352741", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"77744430", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84954568", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"122719200", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"19228601", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"29902952", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37506045", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"46666150", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47077148", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"48006719", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"53952249", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"56096212", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60173556", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66076077", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70144308", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74512276", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"11576170", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28223540", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41642877", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"41922551", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42024925", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"47458016", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"59088654", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59717897", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63615014", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75892401", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29650786", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32733630", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"35652761", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37323977", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68401647", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68617927", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"85219912", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93710929", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"158697854", "lang":"th", "jira_id": "SIRT-3916"},
{"user_id":"30283364", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43799003", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51780806", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52801846", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55127490", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"57578824", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59127180", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"73574473", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31466782", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39117777", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41995885", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"43145847", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"49583830", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"55959806", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"67041613", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"83084666", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"102563658", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"23322495", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31258826", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43827140", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48714267", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54319572", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"55229956", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"55312815", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56742489", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71093102", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74768624", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"78142582", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87294514", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28479026", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41283727", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42198787", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52103758", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"53129947", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"62866279", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"64066439", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70657614", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"27581022", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"38441229", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49661539", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58025337", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67469380", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"74191253", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79696471", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83082356", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28096161", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"41766505", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"51384061", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51900117", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54285371", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67873698", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"89871593", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"30840007", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36590750", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37580187", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37677203", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40671028", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"51663722", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52016489", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55838303", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56636165", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62545143", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63284288", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78409072", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20795306", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24641762", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37538929", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46241427", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54329216", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56999617", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"109550604", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31803718", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35598463", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37090345", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45683114", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46024385", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"53414640", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"109485957", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30162665", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36954713", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38571396", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44508021", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50028046", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51304277", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"51378407", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54758622", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55151997", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62496764", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42259843", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42981927", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43131200", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46257157", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53759197", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54327149", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54336195", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68662386", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72871495", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"100087199", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"117750704", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"27695718", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49973142", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"59882647", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25938965", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50699078", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59103662", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61760298", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65298469", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"1938362", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26889932", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43601326", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48030671", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51574049", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59922249", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71079410", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"81722035", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88706873", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93128060", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101814238", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"23714480", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"32342455", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37498439", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"37661848", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"38719660", "lang":"hu", "jira_id": "SIRT-3916"},
{"user_id":"39966405", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"51146677", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51250699", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53581931", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54933455", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"58303917", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"72990046", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"74355208", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89486159", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97931305", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32073463", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36831324", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42460345", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50178979", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"54271754", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"61728515", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"124083825", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25591295", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30463956", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"40904706", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43453901", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55369022", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"62924033", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68782664", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79219069", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"81096138", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86826465", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"91362127", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"96083553", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96700164", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"6977216", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27438202", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28107347", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32260933", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"45806136", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"46700142", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51320961", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53548495", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56283581", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68996019", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78250400", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43215841", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"50748311", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"51488437", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71074883", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"71342915", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71533996", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"77625588", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"18467226", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26554872", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32236614", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"44124252", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"56144975", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61030947", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68512994", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"17429923", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36553109", "lang":"th", "jira_id": "SIRT-3916"},
{"user_id":"44972869", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45337201", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"51434630", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"52922984", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"28117850", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"35688981", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39548526", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45958689", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"47159536", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76591415", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"91798519", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"105114658", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28522107", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"30579171", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"42090041", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70400802", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79835031", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"152305979", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"37728872", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"50719360", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57919592", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72016043", "lang":"ja", "jira_id": "SIRT-3916"},
{"user_id":"100411507", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27898419", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31112124", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42734945", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45124913", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"48730600", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91036306", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"20540513", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27656546", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"34805856", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48238915", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51783838", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72633345", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38936788", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42775389", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"43059433", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43450892", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51076879", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"53361362", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61696831", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74942827", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75781107", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88379640", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91130904", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"26939668", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28753846", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"28868282", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"45064182", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47206751", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76986973", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"109686977", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"176877962", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"22819777", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36434413", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"49292937", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"60408566", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69108862", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72320198", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78507704", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83880307", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89445076", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"12873619", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37671732", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38547189", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50772177", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52228810", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58083727", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80127811", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"31991036", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"53952413", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54303147", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61381981", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65171537", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65637292", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"81685605", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"128962320", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"29237540", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"29445982", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39692671", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43418475", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44663996", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46625636", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58542511", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71004739", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72534408", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24476057", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25360437", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"29211549", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"35646963", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"36942442", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39155268", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"41159471", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"41624450", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42548833", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56561203", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"62464070", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66481488", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83343434", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"84838035", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87883697", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"105482628", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"23560261", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26925157", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"53025908", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54103491", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60873723", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67856319", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94418871", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"137096785", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"27207131", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"31994078", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37096309", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37350296", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38928832", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44567867", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46057229", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46989510", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48898136", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57367523", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"69007801", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73911873", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75977952", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"142772153", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21716271", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"32287251", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"36202802", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41777751", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48863991", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"50840776", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51655116", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57120936", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78118647", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"104366211", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"35934821", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36110744", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37098236", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37971570", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"40906667", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"43234826", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"47372582", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47833455", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51417153", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52503724", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"60541030", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73549653", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"76312645", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"10915868", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"16193231", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"25923135", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35780414", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41190085", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"43301503", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47524979", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49311686", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55746919", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56432938", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67131320", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31932927", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38723897", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"39586592", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"40401463", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45445231", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"46036156", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"50593777", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"54019229", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55947845", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57472193", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63185173", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"76327657", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82203707", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86015901", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"104241926", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"27489898", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"32127417", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37395761", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46694267", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49034255", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50237666", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"68286375", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"123488741", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"133452506", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"16981797", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25460436", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31087835", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38734533", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41936732", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42370535", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42536152", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"44510031", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53761478", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57814775", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63113579", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94962800", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28283689", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"37158429", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38854490", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"45986150", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"63424421", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74460694", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"122646575", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"125004791", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44137966", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56154534", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"63165250", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81585195", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"91752888", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"98786698", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"20840846", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"22498837", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29213078", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"35762823", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39268046", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"40776268", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42868307", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43836738", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"44543559", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56272330", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"66289991", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87460217", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"108260569", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"143762761", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"11735347", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"17014870", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35615219", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"48904930", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28541603", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41285625", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54637992", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68594195", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"74508710", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"116192137", "lang":"th", "jira_id": "SIRT-3916"},
{"user_id":"31233724", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41814941", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45212505", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"53267846", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54023921", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61242516", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63111176", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80492412", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"84258283", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"16644415", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32093357", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36016025", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60410793", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"283763", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32690772", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38638253", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49546367", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59593768", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79138719", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86111861", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"90256250", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"157716810", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"176473060", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23913977", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26074195", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26771032", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32217542", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"35789007", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39026833", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"42545773", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47319912", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48937001", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60226267", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"114175863", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30088155", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54084754", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"58722720", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"136148011", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23893357", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"25002608", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39954081", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40864272", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57037588", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35057163", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36789396", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37663927", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42898312", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45142205", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"55964514", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"61918094", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95964485", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"18188506", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"18544732", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37732098", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"38588747", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"49591115", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"62904288", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71819756", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"79038300", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81126498", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"7197144", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31169658", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39022737", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41909995", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44250471", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49878128", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58180414", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"61152678", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94371731", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"25080624", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26372074", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"28097041", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"31132259", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36287632", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42356780", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45548538", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69048543", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"104464112", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23960545", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25006634", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30183827", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"32521009", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38295087", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43251276", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"46877260", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50932200", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56779500", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58633396", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64178328", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67522678", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82019223", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84047350", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"132896411", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27921078", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"32759634", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37210826", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"39208324", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"41458604", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61776577", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"64697122", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80268676", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86815632", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91437825", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"96103196", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"195133741", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"3588334", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24071960", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28412126", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29402238", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36894638", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"42745122", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44500144", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48440215", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"48616283", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"50820023", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52309324", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53706383", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58231958", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63271265", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64065482", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97235898", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"114172852", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"137875776", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"28234641", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"35052328", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39188119", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"44425854", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59129594", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59219356", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"15707994", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22444511", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22480205", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23735682", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24464623", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29093944", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40616198", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59520572", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61984624", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91637267", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"2827945", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24760832", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36352534", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48940568", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53307409", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58733825", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"65196644", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"66208980", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72229941", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91027144", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38079222", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"50628246", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54415030", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54579077", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"63507156", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68928228", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"69605130", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73797859", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"108173150", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32502455", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39634067", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54280775", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61757029", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64641229", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22256103", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"22415931", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26316510", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35634025", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39418940", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39468745", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40109653", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44190829", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45746379", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58315959", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59583961", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"61912072", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79287710", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97575706", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"106873360", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21789463", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26745031", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32042515", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"37332985", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"37606119", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37796272", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"41215465", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41728676", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"43622286", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45711862", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"50948413", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51197895", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73510418", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75667928", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"97144814", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"2076891", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38472682", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40273782", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40300105", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51292678", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"51856659", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59694298", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"62934983", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"70065443", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71905874", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"81038409", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43990281", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"52955561", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62507958", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85770229", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25635078", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43923293", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44242128", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51802012", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58335392", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"98470286", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"109638085", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"146463688", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"30557587", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39096342", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42468627", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"43634700", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47807150", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"50999563", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56342831", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69217400", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71745918", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"85004221", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22630232", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23474080", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"36023148", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36546671", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36688604", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42951365", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44467244", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54699677", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63067638", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73221798", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"87675832", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"5926821", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"6850887", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"19677225", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44771372", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50094946", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51194216", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56274899", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60739388", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"72083459", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"94441343", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"114913908", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"119786282", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36665973", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"38682663", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46478149", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51918332", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"52300826", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55430410", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"59667176", "lang":"hu", "jira_id": "SIRT-3916"},
{"user_id":"68253399", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"79383905", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87079948", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"111967387", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"23794404", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26082801", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32098041", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39603922", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46858838", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48207392", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"50753000", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51225487", "lang":"en-us", "jira_id": "SIRT-3916"},
{"user_id":"51843900", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57324498", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60324733", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"60709671", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68228306", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71794934", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"98074394", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"36804370", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"36909337", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49750290", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52269345", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63130959", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77502815", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"23899084", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30521679", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38120281", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"40104385", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49709505", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55853389", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"58129013", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60654445", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73320947", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27360547", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27891341", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28735635", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28867165", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"29884401", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32153412", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45362055", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"48928134", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62002678", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"65353025", "lang":"en-us", "jira_id": "SIRT-3916"},
{"user_id":"67780347", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73209474", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81399576", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"91441060", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"43188963", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47185854", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50201648", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55299511", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56737122", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29942006", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30420666", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43653872", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48244291", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48484096", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"51651865", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54476286", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55528544", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57353292", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"57685743", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59038064", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"60060244", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30831829", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31724983", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37133357", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"38083587", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54327285", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55678448", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"71729858", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72573458", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"77174366", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86438601", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"13407990", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"29016003", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43365025", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46578569", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53904349", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54212840", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55037773", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56017798", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"58431389", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"65782073", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67607127", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"72272220", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81731889", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"143579835", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38546607", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"65657726", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"98805394", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35090040", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41347657", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"44272705", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"46003747", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47522230", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49848409", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58633689", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62164858", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72461111", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"131674659", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"135821222", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"27977814", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37102924", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40458237", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"42256911", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56117703", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67762897", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"107868379", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32702143", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"36155872", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37588109", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40435880", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"41188908", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42914354", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44072128", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49516827", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53229363", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54248162", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"66250771", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66544276", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85389861", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"11430318", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29119573", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38728637", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38899187", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"41861292", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43649256", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"47169578", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51955619", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56751481", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59909786", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62044225", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"70825522", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"76751307", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38873099", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"57803298", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"97476059", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32580722", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38596782", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40246756", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43079650", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44351136", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54382236", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"65985007", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85698227", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25816620", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36250558", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41486925", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41638859", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41798767", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43324188", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"45071652", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49828221", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52957269", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59974529", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68137724", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"92658762", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24080473", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28663522", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46558282", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50186384", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66100275", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83694380", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"144118770", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27618528", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44392410", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45742399", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47663137", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"49353567", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55843713", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"61427321", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62556381", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25465433", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26647063", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31876292", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35823095", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37107631", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38747827", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"43623165", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64066103", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"65977997", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82021706", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"24773437", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29537958", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"32068170", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38742388", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41913802", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42043889", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42667195", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45966693", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47980325", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50255073", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52541394", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53844420", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61749386", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82177068", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"19319941", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31272401", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36824306", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38848737", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"39001147", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42138576", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50432390", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"64947649", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73489710", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"76290743", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"30895258", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"37371227", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39507280", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"39905937", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48338545", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48757823", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"55488089", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56736705", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58990480", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"62765192", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71040143", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"83065803", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92572407", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96376266", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"13243026", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38670642", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38974746", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42183249", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50284469", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50948140", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57854712", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63927015", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101743927", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"145381202", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23420714", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32461218", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32785080", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45209763", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46097352", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60574089", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"116392271", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53187460", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60238100", "lang":"ja", "jira_id": "SIRT-3916"},
{"user_id":"62586726", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67713261", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"25641511", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28441182", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37606572", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"47184175", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73051996", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77720155", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78607398", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30780206", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"41990710", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"42474498", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50296800", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55489135", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62614076", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"62960056", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78139480", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30438811", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37793057", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"47928455", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"62394390", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64380438", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69449564", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69947923", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72134506", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72652924", "lang":"ko", "jira_id": "SIRT-3916"},
{"user_id":"131727048", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26809386", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"45308382", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49500393", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54526625", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55379982", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"129370514", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28359882", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30919121", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40058326", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"41598188", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42539829", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44012818", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44306027", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"44964526", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50923719", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62218081", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"65040383", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"78924055", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82833644", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94824721", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"156904085", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42855403", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"58861013", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"63897283", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25600676", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35780889", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46477601", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75535837", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84912208", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87468337", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91925083", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"135997869", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20875615", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25023786", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"29894820", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"40231542", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40664207", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"42434431", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43022623", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"44603297", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44658886", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46193622", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"51552807", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"51615228", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57132665", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59122214", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61401002", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64479653", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65897630", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"123414650", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"131048677", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"153560594", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26335122", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29929291", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30122164", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32902347", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"44718310", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62842049", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"67593519", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"8499482", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24534372", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"32332115", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39694235", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51550256", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54098024", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59480924", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23326941", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42328226", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45155711", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"48420936", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56101282", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61591577", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63390171", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101008184", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"2893385", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26088955", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36638093", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45234535", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54009984", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54678369", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"58912568", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"68167273", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"114253797", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"82396", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22820062", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29107421", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32148835", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"53510574", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"53563747", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60923288", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"13013898", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"21537752", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27066177", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40399887", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40583591", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43315854", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43791203", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43835744", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44836914", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"55346878", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30430711", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39712570", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40910977", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74226765", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74497222", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"155364711", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"27586811", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40233330", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"61777997", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63040503", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73938266", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"109323195", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"115317933", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"19954649", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29477521", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30009873", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39370113", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56297029", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73847627", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83718354", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93876011", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"132925761", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"137698784", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24759355", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45713218", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52863776", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54836696", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55402663", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64404902", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"126120812", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"144995974", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"10735231", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42213939", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42678248", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44762533", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45486289", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49675934", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54389426", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54610118", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56346478", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76129019", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36340781", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36587781", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40430450", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62092434", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72011028", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72493006", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"78874127", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81830120", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91563848", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"116740427", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"138328665", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29286444", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40289826", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"49199454", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60534957", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"67899428", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78472770", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"102000465", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"123112611", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42828050", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47228965", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72628203", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76063137", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"39900014", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"49258949", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49802758", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49913703", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"64762273", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76684911", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"131792941", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31749928", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"38006673", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42655135", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"48290013", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51431061", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51886646", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"51923808", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53837541", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55018409", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55986982", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63073932", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"133149931", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38132351", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"46662050", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60186688", "lang":"hu", "jira_id": "SIRT-3916"},
{"user_id":"73174773", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"82996192", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"128642623", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20828249", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"29838503", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37815253", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40825043", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44565641", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50821623", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52860340", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60054180", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68330697", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"75697519", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"80630204", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"101993596", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41458575", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43089242", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44261032", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47940430", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53865334", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86818142", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"36630613", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43856645", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46570136", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49687354", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55816664", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61489092", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"61594876", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62790742", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68796974", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80101603", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32079553", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43324112", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65906207", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72419462", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77624873", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88285546", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92678506", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96787865", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"135430285", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"161850560", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"18333674", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"27961477", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42081637", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43643213", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"58421382", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60033884", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"74616583", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"75094841", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"82401644", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"28993674", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"56444881", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57150693", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75126150", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86786859", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"92224744", "lang":"ko", "jira_id": "SIRT-3916"},
{"user_id":"23529019", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25711230", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31591270", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42015388", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42263011", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50112526", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54604108", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72946593", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78418690", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"89124305", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89330624", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"132560253", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"156296546", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"23477172", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31808506", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"35690547", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"39379519", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40504144", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40620571", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40975615", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43625442", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"50422988", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50876557", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53295164", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"53371528", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"58925810", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"68125596", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97329139", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"132569836", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"17066015", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29142076", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37258892", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40437342", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"58265451", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64233585", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"65667653", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66111004", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90267845", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"132093630", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"140904524", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"23733917", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30912945", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32123057", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37382118", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38129682", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38449135", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49860617", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51876632", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"57669494", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59688222", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75735706", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"133245555", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41976896", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"48492093", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53741135", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66039465", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"100842126", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"25398416", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"43260871", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45168095", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46464339", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48548958", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49955162", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"54162691", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81745658", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93812008", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45885999", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56247971", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"61156964", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62180335", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68742220", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"71637780", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"12105232", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41498387", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"41787169", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"44163738", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52124328", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60620969", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"91020776", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"97389774", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"115405345", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"30187973", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32306229", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"37069856", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42081570", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45693000", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47029748", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"55165875", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61798443", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62641138", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"63544452", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"74593496", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"112653703", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"142857290", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28601821", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32323016", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36042082", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39682783", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"43858365", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48412407", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49929174", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"55641389", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"61879513", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75842870", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95990969", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24385732", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31551420", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36294139", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"36556020", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"41406372", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"42742356", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44123437", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50002662", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55460242", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61969812", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71589126", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71673660", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24425103", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40433141", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"41593489", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43120624", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"46361416", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"49399192", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50878923", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"76743224", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79673575", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"109856188", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39262890", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40936115", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45887579", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45975651", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48085883", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54361739", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"56892482", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65759334", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"66497459", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"100516358", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24397828", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"25064698", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37213854", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"43861739", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55051159", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60347015", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62223490", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63981634", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82913300", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89999188", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92859645", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"99041443", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"110846036", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"146794548", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"29007969", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45169096", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45885679", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"47739056", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52236486", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"105643241", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"113838243", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"132100923", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26454081", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"28468801", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"30739296", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35838713", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"36259920", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38918312", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39262353", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"58857377", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61421600", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67073506", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72265506", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"73804833", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74598030", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"6950983", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32304361", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41815982", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44135191", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58769901", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60773593", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"64969021", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67926205", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"77729719", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"29599820", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"41541462", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"45290620", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45516836", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"49160770", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64349418", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"80266244", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"103710103", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"157283472", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27945647", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30065027", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53580451", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53772832", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61104687", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62289889", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"62394480", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67874216", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68118346", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24717107", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41219857", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"53624468", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59457477", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61482778", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"64626579", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77266511", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82485123", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38006246", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42443855", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47637633", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53539161", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55552015", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60846699", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65440539", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65627423", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86224954", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86779094", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"9479487", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"28178009", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37089117", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39598702", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46143109", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48110113", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"51469211", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59823950", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72355042", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"82593688", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95183625", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"115808651", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32134934", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"48692451", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"53402971", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56114568", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58999373", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"61889034", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81585973", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94552343", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21464138", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"27349389", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"34480642", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36181055", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37005679", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49975220", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50348872", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52766703", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77291186", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"9342193", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"25076913", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31666782", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38126833", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39453590", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"51483213", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70018583", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70574957", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74804694", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"110108700", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30272260", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"51869813", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"54227505", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"101911564", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47891487", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49050506", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49489347", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70577824", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"111558589", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"23125828", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"31989705", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35964373", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37056282", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37967956", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43559867", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52364989", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"54872466", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"60835421", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61524101", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62498352", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"63169595", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69989206", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23526154", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25895542", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32012641", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41463081", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45942628", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48691838", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"55170062", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55697913", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56527518", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60979470", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72306046", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87995907", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21845168", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"23082675", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24055584", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"28505832", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41455913", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"47109764", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"48805939", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61693659", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69781234", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78677427", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21159248", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44615447", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57292931", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75677438", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25087377", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"29146149", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43144300", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45015037", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55044061", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"55090442", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55516619", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58227973", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"62009127", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62093951", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70236166", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"74642255", "lang":"ja", "jira_id": "SIRT-3916"},
{"user_id":"80216767", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"98678179", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"24269229", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28329507", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31818782", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37054022", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40402894", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"58872812", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60649559", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"60797800", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"61202040", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61561653", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"68667322", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"115498532", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"142888565", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36043238", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40203003", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"48540948", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"68186173", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"99730314", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"146299006", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"41451611", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"42432821", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53078326", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55856669", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58332463", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60587851", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70444458", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77410119", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23154354", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27926944", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30925672", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57506626", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59242249", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63775059", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"70636635", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86121302", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"156820615", "lang":"ko", "jira_id": "SIRT-3916"},
{"user_id":"7870828", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21547270", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46975700", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47991364", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59056949", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63956832", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"81245717", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82181865", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"102018934", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"7065116", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"31355179", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44366270", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51083538", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51261688", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53174667", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64620245", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"71166586", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71686212", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"75367552", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"134589475", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27087605", "lang":"ja", "jira_id": "SIRT-3916"},
{"user_id":"29684456", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40641181", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"42055448", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56533524", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60181371", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62319803", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"69271870", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69753475", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"71952267", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"107466779", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"6962138", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31663641", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31700564", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37622446", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"39423959", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40255231", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51423687", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52419426", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52887248", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56713572", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"81248885", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"124137022", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"17408064", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"21791758", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37380551", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"60804758", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94618635", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"115622495", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"27446416", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31322464", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38276370", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"44647061", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44902231", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45608216", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"58802435", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86345033", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"99954267", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"29124321", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31038684", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35835305", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36546714", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44835057", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"52832557", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58448076", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27423898", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31807293", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"39869790", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"51218728", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"51674650", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"57812909", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77235195", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23681495", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"27620286", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28663549", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32743620", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40536716", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"52678919", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71051555", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"77721613", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"121442656", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"32353265", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45305843", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45973056", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"49162299", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"91182658", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"26529196", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"30664904", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48532803", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68206976", "lang":"en-us", "jira_id": "SIRT-3916"},
{"user_id":"19400580", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44164038", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"60959998", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70661351", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"135913647", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37595874", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37996988", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45156038", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51816302", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"58554885", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59695654", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68241284", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"40919980", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45165814", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48938205", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"51898173", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52310437", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"68357621", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73389221", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81345587", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84776412", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38165271", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49131060", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"54323119", "lang":"en-us", "jira_id": "SIRT-3916"},
{"user_id":"56492643", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64784242", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"65976264", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72240601", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75458803", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"75895548", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90832388", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38123400", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40242067", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"41368877", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43951004", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45068509", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"50352189", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56625556", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"61302997", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"65783668", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"69923217", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"143042686", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"12946996", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29373783", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66929689", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67432074", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91485057", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"145307940", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20905771", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42517694", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59990258", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67933974", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"115876609", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"147655005", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"35771491", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51114004", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54414671", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"56619875", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57184799", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"61889352", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"68381712", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69011172", "lang":"en-us", "jira_id": "SIRT-3916"},
{"user_id":"71677868", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"72039834", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"77900712", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"46015754", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"50010833", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50045145", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50988750", "lang":"ja", "jira_id": "SIRT-3916"},
{"user_id":"56952920", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57124549", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58999722", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68138034", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"94438249", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"32133832", "lang":"ar", "jira_id": "SIRT-3916"},
{"user_id":"36421255", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42403317", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50390866", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55299990", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58766031", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81816387", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"16807197", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"24112544", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30297691", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"32766877", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36100895", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39269430", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50294012", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55666791", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58560512", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67337240", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68598396", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"70141469", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42769694", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43574175", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"50540348", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55606531", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61299760", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68240113", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"111742237", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39657954", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54133408", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58994480", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61335858", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61511171", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63371797", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67102208", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86937869", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26788274", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"38535720", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"39940576", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49730056", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"55278008", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"55438160", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"59754190", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"62173168", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"94722811", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32654780", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57395621", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66436107", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"2436027", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28083528", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38307333", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"45906146", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54168778", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69564682", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85526485", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"132386514", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"29127083", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"38376897", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40780059", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42575496", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43304920", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74132504", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"10004253", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21697636", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41391536", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"44983068", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50895536", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51433718", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54523628", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"61532891", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"62314150", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65852203", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"86317726", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"108607064", "lang":"sk", "jira_id": "SIRT-3916"},
{"user_id":"4693957", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"9162750", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28344564", "lang":"es-mx", "jira_id": "SIRT-3916"},
{"user_id":"32510674", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37439932", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"45095296", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53493140", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54145063", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"65045487", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"82499618", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"126028937", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"30875299", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38863770", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38866242", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42775402", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47643352", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49515861", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"52215794", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95945126", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"100560960", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30415459", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37330555", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38064385", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"43203985", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50811610", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53623987", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53911552", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65715140", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70912812", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71233437", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79057499", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94522408", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"31671368", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32895354", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"40611075", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"53279063", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"147005553", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38592464", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45458612", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46149321", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68913362", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"86341205", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"27993417", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"28572097", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"29200401", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42048152", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"55706548", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59735881", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75001310", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35708531", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41783698", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43242878", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49600056", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49742297", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57338477", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59694739", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"88587259", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"135939221", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"13074", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37348500", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40605532", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41777995", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41959363", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43702180", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53646044", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53831996", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59843761", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"63310244", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77756654", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80030395", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46687751", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46868006", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46944043", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"71487831", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78556622", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91507826", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"12578353", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20553964", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24248653", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54019327", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"58799023", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61626299", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"100148815", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"108281793", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"154371572", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"31592025", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"40968835", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44052699", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55111297", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70643677", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"78843605", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"113539344", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25984583", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"32221971", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43082224", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"44485778", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"45112562", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51099838", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51720458", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53613691", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"136390337", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"18799968", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"32397805", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32728113", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"39885263", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"41890075", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"44719624", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"44825594", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"58761495", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62107892", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62402790", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89094612", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29464952", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36815035", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38259409", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39861453", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49864390", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53527650", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62707753", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"66781833", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90145957", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"114917494", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"133993455", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"30761903", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35972543", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36101407", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43817172", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44048610", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49690719", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"55604972", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77890436", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"112682256", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"165381573", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"40000265", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"41279904", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50889209", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51799947", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58693796", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59687548", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68511410", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75056853", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25138753", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30331271", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"36092134", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53548536", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"61190343", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67245220", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80078758", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31649111", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32209330", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32280739", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"34533105", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56810468", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80398186", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"88654207", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"109628358", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"32024213", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36864056", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36993601", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38766171", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38820488", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44785655", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51846199", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58485345", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67639075", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"124965469", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"4781156", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27967104", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29503183", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39423747", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40929973", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49276187", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55446893", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77667764", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86105123", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"101445324", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29024912", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39187244", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"46791562", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47086886", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56635303", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63632532", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"145781136", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"19040165", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25700751", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29586589", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"31408701", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"37145563", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"51450922", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53144042", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"55672367", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"129701265", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37105818", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"41430963", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49993880", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54212633", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55221954", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56446565", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57322969", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"86895259", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25852029", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"46110639", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48434524", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62432286", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64692369", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67972012", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68386832", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"82131427", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95276966", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39157340", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"45040952", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55318258", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76897197", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"38188768", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40510034", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"54968216", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56606860", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"61759410", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"65287901", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"109840097", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"176339922", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"36982240", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40168413", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"41288386", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43811457", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"44670707", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44951027", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"45989584", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51324646", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54454249", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54583823", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"60137143", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68782085", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"77153884", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79802661", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27754846", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"28587852", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35422030", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35598748", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"47975541", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50329414", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56206061", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61076729", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63048612", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66698517", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76276798", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22615771", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31944501", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40351585", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"40581820", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"42737031", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45139183", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51327289", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53979367", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"59407581", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80376677", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"104347051", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"44372131", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"55290190", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64176006", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"90422277", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"32568471", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40324828", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"54961202", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68083054", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"32107765", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37354699", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48871312", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52541566", "lang":"en-us", "jira_id": "SIRT-3916"},
{"user_id":"57097076", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"62453743", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66895347", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73411939", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21545963", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37696048", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38911962", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"45138241", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46652778", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"47287749", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"50903916", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55462050", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75843173", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"34639444", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"42337216", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51546861", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58346169", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58888516", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"72202504", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"85925063", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"18766859", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23503097", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35795837", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51784065", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60460491", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"65094391", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"67925678", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"78849273", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"134407372", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"36340120", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"37456025", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41130717", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55805313", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67360307", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97930969", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"27268619", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"38009892", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40241246", "lang":"sk", "jira_id": "SIRT-3916"},
{"user_id":"52958208", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53312092", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66707402", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"14823814", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35056521", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36942894", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"37022834", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41667325", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"46363241", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51063628", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54756958", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59055042", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"62720288", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"809282", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"25808729", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27945386", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"40100544", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49772671", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61621273", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63822180", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"48002478", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62130183", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62301498", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63690997", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"21082491", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37461821", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47551339", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"47602272", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"47636213", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49494600", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"56752543", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59670943", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"66475561", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72236211", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31649293", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32497942", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39636021", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"40627445", "lang":"en-us", "jira_id": "SIRT-3916"},
{"user_id":"45861321", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63607092", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67013728", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71681339", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81661224", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28008377", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32313653", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"36027623", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41854470", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42813501", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"49201912", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56295745", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"58398332", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60164072", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"61275938", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"62712937", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"63413038", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"83282083", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"130312525", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"28749105", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45338398", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"46392201", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48361065", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48469236", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48621719", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55292128", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63910451", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69518909", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"44067009", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"53604832", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59977016", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76501196", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22940776", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44988294", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80898662", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"12720031", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"22583966", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26187475", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"26649498", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27657451", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"31152542", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"45627154", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54591977", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"58340833", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66590602", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"68474022", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"131908087", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"22072567", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35728132", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47739386", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52280483", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77545633", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47530246", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"51664592", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52657935", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"57211687", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57972957", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59986632", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"76861471", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"109034003", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"2834670", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32305842", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"36752088", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38379588", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"46180812", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"53763568", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55991749", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78127706", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80004625", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"30451409", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39039602", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"39490396", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43848486", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52272223", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57112307", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"60961506", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78236674", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"133820915", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"18018430", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22740985", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32466861", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38265252", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"48647694", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51988563", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57021443", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"58487011", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79155998", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"143999536", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"152752139", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"37349487", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"49725920", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"56262845", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60990094", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61507101", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61730617", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62142823", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62262482", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"67949133", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"102952170", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29835363", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36189966", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88911068", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"11219011", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"16986926", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36547537", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36561980", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45790799", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48269714", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53883199", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"56784994", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"62029781", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"32141045", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39857826", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58709061", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59473807", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62542474", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63254486", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"63504152", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"82181641", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85987252", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95956691", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"26877080", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28526125", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29761913", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37003747", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"54387311", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65493873", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71009931", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36299018", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"37103921", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37966429", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41590538", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"54274894", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61999354", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62668982", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"98756234", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32354400", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45711567", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47665444", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"49555367", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"52932203", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"61987316", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"91028030", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"134809891", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24034881", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30204089", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"45506774", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46068169", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"111389422", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"15438005", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24250859", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28067748", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"32351589", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39656785", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40862530", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41298517", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"49412382", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"60117736", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"72418477", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"73175206", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"84015002", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"152633406", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"13530904", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23824480", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"38324368", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41006667", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42237503", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57752794", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79292022", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97683380", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"137119260", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39608065", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43732163", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"46472089", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49802817", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50923254", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59703197", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64113382", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69918282", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"116631731", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26856161", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29700596", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37051046", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40588755", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"48136987", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50644378", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59859628", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61139240", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61866123", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68029836", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69509482", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"157434732", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43324806", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"44487116", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"45295580", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"49137931", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"54490384", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"54524156", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57449554", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59573080", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"60209335", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66494695", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"18737867", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31829010", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38977962", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"41706428", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"45166739", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59030548", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61492714", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"87890833", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"22998200", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32354495", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40843429", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50755751", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80629671", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86961494", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"122715205", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"32029813", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32660438", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48281311", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48924796", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"60061801", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61267669", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"68656919", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"147212669", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25256458", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27299514", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47765994", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55919401", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57325221", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63076988", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29668404", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36186966", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42361639", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45951482", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47728266", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48307384", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49690576", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"62483778", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20854795", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44811337", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52596286", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"53937225", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67472709", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70274642", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"27528505", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28219022", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"52540103", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61343169", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"65610142", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72113193", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"78749872", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89311808", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"124027003", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"21356454", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"53472352", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53884446", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"106674660", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23154176", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26289521", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32757621", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37551511", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39839741", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"50273424", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51443649", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"54974371", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55580057", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61067581", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68034464", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91934287", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"102246609", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"110524847", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"25029063", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26991085", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"38009300", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"39976476", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"43276986", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"50877994", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69861708", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23946497", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27651481", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29046941", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30406702", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47530439", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53586020", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56834491", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"58155731", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"61832898", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"67316892", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"71713523", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"103336692", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"105995140", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"109104995", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"41897427", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43858468", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"54509284", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55247169", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55464502", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56250517", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57949869", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59070764", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60509391", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"100697579", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"41008133", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"43935476", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"45088615", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46663513", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"47147210", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"65217993", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78724489", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81827606", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23620443", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24125432", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26474704", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26854142", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43045922", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"46278408", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"61628140", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"64877133", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72964697", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84641532", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90211606", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"113698269", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27116723", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"36206649", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45002348", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"46101891", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"49460652", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51662456", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"55395647", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60306161", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62773276", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97962968", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"112598521", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"6477463", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"21894059", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26586080", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39903617", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55215892", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73685146", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73847068", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84667856", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25225389", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"26212908", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28812451", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43750049", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44145541", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50802768", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51174741", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53780784", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56658328", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61486500", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"23471406", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26621442", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41716869", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46087734", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48527007", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68265130", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91795418", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"99712582", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"105872376", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35911642", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"37138027", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39027302", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43065604", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43266701", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"45735822", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71312313", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91576130", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"144454797", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39400935", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"50375036", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53105150", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53328499", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68725905", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68853877", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74813440", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75579945", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94703069", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"118543062", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"10661718", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"29225008", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29311056", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31437151", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"36112199", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45838697", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47783180", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51251986", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59234492", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63188640", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"85314719", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30230561", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31445484", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45317151", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59309168", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"63239429", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"73061339", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"96258512", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"131571317", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"2976372", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56022715", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57218658", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62601485", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25681428", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"32079786", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"44736414", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50407750", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75485460", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"116643827", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"136199805", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"28046231", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32719877", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37288240", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38923070", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39050441", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40499324", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43903377", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"47622078", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54870968", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"59721089", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"62643743", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"64822820", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"119182789", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26924900", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30772578", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"36333298", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"37995278", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42498457", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61435183", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62594106", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"119642097", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"20589946", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21721179", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23878497", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"26661533", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"33396292", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39395153", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43643544", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44764131", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53908443", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"71498318", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39897716", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48515565", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"476039", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24562788", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46532118", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47090629", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69002843", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72957206", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91459690", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"143918021", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"51930152", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63466471", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65302746", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"66020973", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69552594", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"72869755", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91562916", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"10038713", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43825304", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45895340", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49867254", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"50480140", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50800447", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52168452", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57870918", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62744093", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68881951", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90994316", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29341516", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31711351", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"38752151", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"44561697", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55131829", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"58899176", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62541010", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69838714", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"20027899", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26906034", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"30209692", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42370201", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"57058355", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82419957", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"85449567", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91731482", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"107652796", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"107670482", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"320878", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37000753", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38229665", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45083341", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51070680", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"62766332", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41224177", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"43925418", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"57911352", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"70835577", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37364776", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38929901", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45097578", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46868978", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50064304", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"68583570", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77530294", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80521541", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"189923877", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54502482", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65785041", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80320969", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"32244139", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44290705", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"55026928", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55479304", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59065942", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66012606", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"120814201", "lang":"hu", "jira_id": "SIRT-3916"},
{"user_id":"44010440", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"53920769", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81168325", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"91456075", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"122450673", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32215820", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32607302", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38470131", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"39555676", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"42486937", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43285520", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"47485493", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47896635", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55434833", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59901205", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"66201899", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"3791274", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24371516", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30067930", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40998234", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51714637", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56625428", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59043348", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68844798", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"20880226", "lang":"ko", "jira_id": "SIRT-3916"},
{"user_id":"25942732", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30429574", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30562213", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55253684", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"57757004", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64706340", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"81423759", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24723738", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38675541", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"48766742", "lang":"sk", "jira_id": "SIRT-3916"},
{"user_id":"53064615", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"61625330", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61633802", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69023918", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"98968225", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"117932836", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"123923310", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"23529836", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27924545", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31187787", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"39720497", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60541729", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62807497", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64973554", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"110232180", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"21292708", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22918485", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30035947", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53621459", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54960408", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57120162", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63839897", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"65893071", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71661976", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72171511", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"153156489", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"41551869", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48794945", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"52519530", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53793004", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"66053413", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67114105", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68374596", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81602143", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"87304189", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"88105629", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"93228899", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"109708753", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"25681094", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41358275", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44592379", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"45967919", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50542713", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50726986", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55524388", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56615867", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61915641", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61960500", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63768160", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65396172", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"41173820", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"49126958", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54154376", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56029841", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"58689945", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65127541", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66291228", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69102599", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72320858", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"27320321", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"41635397", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41850139", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"43307914", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50824686", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51502219", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55682496", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"59943019", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65966864", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75252556", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76585463", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101164763", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"136687533", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"153131030", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"30673615", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40584095", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"41851876", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"54462119", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56239231", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67556555", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"12959605", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"15213012", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25004634", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38142463", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39566153", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48600226", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65825309", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"82289246", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89530946", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"17678435", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21753447", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31073524", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"40131043", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46402522", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51054737", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55075014", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66851892", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"94544897", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27187897", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"29750090", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"40781030", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"45068544", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48804085", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53678578", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55606555", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55694072", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61274492", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"75264080", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85401092", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"137062707", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"27793997", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30778277", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"32647580", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"36240082", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43522010", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43780499", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"49964072", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71921752", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86738207", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91875471", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"136527870", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37293155", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65199483", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75445290", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"110355354", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"38414305", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39206988", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42731222", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53562427", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"59599373", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76371266", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"143382855", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"32113546", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39398550", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40082309", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40537593", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"47219029", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"59484617", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59596089", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"63763109", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77440330", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"96626678", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37581316", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42219861", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57033832", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"60942872", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81335621", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"105987912", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32919611", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37717273", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50793177", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"53765120", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57125446", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"29358968", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77629741", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24002262", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36436463", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38960174", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43975433", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48222714", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48680189", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"50486911", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52821739", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"53367568", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59990014", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62553494", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85916821", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96683056", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"28733337", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30905307", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31761074", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36978651", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37124054", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39467701", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43308682", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47424487", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74083243", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"100685903", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"29850301", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50967977", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71884518", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73424778", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"85562349", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"4536147", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27273537", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"28899712", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"31903467", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35273948", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40295172", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"40860651", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41226955", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"66817885", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28734990", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"31909017", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"36942518", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38394223", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"42489912", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"43477290", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43737708", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"74353731", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95647801", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"24250779", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27011250", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31319090", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"34243464", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39436554", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39691216", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40516591", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41078941", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41456726", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41464058", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"47856653", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52438045", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53036643", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"62549892", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"67354285", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85795936", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87137844", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"105832757", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"118347972", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"27329938", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32701774", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"37224779", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50909778", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51593873", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57809857", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59663335", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"32669892", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"37113236", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"39395006", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41740145", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42679583", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43748521", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50028062", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"60218713", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61575728", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"67728645", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80443117", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27798106", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35079030", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36487161", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"38650882", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"43356877", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44797977", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"46132366", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58341273", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60634549", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"60964393", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61341186", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72174010", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73683517", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"5214349", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25696556", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36135415", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39863627", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45945789", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50620828", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56189441", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65426686", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68097446", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69221166", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83109678", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85904611", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"91751115", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"104726693", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"115711963", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"37600001", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67510191", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"73671082", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"98965008", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"656779", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35719343", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38125052", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"47439200", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56822856", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"58510813", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"60509364", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"86744838", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37008205", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"37829750", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"39619598", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"45365338", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"48030956", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"53107620", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72539229", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"102902360", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"106235470", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21588571", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26801407", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37639177", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"41848784", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43488158", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47034673", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55528204", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56652223", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"76932904", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"111755493", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"25609280", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26922272", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31522594", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48715405", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53841522", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"54688484", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"57974299", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59168503", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"73645908", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86007862", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91262743", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"134869759", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"24150835", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31614586", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43450620", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"48344039", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"52406147", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"66994708", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"77880045", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"133402835", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26652944", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31107381", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32766903", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39151884", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47224432", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47946501", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54965587", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56813589", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59026267", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67919379", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"140857", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"9167681", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30131985", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"30684909", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"42428533", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42529008", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44978782", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"47196079", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"47524961", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75373958", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"77708557", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22856248", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29715641", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32547599", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48439089", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64900301", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89738689", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"98864831", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"114723471", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"5638935", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"17976205", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29537189", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"38103208", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41973829", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43014814", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"54667700", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60085291", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"69356031", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72082114", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"100674039", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"103045931", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"32438605", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"36998654", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"37485338", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38202190", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39497823", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"55565630", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77895018", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"103228420", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"41203201", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46054244", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49042907", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52935905", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"53731409", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"63891504", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101196546", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"116156187", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"10126674", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"29687609", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42864289", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43893018", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44977315", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47357786", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79688224", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"87381544", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"141903848", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"38422388", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42807636", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48711887", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"52777652", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"55185172", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"72534748", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"100968082", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"9712289", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37097471", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39029482", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43399647", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"46390933", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55526326", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56975235", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59423400", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72681438", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"74857574", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"32184978", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"35611366", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"40625207", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"52758805", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63634968", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92028124", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"138381359", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20479266", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28067412", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"36127392", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"37383213", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46457994", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61061942", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65921551", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29054430", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37742173", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"40721011", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46511068", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47620116", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48436563", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"50113563", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52132427", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52953573", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53558596", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59960822", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60085640", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"61539369", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"71730807", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"72843927", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"74370842", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85866745", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"114442075", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"38821799", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51415354", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54246507", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"67628213", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"120646121", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"39656078", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63234601", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"64115201", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"25161913", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44584177", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50574591", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55345763", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61162985", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74123551", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"77454523", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31027941", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"37273605", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38341030", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40983873", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"41583157", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58598580", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61062706", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66071348", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72521556", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75884733", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"12410174", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31364454", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"37221994", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46221718", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"52379018", "lang":"en-us", "jira_id": "SIRT-3916"},
{"user_id":"54865449", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55583283", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56429755", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69613280", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"97709970", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"169842898", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30806346", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36227352", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40496410", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42631954", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50534644", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57204695", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62920686", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63355243", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77714663", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81656806", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"161731551", "lang":"ko", "jira_id": "SIRT-3916"},
{"user_id":"19500724", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"29228300", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42296879", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56645470", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60765786", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30679165", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"32043076", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36136238", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37671711", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39360360", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39771960", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44831006", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49911634", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51218977", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65590401", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72454505", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92379667", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"27848896", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28269648", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"36800240", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38930354", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42518421", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48946177", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49583057", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"53438091", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62890423", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"66089301", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"69002674", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"70665822", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84856679", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"30495554", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"44736606", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"48903272", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54791401", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61770808", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70519523", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74245646", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"96198135", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"135407049", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"187529885", "lang":"th", "jira_id": "SIRT-3916"},
{"user_id":"19526924", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27627300", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"42175258", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43029529", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"47967592", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"51002799", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51627028", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52592732", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"57060481", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57407430", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65148870", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"142981143", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"31435475", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"37571481", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"41845981", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43802925", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"57285727", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60791579", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"70760496", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"89890476", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29956324", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37074095", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45092644", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47547380", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48211313", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"50488538", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51115642", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"61657355", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"71577292", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88381382", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"113186907", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40085293", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41032478", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41116874", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42425294", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"50505086", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83640817", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"103368190", "lang":"ko", "jira_id": "SIRT-3916"},
{"user_id":"29668594", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35774504", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38533471", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42510851", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44941620", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"103584988", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"107127910", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51479325", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"54655698", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59580657", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84007547", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39420232", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40790954", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62776445", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71680817", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"77685627", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80732786", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"92952202", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28098904", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40392601", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40431277", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"50486475", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87197945", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"88027764", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"12956525", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25172128", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"36718927", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39586608", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"40317410", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"44121909", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"49992982", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51854002", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58870062", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60879019", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61109718", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61303865", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63827760", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65739721", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"30572697", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31355968", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35736851", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36514584", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40622915", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45598984", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55465990", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"56065767", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57002701", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"71076694", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"186010231", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"57908607", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65230373", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"66416019", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73685145", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"18887776", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"24512758", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"85076266", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"94418933", "lang":"hu", "jira_id": "SIRT-3916"},
{"user_id":"32915123", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35733316", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38634201", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61039187", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66463261", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92883714", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24755238", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32630515", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37874998", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41758332", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53378973", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57369325", "lang":"sk", "jira_id": "SIRT-3916"},
{"user_id":"66411342", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"70435913", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"133387726", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"137378708", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30042807", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"32004734", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"36142000", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43765833", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45213191", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"50373484", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52914764", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"71121635", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"114095498", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"119225031", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32145406", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32427600", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36468465", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"46449423", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49159160", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51499849", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52650274", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"66032061", "lang":"en-us", "jira_id": "SIRT-3916"},
{"user_id":"81808974", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101726052", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"143492474", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22043840", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28570688", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44010893", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44091442", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"49435031", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"50567460", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59806189", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"72793285", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"81037273", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"95832710", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23460970", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28633299", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35823397", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"45816402", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"52032945", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54435345", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71228001", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29366191", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30668841", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31773420", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37438044", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44119231", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44758219", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49844466", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"52141136", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"56602833", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"135617506", "lang":"es-mx", "jira_id": "SIRT-3916"},
{"user_id":"24535279", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31674535", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39201257", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47142327", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50469485", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58832687", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79604059", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101525649", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"109139394", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"129967995", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"36205287", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37566542", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49078755", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54795151", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60323959", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62347905", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62427341", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65653080", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66177146", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95965080", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"27406324", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28147105", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38237848", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"48648464", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57010725", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"64569403", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"84281884", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"92388372", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"28813338", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"37172685", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37867590", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38401542", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47069428", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56514708", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"59421378", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63122551", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66755869", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77096843", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"38268255", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43545039", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"45805700", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"46154810", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46181101", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79606755", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93196435", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"147584277", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26822474", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36428240", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"45212957", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50017929", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54960159", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58215778", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"104655559", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"117870080", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"35602421", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"36310817", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42995782", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48310279", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50957790", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53789034", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56503269", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66139538", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81656150", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84938790", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25833772", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"28241729", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"28485517", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32882676", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"37090411", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"47182659", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56497597", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"58573572", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"69984551", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"96660142", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"25267296", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31429591", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32407652", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43987086", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"67582150", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75558408", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78569281", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"99942601", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27320841", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"28308658", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39717198", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41077306", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45150665", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"54436825", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"56262231", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57173046", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"58243441", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70743167", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"190462152", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"17878657", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"20047263", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"41948624", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52480216", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63278361", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67027832", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73449305", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75541729", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96930216", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"329596", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"23202235", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"27842883", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36541115", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39167964", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44344315", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"45639008", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51534947", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59448663", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"125546825", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44148866", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44341524", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58941387", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67403748", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"72453342", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84184518", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87569834", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90063395", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"133479194", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"142595023", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"22346165", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40062523", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46585801", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"22965294", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27495317", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28758678", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"39550298", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"48289225", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48895038", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"54610420", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57066248", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73786392", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75881886", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"143783658", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"27762885", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"43460367", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46058360", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"75896865", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81920307", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"134718665", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"24943033", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"36085614", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41681473", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41688269", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44919401", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"56012954", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"56786983", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57129245", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"59179619", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"71943306", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"107318292", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"157125813", "lang":"th", "jira_id": "SIRT-3916"},
{"user_id":"23658915", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24934758", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28428038", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37557535", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"46005832", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51003656", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51059552", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"59098496", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63971248", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70552610", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"79606412", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90975890", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"114734522", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"158978188", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40793850", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"40872188", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"45390625", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52758778", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"67071014", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37040814", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40137348", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42629221", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46523689", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53667036", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55795447", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"56159500", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"60083765", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63234545", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64687888", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67600238", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"17089325", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"27226380", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32066895", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55600956", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28628895", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32715035", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39858348", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40758108", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"46476934", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55794813", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"109422045", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37649691", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44447891", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46049994", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54125679", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61383353", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64267935", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71166829", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89983592", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"23068701", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25862780", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"31222134", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"42907083", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43267660", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"59625960", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93951205", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"133120078", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"16226414", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28393500", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30387074", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"33392303", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"38153692", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38345387", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51846275", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55645942", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57958476", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"22158625", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25689219", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"28028314", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28180283", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"37043362", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41247512", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"41266410", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44062165", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50611060", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51035475", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54601573", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"72502620", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83438693", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"94435737", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"24516001", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40504651", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"46513847", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47933118", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"48341152", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48530987", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54462053", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78132673", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88273479", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"118223549", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"2142933", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"29016872", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"31832917", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"32339984", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"36611299", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40215142", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"41520532", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42896290", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"52865263", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64524284", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"109859402", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"6955428", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28882342", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32124799", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39046799", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"39748945", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45182095", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"48312101", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50155994", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51710249", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61113171", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65820074", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"100986547", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25852173", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26764651", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"31582993", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"35865074", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37809653", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45445134", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50052283", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58838153", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20514290", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22727344", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31252323", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37931748", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"45844629", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"51268655", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53087117", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60024601", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60324462", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"66502627", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69880289", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92492462", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"106899838", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39101961", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49920866", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52235666", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57495376", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57865458", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"58000401", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60896493", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65709329", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68473005", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"69222231", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86686982", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"29189885", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42912446", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"43562302", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50057547", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62635251", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88773104", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"108473035", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"135709747", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"27638591", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35679506", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"44595599", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47247049", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73260639", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73994314", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"100022613", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30692758", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39879289", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"40279023", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45511478", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"46765175", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48418938", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"135916638", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25410567", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29264361", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31461637", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45277280", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55096946", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"55997272", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70878821", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90840690", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"98267332", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"102984343", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45753639", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"48928212", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"53852364", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67122909", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"31845108", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40283064", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"43325648", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51056021", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"54420755", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"55698107", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55773971", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"68723270", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71699486", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"72763271", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"114158251", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"143381278", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21486148", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28284231", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31248602", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35801032", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"40222212", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44519528", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47098091", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62051354", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63409147", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"21736276", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30078904", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37988490", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44032473", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45098240", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47457274", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"48466287", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"50344335", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53599360", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70241868", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"90628204", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"114616351", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"42730089", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49789271", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50231266", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"50874110", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"100084972", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"15185913", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36420677", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"42885810", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45384414", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51935413", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"52223987", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"61278101", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"62949490", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81028901", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23191871", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26959170", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39324327", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40064090", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47900688", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54554613", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55707710", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60509134", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"63408876", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"80123902", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20442569", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24956789", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30388390", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36343285", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41083509", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56603663", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58685000", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"104927784", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"7174539", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37829557", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41200174", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43796844", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"86348743", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31782012", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38108224", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41022314", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"48596983", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"72550496", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47391614", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48882999", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55845150", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67199038", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"100783544", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"132612386", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"132673822", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"33152746", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"36000826", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48537329", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"51516852", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"52175598", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52944305", "lang":"en-us", "jira_id": "SIRT-3916"},
{"user_id":"55737183", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"60296777", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70293666", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75546491", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81620640", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"109614131", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"8481891", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37990065", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38203121", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41555271", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50458027", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58883954", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29488336", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"32435220", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37521637", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50470635", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51824156", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52542018", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53696642", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"55648980", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"60768428", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66062535", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89699713", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"125480254", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"131654351", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"152478961", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"28686160", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29325229", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32179121", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43054687", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53723615", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54853125", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57462643", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"61492658", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"77862796", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37123301", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45325629", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47976454", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55400796", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56682891", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59786839", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65210541", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94525287", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"200745", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31408842", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39908260", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42623184", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"43023814", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"50599065", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53898438", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55519762", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62417056", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"64761551", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67596712", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"2156562", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"22409523", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28036094", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"28472579", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35729468", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40408768", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46121992", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51054365", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68038009", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70402299", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74900461", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"41020687", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"50675721", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53158592", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56986515", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73852323", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"11975927", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21693738", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"29093892", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"30947583", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"43746874", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46570076", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47692776", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49609359", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"52156341", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70173332", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74085174", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29480067", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71206784", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"99655506", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"21047548", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21227092", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29590719", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"31272684", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37785929", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38424976", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38970168", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40489427", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"40954290", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"41749145", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"54560395", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55304233", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62405132", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"64909004", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65991263", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96162492", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26929178", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"36576566", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37978454", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57349717", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"64118157", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"7193", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22865402", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37286382", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"45713230", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48883162", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61564524", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72759518", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79947712", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"94177336", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"109012371", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20689533", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32394295", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"36231226", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"38803899", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39594008", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43453677", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54692436", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28757566", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"36645765", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"45507848", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"48037344", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53488196", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"133568127", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"44524243", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52561927", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53575409", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57011781", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"72746209", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"98189657", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"137626964", "lang":"ko", "jira_id": "SIRT-3916"},
{"user_id":"22835462", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23872010", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26696343", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35107104", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36580516", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36701179", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"48018579", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48387407", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51537883", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52020246", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"53416782", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56333531", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59676369", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65121776", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71136806", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76582029", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"84751203", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31658039", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"36902550", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38304736", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43862648", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51439276", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52196801", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"52349358", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57175881", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"107773193", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"125385213", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"7400301", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26308526", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"36440863", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41373668", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"57199106", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62160686", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67967189", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90062768", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"28860644", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"36137449", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37350829", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41520580", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49974395", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67062500", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26236476", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29418234", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41963632", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53901736", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74818546", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41235408", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46573999", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54358638", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56597385", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"112254164", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44439332", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"45540390", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56052968", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"66283728", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96769198", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"41661527", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43476473", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43614658", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"57608032", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"60979986", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"64984415", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"22449618", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"36126211", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39384322", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40713795", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42544171", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43145818", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43469679", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"49908694", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51363269", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51446862", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"54649149", "lang":"sk", "jira_id": "SIRT-3916"},
{"user_id":"59243219", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"62744025", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65687988", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68398788", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"74028419", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75429055", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75677953", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26012048", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29578325", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30814770", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31075913", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42136676", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43854517", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"45675060", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47789246", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51822805", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57924366", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"63028623", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68050517", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86798092", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37668535", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40059679", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"45623389", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47914440", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"49128806", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"51284644", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61905636", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"64453564", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"71625912", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"78339963", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95392809", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"108046825", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"36290895", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39173262", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40744370", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"43542514", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"50898188", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"112411278", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"36330399", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"38818165", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45215579", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"67076533", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75491320", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77424511", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89955887", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"132643629", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26841402", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36135238", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"37205593", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"45923453", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47698645", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49622923", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"63645385", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82787775", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87918666", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"109244913", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"159165721", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"13472688", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"19291261", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28227620", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30993962", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36941798", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44021745", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45238599", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49853482", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51926152", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"111792972", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41162653", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41966292", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"55135672", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"55265783", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60845677", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64671242", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"76088576", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82172731", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91700808", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"129517005", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29421321", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"32506939", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"38030725", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67626853", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69564962", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90591734", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"23671109", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24969609", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35635182", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37453376", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46865959", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"3481156", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"29131629", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46344587", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48191235", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"51640483", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64756454", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74220661", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"78469097", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79802054", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"39738335", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43051853", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43776480", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45813265", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"52133092", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52942909", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"54480584", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"55589603", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62099485", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65977302", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29484085", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38459387", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"39618101", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"70075625", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"26744720", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31834230", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"40851876", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"49646142", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50098574", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54790647", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55215922", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60468587", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91545378", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97965367", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101713547", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"112281827", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"3650097", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"6685380", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30905585", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"30998096", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37451894", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38565829", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"39939455", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"44923323", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55311004", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"59959946", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61129614", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70447329", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79391471", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32855986", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44974336", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"47814279", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50675298", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57179273", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77733872", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"32569539", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"32745095", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"36209280", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"49102767", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"57750624", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58848960", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70523525", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76497693", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"14794058", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28847299", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29031385", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"32400975", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"37800818", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47827049", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48956469", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50004667", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"64819348", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74889947", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"22745766", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"38715138", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"52342128", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"61627987", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"7302990", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"46838022", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47385418", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71563720", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"19762130", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24022065", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"35483229", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37397142", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"37751921", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38839222", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"52747456", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"53607259", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"55280233", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60544857", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63295962", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"63588337", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86513331", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89651623", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89840268", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"6936526", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"12720062", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26156117", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29496292", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30546976", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50465619", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53637035", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"58255198", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"61117243", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71881296", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73463574", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74597647", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"77699949", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"82552998", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"25303545", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37394896", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43702308", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44603067", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50680513", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"65855853", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66154457", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69577864", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94279188", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"100016836", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"36805473", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39616604", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46750692", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51390076", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57250762", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58993985", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22743256", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32578859", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36289839", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"36446491", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"44028663", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44263637", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51078315", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51753746", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54202968", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55395594", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"66330855", "lang":"en-us", "jira_id": "SIRT-3916"},
{"user_id":"69281077", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89961433", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"22817520", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31933083", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44265287", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44750041", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48285562", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59209350", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62344649", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67059094", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79002527", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"8649223", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30676462", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"40975343", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"43450320", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48960956", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54964813", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"62257449", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63901034", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67511243", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"84770676", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30058633", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41843065", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50283407", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56128718", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62339483", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77095388", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82200633", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"17127816", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32071479", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46032848", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53932552", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56103756", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"77721787", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"78764990", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85241753", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"6824633", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27526331", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"49887608", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54127052", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"59984308", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68614166", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79602899", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92824267", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"12606037", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30914251", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"31359770", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"35712838", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"55399044", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61306626", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65825831", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"67845863", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79641353", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"83537004", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"27897050", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"29461423", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29840789", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"40056300", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"54330173", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"62335501", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62763752", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"62844835", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"101165002", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"18409996", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31450613", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32574993", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53925302", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60571513", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"147647059", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"55330518", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56852575", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"112977665", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38210614", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"45917371", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"54978874", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"55304920", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58133020", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"58191936", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"59686140", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66235992", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"68268741", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68814233", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"73301960", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"74120489", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"75420114", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92679565", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"102713723", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36715380", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53138427", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54980622", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65976261", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75751462", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"467755", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27054682", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35946796", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44262211", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45550336", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46180969", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54488745", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59803485", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64505459", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69320552", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42983806", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"53377510", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55704591", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"69751654", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"3012348", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39767429", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27614460", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32089036", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45058484", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47734308", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48708302", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"49829698", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73581960", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101347792", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46901789", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47966138", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54094708", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56496279", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"63825346", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"71312703", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75894460", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86796245", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"135250438", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25817762", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30743485", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31395612", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32056939", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"35797593", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40662784", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42232355", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47185527", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50852312", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"108596777", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"16514453", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"46671324", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49758703", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"51914998", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52035075", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"54613622", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57398829", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94713899", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"115007996", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30473775", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54066912", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55361634", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24148863", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24429634", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32573519", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"40295380", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42049440", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42336961", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"42460717", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42511948", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43425831", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"55309214", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58977083", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"59808161", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65616156", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69109426", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"87193418", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"91695057", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"136794798", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"194577317", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27085209", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"30131293", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38756978", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41139229", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44672842", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"48689429", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"49721448", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50882907", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56583942", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58036844", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"94194798", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24346520", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30765746", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36296597", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"62595586", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67149541", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73710224", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78959031", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79742456", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86030052", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97098776", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"102493076", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"135828185", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"194556792", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"21018440", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24538518", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32760477", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45544067", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"51418768", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60608488", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68502999", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"72263796", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88404295", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20090948", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27243198", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29735222", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29777354", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"39946978", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49820459", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"60056571", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61508248", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24667912", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29047151", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"39880055", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"50259633", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51488053", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58301083", "lang":"ja", "jira_id": "SIRT-3916"},
{"user_id":"60037079", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70795534", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90997180", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"19654336", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36672485", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"44827022", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66039903", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77946628", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85959555", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"86319273", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32126202", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39818411", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43252971", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52447163", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59803387", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62955821", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68441654", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"68764838", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77599577", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79360150", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"19336638", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35817560", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"36501407", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40616389", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41893752", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43957044", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65589966", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76914878", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97703217", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"19942092", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"23728128", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41627022", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51476643", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"78510382", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81013270", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29399004", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30065078", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45348185", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48340907", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50182519", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"52429678", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"65399529", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"73648299", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"79844714", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"121155327", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"23168796", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32176898", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37541229", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45969833", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"53705387", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54559073", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60608122", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65197895", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75468214", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"104581258", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"129117300", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"7502836", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"19659771", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25655285", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"27958904", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"30084132", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41801780", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44909066", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46137378", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53376123", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55912774", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67882879", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77932043", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94572338", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"133342247", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44319522", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63447695", "lang":"sk", "jira_id": "SIRT-3916"},
{"user_id":"63580022", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"97315226", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"52722825", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54220136", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55394429", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61996780", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"68778288", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69392195", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26498424", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28762133", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"28891299", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30218226", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36334529", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44700117", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49603918", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"58428479", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"25239979", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43622540", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"45630268", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46866489", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47028277", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56792668", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57629891", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"75974670", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"80955025", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"96168986", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"117175190", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"30574085", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"38560899", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39417023", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42238349", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"44857542", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53851680", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67943755", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28003721", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49064566", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"49570848", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51197472", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59692094", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"60720401", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"63057153", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68596268", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69273118", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"143511966", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"26311194", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31606651", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31840313", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32867651", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41204065", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42112897", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"45572570", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"58796631", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"70984771", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"76639357", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79263784", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"9256066", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23456471", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46332968", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"50315693", "lang":"hu", "jira_id": "SIRT-3916"},
{"user_id":"59978953", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78828468", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28226024", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39212536", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"42317441", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43627771", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48833683", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"51791826", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56448168", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"90274361", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"29841994", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35690861", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40274164", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41226924", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44922600", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48517278", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52732614", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"61638759", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66116150", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67661311", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70844872", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"108134299", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"112331026", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"32109667", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32550425", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"39293429", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"41910327", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44373254", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49271775", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56234109", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62036385", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88419912", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"26288737", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35816953", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47424188", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47767324", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48183120", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54090083", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56968160", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63874253", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"65434857", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"71107234", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"91981520", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93098292", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"100530215", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27913743", "lang":"ko", "jira_id": "SIRT-3916"},
{"user_id":"29644574", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42523723", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53221987", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"53270866", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"53829167", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26652220", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30997351", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36266615", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39655745", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42147698", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"51295717", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54925431", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"60679418", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69236146", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79987350", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21392535", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30121331", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"36317064", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37660642", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"48602099", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76995524", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"142321288", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"29766415", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39124807", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"41249829", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50464283", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58767487", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62844285", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66184231", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"68873242", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71204387", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"74498850", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83235255", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97871310", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35801446", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"40631435", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51065033", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53711487", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"63742789", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"77773394", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"147465016", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"50531774", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"51744800", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56362637", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"102610175", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"159569579", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"17242154", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"25199180", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"36488314", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36983084", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39988047", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41712079", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"41906613", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48875363", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50196521", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"71227055", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"260981", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"7183859", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"19451987", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29671567", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32954353", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"38275621", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"39521335", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"53424861", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"63505416", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"68272370", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"77949287", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"110310189", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"152470122", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"170000725", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38208965", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40101584", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42287781", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46872007", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60554361", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79661473", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"29156573", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"34406049", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42545329", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44411361", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"51105823", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71764624", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74900780", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"80741638", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"112564392", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"22828077", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45409508", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48909476", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52543740", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"106737269", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"27203725", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"32837185", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41177625", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44602835", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"52138311", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53689928", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62815373", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"90793631", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"172833221", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45935510", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46553424", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49320410", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59737315", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63626609", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"66701742", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"7990181", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31561227", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31991196", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38480431", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39176440", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41382555", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"41495047", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61999236", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65099922", "lang":"zh-cn", "jira_id": "SIRT-3916"},
{"user_id":"67072973", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"93976138", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"24308210", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"27215694", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38814403", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51913514", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"56128244", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65769163", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"82824168", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"114193343", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"15382403", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"30940574", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"37225674", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52067125", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53786965", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59193141", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60273639", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51038775", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51656006", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54165172", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57146318", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58435282", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66902086", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"86348966", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"26202795", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35938167", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41641266", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42300592", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45178223", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"48609202", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54045752", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21918922", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29832698", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36765862", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38886041", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39827958", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"41250479", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41779407", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"86155593", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41990423", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48877286", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50666575", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"53825833", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"80951883", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"29363106", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47319442", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58981664", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66189403", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71917266", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72193526", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"114035964", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23487529", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31049224", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"31862798", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"40834921", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"47942611", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70637049", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"80247947", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29132909", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"32701136", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40293974", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"65170618", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"81820380", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"82385940", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"107914911", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"8730441", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24370885", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31513713", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32420922", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"35734217", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"37926423", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38688519", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45924552", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46157284", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55236117", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"55476746", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56274725", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57149618", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61658669", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"91140879", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"122523613", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"7513120", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"30824048", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"35361924", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41815218", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43886503", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44339944", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62356365", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"68301770", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78422943", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"25567119", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26540218", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"40256521", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40325142", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45440086", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46578587", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46593563", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"51024040", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51937440", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55955387", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"66630094", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71790708", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93241760", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"108039626", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"26068448", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41696198", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44734506", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"45076469", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49747508", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52090191", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56678701", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57388413", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"66512810", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69254982", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"72980184", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90727846", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27435637", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40145192", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49707058", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"50275500", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"98274439", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"14040866", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25992306", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37271804", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40162674", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"55180116", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56789765", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61064457", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"83139004", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93908926", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37389176", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40456327", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40564635", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42844004", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45832964", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46078507", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"46158930", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"47872634", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51821758", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"58859263", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62743796", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62844105", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82394955", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"103455524", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"109009012", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30262821", "lang":"zh-cn", "jira_id": "SIRT-3916"},
{"user_id":"39730432", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59130537", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61997604", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73090528", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92278014", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"143744958", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23256528", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36924651", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42528265", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47242667", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48702842", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53961819", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54243709", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54258493", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65770199", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"147465020", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25498581", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36884735", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"40596032", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"43469813", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"52357530", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"67041620", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"69524440", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70197964", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28280300", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37318632", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39560575", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39947956", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45826615", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49224631", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50038012", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59183006", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67235144", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"76905401", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"85966766", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92392481", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94178354", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"129889469", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"25170471", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"40417757", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42406618", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45490675", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48123014", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51518991", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"55766492", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"61866777", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66990656", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68405979", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70516151", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72385138", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"94571527", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30149449", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31712112", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37267169", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39357374", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44659658", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45200799", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46511509", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50055954", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"59252841", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61313368", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61354039", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"66940817", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83761023", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21533453", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25852022", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27852179", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31669987", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38269249", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42048607", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48085640", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52799509", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"67899898", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"75118604", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"99673268", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20616491", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"25546940", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36174127", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38696168", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41853162", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"50556302", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57429299", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60955962", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61512902", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97984984", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27337874", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35962016", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"45640548", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"55029969", "lang":"vi", "jira_id": "SIRT-3916"},
{"user_id":"59476426", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"63886706", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81443690", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83556723", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86733210", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"14029589", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35856714", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41328719", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"43622753", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"4289206", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29414770", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38983621", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"69395140", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72844481", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"77916026", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"100959829", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"10661130", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29223216", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"38958014", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42615519", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50934595", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"54024188", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60636451", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61459198", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63124718", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63622466", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84266708", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91997019", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"27407433", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"28459015", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36105125", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39868850", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"50962987", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"157425779", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"24026855", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28694595", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39728586", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41078612", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"44684826", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52846946", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"64945569", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66155759", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"31486526", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39431491", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41688025", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"45221641", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46118763", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54097325", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54929388", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78757420", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23371016", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30400558", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43449726", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46683191", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51724602", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60828954", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62568446", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"64196332", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69483103", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72962776", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78245034", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"132240779", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28489402", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28800122", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29523848", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29645723", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52890347", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"144225380", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"179552182", "lang":"ja", "jira_id": "SIRT-3916"},
{"user_id":"32005593", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38261038", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56981595", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57033784", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77601716", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80194204", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"98445155", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"30780136", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38411343", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"46864735", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53248973", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"53490918", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80133640", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"83275645", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101580166", "lang":"hu", "jira_id": "SIRT-3916"},
{"user_id":"29737277", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31965252", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41345328", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48933490", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"55145007", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"55618221", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62280549", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"16696896", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43193923", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"48551308", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52250806", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54655886", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57968586", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"64167718", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28513406", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"30411822", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43946923", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60300696", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62910131", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"133587543", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21013275", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"32141148", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"37494936", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37543674", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54925217", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56672883", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62961848", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68428098", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71937056", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80460855", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"105970141", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31135934", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"32000804", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32096472", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37176243", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37689866", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42401120", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49530403", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"58857358", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"62409690", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69067313", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"71979777", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83299619", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"98450444", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22411977", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36017208", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37143023", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38297472", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41161954", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"42361676", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"45844130", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"46857557", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"47638566", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53887229", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"55211548", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"68724786", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"84789988", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"27401271", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32819597", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44973309", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44978834", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55312789", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56569694", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"62310373", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"116490797", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"150803955", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24002392", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"29061761", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30240704", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40191184", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"63649572", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75961377", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"27094856", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31498402", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52226109", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62178928", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89887473", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"120460051", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"7048387", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23184657", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24844006", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"39595102", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45092778", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50689706", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"68976431", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21921749", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32478150", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42804130", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49876898", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58536761", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63536971", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67799476", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"20173706", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"49857952", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"50040735", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"55324947", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59805163", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"60394101", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71852631", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"29920265", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"35665366", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"39159998", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"41122850", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"45029899", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47667266", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"68628608", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"95125062", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"103990042", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"9012352", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27893781", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27950759", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32835812", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41712446", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47937271", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49946840", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59669894", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60119167", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61606066", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"65286530", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69884499", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72854978", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"137140358", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27880961", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"28999403", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"38906294", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"39007334", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40153375", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41063718", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41137724", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41738168", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44567934", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48215781", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53506377", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54122922", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"56674883", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65089079", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95337506", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"134689413", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"7937045", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37951780", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38696926", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39972861", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40104881", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"50962012", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61891301", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71076964", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76562675", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84472219", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"86142148", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"21736388", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25425127", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42244016", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"45784571", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61310916", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"102655684", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"10744561", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43416139", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46332610", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51237588", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61335104", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"71297091", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"103927275", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"109509606", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"38064184", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40338507", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48162500", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64252381", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81088178", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23994815", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37451771", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40042541", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"41666581", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"49852237", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57313219", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70670746", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80199429", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31289928", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39905718", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45003450", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"121564374", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25646132", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32431016", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39417802", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39766557", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"40973367", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"48753451", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"51771257", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59249436", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59873793", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62011314", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"66535235", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"135380114", "lang":"ko", "jira_id": "SIRT-3916"},
{"user_id":"159522071", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"20666751", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30724380", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38431453", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38647167", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44294022", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"45813996", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48591327", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"56701560", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59147830", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62486420", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69466926", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72195687", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80974989", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"93534219", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31998124", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38876314", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42124642", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"53007615", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61209238", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63218988", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"76490066", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76962837", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85087318", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"114504037", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20723748", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23144154", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29464591", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37440171", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"47771638", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54111920", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"65108205", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74228599", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"103222771", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32219219", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36499503", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"45864834", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46862144", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49378678", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"108016867", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"17604098", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28914984", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31303681", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43592635", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44939253", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50073984", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50700875", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"54397113", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"73378458", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"6976015", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23583923", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23604249", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27478382", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42204277", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49119471", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56647305", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"83693302", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93365875", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96111883", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"256935", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"12929055", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31882419", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"39143482", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42999061", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59789943", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64881115", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"81357669", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"135086171", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"32069821", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32154600", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"37266313", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42676963", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"45159863", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"45615478", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"136208029", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"11904971", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"22739311", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30901614", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"39300603", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"49384331", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65721549", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"68404094", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78110707", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"80663290", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"105449831", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"25003761", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31815543", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36722216", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40668946", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"42057916", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49711509", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51507012", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55097549", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56294028", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57377535", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57558048", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"63033449", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66754725", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82527317", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89031323", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"108144607", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26302559", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37281069", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"43984440", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44268245", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45013592", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47012525", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"54290744", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"54944723", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71618603", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73152811", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59524297", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"64960418", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65126791", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91616781", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"133755371", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"24999764", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32956273", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35624261", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"39653754", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50406994", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"52610724", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"60285048", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64838462", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73766666", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"8950829", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30597002", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32574850", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43805199", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"47014509", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47534096", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51173284", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"63959665", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72967664", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29352276", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38332942", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"40494668", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45615432", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"61766734", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"63985691", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27442630", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"36153391", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41953209", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42021422", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53859916", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"59993227", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63551705", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66991582", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"78726955", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"29096366", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29919169", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"37227904", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41068357", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"47143247", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53625689", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56263280", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"58916565", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72953179", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73572169", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77537442", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96583763", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"101631275", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27413375", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29851988", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38607967", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"47809623", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48415441", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52163839", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55426474", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"58763940", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60433623", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"21298336", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26642968", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"31273165", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56878775", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"60182395", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60494876", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64808496", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68066443", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"72180465", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"103680873", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"140076835", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"1162924", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24919740", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36257660", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39643033", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39869977", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42122738", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43392729", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56752340", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20516794", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29186902", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"40520684", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52693658", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"60399497", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"66983291", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68509258", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74843353", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"154758505", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"43374564", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43501073", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56165249", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"59080326", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79834906", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"18793339", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"32835036", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"37237320", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39004723", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39565041", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55003716", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91451746", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"118339119", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"24562522", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32382740", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43654672", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44013354", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45707201", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"50948324", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"65839190", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81267045", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97870491", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"31028200", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37934170", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"39334979", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"43898453", "lang":"es-mx", "jira_id": "SIRT-3916"},
{"user_id":"44956706", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56897192", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"102091682", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31892259", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32563468", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35945307", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"47250099", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"51498989", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"53796920", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56715997", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"105282108", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"108819609", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"115270942", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"1373296", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"27520625", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35856506", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48764784", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"56750552", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57803654", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59355102", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"69630121", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72460165", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82770476", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"27092921", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31305118", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32606098", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"33294761", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79325086", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"136063170", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42756096", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43527101", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48742750", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"57980739", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62951382", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"97225302", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"100172455", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42777833", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"45176845", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47481111", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54318087", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55340553", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71094066", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"78918498", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92838759", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"137120389", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"24175549", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35599899", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36863798", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50085960", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51217073", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"88349978", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"107731658", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"7820576", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42770749", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"45456681", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"54415456", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62107603", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"27357838", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36969080", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37095150", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41116355", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"43062344", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"46136826", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"49258135", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58642807", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61754096", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73548873", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73663902", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74819356", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"76948689", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"89081828", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"23445632", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25284356", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27251284", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31304611", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36105126", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58381052", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"60862017", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61706870", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"63675549", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"77823259", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"43034605", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57584195", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60463250", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65500430", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76924562", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"107445273", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"15166633", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24400903", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"30193661", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40111404", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48161155", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50238103", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"58617737", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72142123", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38543308", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42011716", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65297485", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"91516474", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"100322362", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"116035914", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"22960907", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27091145", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36921851", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42400304", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50154496", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54328536", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54694867", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65147036", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85999371", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36611620", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42080145", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42539296", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45077350", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48711513", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69347473", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69762062", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"77450335", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"91400871", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"99778548", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20701686", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"29003048", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32984342", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43749167", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57843855", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87854103", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"98510364", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"134072937", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"28384824", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39518681", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"45458787", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46615161", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"54611494", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"57617682", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73961817", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"92720984", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"98328387", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46705949", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"55361037", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"62010638", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78552594", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92848757", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42923138", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"46425060", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50916984", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51920987", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54204360", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"63868722", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69622411", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79919803", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"146022156", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"18626640", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23926445", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27125670", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"39789550", "lang":"es-mx", "jira_id": "SIRT-3916"},
{"user_id":"51121747", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65336728", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79176074", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"84527460", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"100739052", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"133925927", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"26265082", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36380024", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36400401", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"52290622", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58927597", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96704741", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30128547", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"32280738", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38183656", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39330061", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40845289", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42104495", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47366285", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"60185848", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70279343", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72832378", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"94419662", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23714210", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40465228", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"46202443", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56329367", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"61116952", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88312775", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95870114", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"12937960", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24192963", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27419672", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27507129", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"40262118", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"41237343", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"45489083", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52533613", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53937510", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"64951595", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75994658", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77790131", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28574030", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"29484037", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32845144", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"35968219", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42713010", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"53049822", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54710058", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57057987", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57227321", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63046122", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71654279", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"73375119", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"79374362", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"12703296", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28103268", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30099416", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32363400", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"35031460", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"50248173", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"57462518", "lang":"sk", "jira_id": "SIRT-3916"},
{"user_id":"62718253", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66415735", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"25368871", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28006220", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35628972", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40396621", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"47719161", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58094185", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"64122681", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"74997784", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30513608", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38842120", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40923245", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58123272", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86431905", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94339750", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"28256431", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"31086523", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35996588", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39117910", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"64270867", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69040715", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"81181559", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83863958", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"23758399", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43325673", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55521770", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"55718969", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"63956762", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64819534", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"17662106", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20260659", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40357878", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45859775", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47131996", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49611617", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58200967", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64340403", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"75204747", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"80715761", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92436465", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"106690387", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"25169910", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36889942", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40362118", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43282217", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43514508", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51503763", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66815768", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"112422908", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25405802", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40403763", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43126133", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46545274", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"49891824", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50987629", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"57826666", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64060472", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78481874", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87778755", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54254166", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"55043706", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"61904261", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65898323", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66717318", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"75360082", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93447948", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29817507", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39068031", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44865856", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47233049", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50001387", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"59317015", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62253893", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"93518952", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35700567", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"40628665", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42823694", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"46228346", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"52121475", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57315365", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"58711798", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"63187665", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"145263673", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"11485263", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28833403", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"36143816", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39589073", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40797579", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"49083645", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58956697", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63411423", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69054169", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"74952819", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"22680841", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"28205982", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54251341", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54693820", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"59758800", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65141892", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"136152195", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"36900758", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40168053", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"45663667", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54740143", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68157931", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"83441472", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"99922605", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"8903810", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26575362", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"27505035", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42177765", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"42778703", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"30707835", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36799619", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37582655", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"39138591", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41630573", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44246346", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59475078", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75292648", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88820722", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"144236560", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"10488200", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37880923", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45065126", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45718672", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"45720162", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"50662496", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54658175", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60214162", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64613486", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84968039", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"85046095", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"123750398", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"32085830", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39854553", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42844930", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43821553", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43845139", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47298901", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54927142", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59355927", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"62777193", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65688955", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71266952", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73689398", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"142490045", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"843399", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29696253", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31263809", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31758275", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39300533", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47071880", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52073288", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54449541", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56624747", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58491841", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65504809", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69400185", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"154680569", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"8730749", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37785474", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38099074", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39478743", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46998807", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50857221", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54602712", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58699670", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66912782", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50313732", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"54139597", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"54825500", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"54843664", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57422088", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"109436854", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"27805919", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"33834872", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36492051", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57242266", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57389413", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66769307", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"77869380", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"80069963", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"116496859", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36787411", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"47409026", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"50974238", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58243514", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62832386", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83053832", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49133986", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21826731", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29000874", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31561427", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"36267879", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36488281", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40716082", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41441079", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43740997", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"47458052", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51849906", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52666084", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"112390047", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"14315524", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21609623", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23510846", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28068583", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41134315", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"46482508", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47519353", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"52715642", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56993756", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93782499", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"108633017", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"50539270", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52611427", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59121983", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71593243", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79394850", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"120283401", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"28515159", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"30500590", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31281375", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42646684", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43513714", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46000888", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"49315554", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"53179775", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53445097", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58593509", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30061283", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31615420", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45490835", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61889453", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69872904", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"71940795", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78526920", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"98324915", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"114333652", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"130118642", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"153869904", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39618484", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43298882", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"46918204", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"60913925", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"137368561", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"146197768", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"22079250", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28413930", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"32071867", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40697080", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"45346543", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53760990", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55122038", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64438586", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64715456", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25972115", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44155821", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51026541", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"58826629", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66254286", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70976201", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76829058", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79004839", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"80379348", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41478583", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50988147", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"59828414", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37991578", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38962147", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39175996", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56562758", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"110184819", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"147434154", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40182823", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"47333531", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49946825", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51949683", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52565010", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59103282", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71277796", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"86066609", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"100632337", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"41193043", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48472029", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49686211", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50332395", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57861679", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64527031", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73232922", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35082174", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37287763", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42219916", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42844815", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"50075170", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51220271", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"57259734", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30388367", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32880250", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36847920", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"37491722", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39855269", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47918858", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"57570148", "lang":"ja", "jira_id": "SIRT-3916"},
{"user_id":"70624744", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79181561", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"94270169", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26551038", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35690830", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42786293", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"58873252", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68748462", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74840050", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"129755674", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"23923583", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"25949074", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31574543", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37326330", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39539041", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"42406906", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"44578634", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"46927913", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61815071", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71239180", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"72238914", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82884652", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"105925710", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"519313", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"28337246", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"37737814", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"39019835", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"53435762", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54147395", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"56451572", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69381697", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"25853162", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43627310", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46964667", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47681327", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"53411510", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"56738393", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61114891", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"80523922", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"120472594", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"27598211", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"30086738", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40008866", "lang":"sk", "jira_id": "SIRT-3916"},
{"user_id":"49781866", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51849471", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54112620", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60707307", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"75993407", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"126326100", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31124898", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39481735", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39886495", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40837587", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56874058", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69799946", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"70591701", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73602052", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"27933438", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32294150", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36882095", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"38809551", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45914729", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46522705", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"50616841", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63887646", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"67489620", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92400193", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30532936", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42074267", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43950051", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61975767", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"62929608", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"69980559", "lang":"sk", "jira_id": "SIRT-3916"},
{"user_id":"70528768", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"78845780", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85527722", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"98841700", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"107824097", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"152266382", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"172655729", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"31499144", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"36945615", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40277731", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40445616", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45736852", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48349295", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53787038", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54189733", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59308441", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60547573", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61133331", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61145308", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"68444700", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69883674", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76224046", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"133416201", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31951884", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40640039", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47124748", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50009040", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55037114", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56642643", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"56866634", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"59240913", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67230422", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"72975963", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93356016", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24481800", "lang":"hu", "jira_id": "SIRT-3916"},
{"user_id":"36129860", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36366970", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"39556788", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"42430645", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"46832670", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47076867", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60740918", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"65461104", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96157120", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"28065860", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43051319", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"45546093", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55069651", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55537869", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"86920457", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"105421781", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"21605439", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24267781", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"43268725", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"50608138", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57170589", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60867798", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"75226178", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80762616", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"86082960", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39045937", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50167594", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50642348", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74092555", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"118041463", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36706998", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43162226", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"47237702", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47573232", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"47706891", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49746120", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50738446", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53606240", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65321566", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"82452423", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"103954443", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"107894754", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28219127", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44023463", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"45090560", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"52605516", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59608525", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64903225", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67574019", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91793318", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101340496", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101923614", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"136755580", "lang":"th", "jira_id": "SIRT-3916"},
{"user_id":"136923251", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25821090", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29340071", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"38159956", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"48968015", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54260255", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55035342", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"26234368", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"36063144", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"36725464", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41955602", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"64158752", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"69802810", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"70566659", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"84299578", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"110335013", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"177239148", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"29214396", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"30014640", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38446744", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42305259", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"61844460", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66495767", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75079788", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75499535", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"87164705", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"136009703", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"147470787", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27787989", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"77837331", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85181329", "lang":"hu", "jira_id": "SIRT-3916"},
{"user_id":"91994763", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"26369720", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"26789275", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27401723", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38595107", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41103590", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"45896083", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52845228", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70336888", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92957150", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"127147963", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"31036429", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39996748", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46674077", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"59764483", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62135165", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"142562841", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"17061121", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31181596", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35637896", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36459312", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41217080", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44392775", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48261709", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48558919", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53081919", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60948823", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71180791", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72653861", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88061964", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"117770018", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"130056268", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36670471", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"37722671", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38148173", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41080002", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43104352", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45424938", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50378433", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"61580353", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71362723", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"81342666", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24099864", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"31102328", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31298189", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39009866", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39370312", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43699932", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53942204", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70962915", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70971973", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73263536", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"153817690", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29321149", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30272568", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"42583251", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53729278", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55300877", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55798758", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56732305", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58503114", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61894613", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"70424952", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78894133", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"21382466", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41490331", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"46023550", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"47299056", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53870567", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55164340", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57299501", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61128783", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62876712", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67822025", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"71264987", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74649753", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"86718120", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28230981", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36483265", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46033023", "lang":"sk", "jira_id": "SIRT-3916"},
{"user_id":"49036136", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55709085", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70921989", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72571783", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"110189857", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"117324011", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"134956957", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27124836", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"32981090", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"39924442", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"48320828", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"50600441", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"51985175", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"92877020", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"3330484", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31163988", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32017282", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"32459360", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43248407", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43497127", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"45909112", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46752236", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49421430", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52895076", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"71219655", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"74222734", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32354891", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"39454452", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40073764", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48285755", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53300257", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55699568", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65115437", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84195373", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"25920913", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39914864", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"41720976", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42552510", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42929257", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43704752", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"63319127", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65997495", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66508378", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"75742943", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"15880873", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22844339", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"23798339", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30733157", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37306365", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38565183", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40329194", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"42080045", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42415714", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"45949911", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"51868440", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"61098155", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"62495058", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69215504", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78707035", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36048285", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50876334", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58335020", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75714691", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"116715722", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"21984582", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31888180", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"36498929", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36896851", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41531354", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43095085", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"47582824", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"47951395", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59713845", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65673623", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"67036983", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78327722", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"108641075", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32625784", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"41901158", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"52723616", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54516885", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71505835", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73675611", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75079925", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80772554", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"154956106", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"25706726", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"29299254", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"36776103", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38520146", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49134773", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54237106", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"63225922", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66246079", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"117214126", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"129808021", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"135130231", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23532013", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25767683", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36666040", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36880378", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38649140", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38898184", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"41835585", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"54019488", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54217461", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67139102", "lang":"hu", "jira_id": "SIRT-3916"},
{"user_id":"135418077", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39227018", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40108948", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41936903", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44276997", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55423701", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56631582", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57574477", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58589157", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60160249", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61303934", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92655474", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"23420499", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26003207", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29188740", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"31694233", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38545842", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39777111", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45370429", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50981226", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54159118", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55301896", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29624475", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"31518176", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37475170", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"38322025", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42400136", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44507616", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"51989195", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55161107", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69624236", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"14134953", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"14226214", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27325992", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45114735", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48806662", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49864034", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52478530", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66991107", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83024028", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26759694", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28640280", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29379201", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"32182510", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32471305", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"32726179", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36458068", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"41749668", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44929280", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59641277", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60059984", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69565931", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81863700", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"124634493", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"6578720", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"12614798", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"37739879", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39208393", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41059894", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46532655", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"60681269", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61247609", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40651459", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40983481", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"41373712", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42937591", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"43277855", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47210949", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54788156", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67599984", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37743489", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"38107390", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38976676", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39993416", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"43060478", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43591402", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45021510", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45101080", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49925463", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50991831", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56754269", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69483842", "lang":"sk", "jira_id": "SIRT-3916"},
{"user_id":"41535625", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"41718154", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43761323", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44872354", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49688699", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52280445", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40037995", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48580491", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49200636", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"51703922", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"56241367", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57100373", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68274438", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73179333", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85998985", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"23680120", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23809762", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31184815", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44959013", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49164897", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56997342", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"62973517", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"98993308", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25796731", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28821399", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38399401", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45208905", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"49398331", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51021856", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61091031", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63482603", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"70300300", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70712782", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77255855", "lang":"hu", "jira_id": "SIRT-3916"},
{"user_id":"153218168", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56329331", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62349380", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"63894367", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37226538", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"45434333", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51336610", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54472092", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60835799", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66365608", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"116164153", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42046536", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43349736", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45556768", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"48792709", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50602508", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58288591", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78012920", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88262364", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"114593350", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25715689", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40319696", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"50984496", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56225773", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"71938207", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91432670", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"122190586", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"140094083", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27197240", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"44078558", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45144490", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46964879", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53350468", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72005326", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86944883", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"112004741", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"114525044", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"30211040", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46656134", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48649087", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"75045835", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77255996", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"89836449", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96312020", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97878642", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"120939276", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25004599", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30796221", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"31318158", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31513984", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32488443", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36595026", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37706355", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39353792", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43197531", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46097036", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50563428", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54154493", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68987570", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"71580404", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"90336916", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"123555077", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30886865", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37618289", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39595876", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40473072", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46230657", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"47652070", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57088066", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59595562", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87802159", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96105926", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25015411", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42729337", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"44024121", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46047435", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59126259", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45229375", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"47642070", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50282856", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66778407", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27288955", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32329897", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42884878", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"52190906", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53108180", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72435783", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79557324", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"25311754", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29554437", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"36396364", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"45610842", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46486069", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"48290049", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50828749", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"52624133", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60277020", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"61939591", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"70971989", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93747791", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"95129362", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"36504764", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43617168", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59719043", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62617632", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"71988928", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86678313", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"21832505", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24386897", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31753576", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44092966", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62754994", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23757098", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28969594", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41183092", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58220737", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65033473", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66901363", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76567938", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"119036208", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"120735168", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"6654940", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"44963736", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47732339", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56385693", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58202214", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64323256", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65364683", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"100977872", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"117907099", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"32227988", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40404107", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"40625929", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43828951", "lang":"ko", "jira_id": "SIRT-3916"},
{"user_id":"44390067", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"46820776", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47565549", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50715900", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"72036025", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"73294347", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90116991", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32375061", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45220834", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46335929", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50539959", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"58274684", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58776307", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63878552", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76928311", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77537236", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"111549689", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"29644849", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"41029693", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"49919996", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50350322", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62795841", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65331199", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"67388953", "lang":"ko", "jira_id": "SIRT-3916"},
{"user_id":"69309435", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74244302", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"74503086", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25821013", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"27894853", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49270438", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56865590", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"62861675", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72312198", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89162516", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"109699823", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"10067459", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"21461563", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37932573", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45805726", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53163843", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"63100470", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67362470", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"70368590", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96213264", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"102700942", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43454796", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"45975289", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"51154643", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51965414", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"54731572", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57626776", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67657806", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89741010", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"141713412", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36798715", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44535423", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"45926955", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"52319171", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"66338188", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79273174", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84214410", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21575376", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37466416", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44831682", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"46885244", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50080261", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53744321", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84662000", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93564917", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"121555007", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"2767224", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40137007", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45279841", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47373379", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47706484", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50598420", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"61403382", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62066081", "lang":"sk", "jira_id": "SIRT-3916"},
{"user_id":"82497458", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"28380132", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30684173", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35045810", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43043859", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43370914", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"49815915", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51292476", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55129239", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59946974", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"67091183", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67786828", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71493309", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80417639", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81128840", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"36715612", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42714705", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44956295", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55716342", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"58226606", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92084289", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"157370999", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"2223691", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31104319", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39560902", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"43596870", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"49471957", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"50747956", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"52359922", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58748556", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68975433", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28230373", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29894188", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37071834", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48853835", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57378393", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"62669485", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75045078", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"132268230", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30570690", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37169010", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42433712", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65311054", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75079468", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"98737049", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"101546473", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23701193", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"23779605", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"28010736", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"31368970", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"35692494", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38879937", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47446843", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48608193", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"51085682", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"53773816", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55611088", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57367534", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101401510", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"103143116", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"25361510", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27374328", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40463356", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43019422", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57916219", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59728239", "lang":"hu", "jira_id": "SIRT-3916"},
{"user_id":"75018930", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79919295", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24276512", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25130818", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31321243", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32190319", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"38428072", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39587199", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"40777133", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41306517", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42302780", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"43939978", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"49210136", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54989046", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55541597", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56289497", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57309543", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65203273", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"66411215", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90295835", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27617001", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29303876", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30278353", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31864955", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45883946", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50334750", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56406105", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56799309", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60522357", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"61238077", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62827449", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"14297988", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"43128600", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43335318", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"52034641", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"53317588", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60480830", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71785889", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72091961", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"75515269", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"85978820", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"89684276", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"119271330", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40087698", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40411483", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51155152", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57759212", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66934964", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74421520", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"93583295", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"7285476", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20579116", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22517518", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63079535", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"94603326", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"6842002", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"37848706", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"39577044", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44762201", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47151752", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47764114", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52561979", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"56896393", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58277035", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62586142", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66288834", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71428712", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72272284", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95368713", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"157932151", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"161220355", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31302601", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39339833", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40785763", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42774566", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44110799", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"45538375", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46347487", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"50885938", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54762351", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62627451", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83944933", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87934493", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"88899845", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21384617", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"21926346", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37976667", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"39185963", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"50450437", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"56724093", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"58929486", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65897425", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"93851333", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"31915531", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41004806", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42149916", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51088298", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71332708", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"35610243", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40983261", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"46773670", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54673091", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56574607", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56913326", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"58242988", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69424650", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94182271", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"10091026", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24549194", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25982682", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29113478", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"40093462", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50958624", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51923265", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"54318956", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66539853", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74987631", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77961523", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"79780840", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20641959", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29864990", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"32975295", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40380543", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"48137444", "lang":"en-us", "jira_id": "SIRT-3916"},
{"user_id":"50485636", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"53641539", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55757492", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88880848", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97777713", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"17401109", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"25600344", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35880288", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57164520", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80894284", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"82332543", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"16443801", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23241225", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28040153", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29926141", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31155628", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31254038", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39423824", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49417165", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"51776200", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52983399", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"55436480", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"58633052", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63294463", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"73869930", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"78363154", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"6591856", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23521153", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35300732", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45118372", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"49606711", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50942607", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"53528727", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54173016", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54324493", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64461210", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"68719490", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"70440620", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"104782294", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29022933", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"51223121", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"56517660", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"36666259", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"44693202", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63667648", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67046631", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"76574044", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84901898", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"103353211", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"1605158", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"50246576", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60419512", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75761335", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"32384768", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"32679737", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"36618722", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37832693", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44485526", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"51852132", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59664014", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61521795", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"63534764", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"124789912", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"13832885", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28811065", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"44077933", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61746425", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62532459", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64305977", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70834843", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"124930527", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22401442", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"38289721", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"48322591", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"52359675", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52955687", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54501230", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66683514", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69346717", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"158926729", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39574001", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43527351", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44967901", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"46759277", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49744929", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50815146", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87447873", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101920165", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"903847", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"27193811", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"29501710", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43068911", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"49381384", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52721446", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82110296", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"89208271", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"5389951", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"13193409", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23764368", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28749697", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38285439", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48729798", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55820710", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63937207", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"85665478", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"112143025", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36585779", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"42104307", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43810988", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"53806452", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59237586", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"59642496", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"75367281", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38834997", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42491270", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49532506", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52056877", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"52877857", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"53980606", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54613546", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60447303", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73550308", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91256630", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46392776", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52319068", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65732787", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66684473", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"16389007", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39964480", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53311131", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"54320516", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55013781", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"66507821", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71279188", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"124080433", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"25258369", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26292819", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"27019415", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38261363", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38423499", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"57770575", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"58381444", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59262850", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"119880878", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28522505", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36394448", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51225201", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52377269", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"53462193", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"58322722", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"60931400", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70454346", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80083650", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"23594116", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39457649", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41478208", "lang":"sk", "jira_id": "SIRT-3916"},
{"user_id":"52170429", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66432819", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96845621", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"157952375", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"23726019", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28375884", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43502403", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45687661", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47274983", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"129451664", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"32507809", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"32628058", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36578902", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"48166963", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"53990234", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54802602", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"62066976", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81701856", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26009728", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"28297485", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30340801", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38164352", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"47001653", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48217807", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"49441415", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"49639533", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54721366", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"57322446", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57934844", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"72795753", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89405862", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90922895", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"92159584", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"144740532", "lang":"ja", "jira_id": "SIRT-3916"},
{"user_id":"27692826", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"39826786", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"43845692", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49680537", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"51700007", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65395108", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65647487", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74224918", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"78359882", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85448165", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29791550", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30556896", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39919554", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"63352151", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64504225", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68510359", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78571356", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"169439783", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"17352369", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30058112", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43726488", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54044616", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"58879731", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"67972302", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"69520193", "lang":"en-us", "jira_id": "SIRT-3916"},
{"user_id":"91849748", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28996008", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38019896", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44847483", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"157317839", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"36998107", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"41078444", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43819845", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"68764066", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72878139", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73687861", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89011830", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"152307273", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"18469201", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"48675354", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"68443838", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83852240", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29497124", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"43264680", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"46932767", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"54212603", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62017584", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67896970", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"78806865", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"35050562", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36201229", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38962381", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39795492", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"46751185", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58489283", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59273004", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"61658826", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"65713746", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72733405", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78748517", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"104882378", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"27361385", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"30823512", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32451182", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38614528", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40716338", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43910122", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52423821", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"53518701", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"81115046", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21738334", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"24703731", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"40702678", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"44567421", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60573929", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78507769", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29919451", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"38557473", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41119265", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43893468", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48779437", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53835295", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"59947971", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67023657", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"21431910", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27388752", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37006338", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"42379675", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"51820307", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63056472", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82935577", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"22866333", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"37967769", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43638304", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"52572996", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"53128604", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54764538", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56903223", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"59337108", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61159987", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80133642", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86527823", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"91580306", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"102991313", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30826138", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"38873536", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45029633", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"51759022", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53719220", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63547601", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83398795", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25578186", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36172662", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36318805", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"46700461", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"49726917", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49837515", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59987686", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65724110", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66137288", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"67520328", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72178874", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77034031", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"22386881", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"38066739", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"49294813", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56352710", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74229096", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24808856", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"26532819", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36511421", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39888046", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"47962448", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"48124568", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"50815886", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59186180", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"63643059", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"205008318", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29309881", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31827586", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37974838", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40625141", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59970568", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"95951593", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101904038", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"85167", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"10815652", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"11979051", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42547673", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43084467", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51852268", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65342963", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65779921", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66436976", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"69712867", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78273941", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81656129", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"103193502", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"19458982", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"23892287", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"37138173", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37620108", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"52393776", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54208531", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59775016", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101166412", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"43141442", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45532820", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52082372", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52617914", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53380499", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55003026", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"58623716", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"112976440", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"144760649", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"29335573", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29721751", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44816129", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95754944", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"5123", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31618338", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40526839", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"48931040", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"49872169", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59308430", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61196757", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"69765997", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84057034", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"18351748", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"36841405", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38931449", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44338222", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44395168", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"48406701", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54611383", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57410122", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68293846", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"77208443", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"81065987", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"21390470", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35980760", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36698261", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37730025", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38711303", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"41551691", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"134645498", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"26021133", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36503023", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39073109", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42263870", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43349682", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44061006", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"47621062", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"52321713", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"55042123", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67017834", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70733822", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75023664", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"13313812", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23987024", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37014214", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41116814", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42788936", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"43199383", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"54442432", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63976768", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66953128", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"19877307", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36090855", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"37658156", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44496119", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46588937", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"59117089", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"61421755", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"61541439", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"6149962", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"6659152", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32593468", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42231002", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48953073", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50864032", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"57086280", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64390837", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"86483670", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95284046", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"136276983", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30093474", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30430851", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31470580", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35664178", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39451515", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44343661", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44967566", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47957281", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"50295994", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24689858", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42066675", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44230571", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"52968416", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"58533940", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67719371", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"68757911", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74373409", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"20165147", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32712563", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38818226", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40327935", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41584667", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45767627", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"50084000", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"60757333", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80409321", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36865813", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37217506", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37888033", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"45685871", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49718290", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51546474", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55126481", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"55980026", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61259637", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64171266", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"107114905", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"129615084", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"20861797", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"28293628", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28754205", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36950535", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37046288", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"38438078", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45482339", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52239646", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55971492", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56441027", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"68667617", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"7505167", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"33030624", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"38477193", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"43695515", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45226865", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46136100", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49703351", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69722220", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79131891", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"115185603", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"31175030", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"41526360", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44324199", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44338537", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45787514", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53135896", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54475952", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"58085038", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85576824", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27187962", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"29843249", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"40538106", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41640287", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48020731", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54215875", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54616038", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67003745", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"71087430", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72348161", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85913218", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"87880287", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25717656", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"30011711", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52505815", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61270610", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72447830", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72767487", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35599629", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46579328", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50256222", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"51056468", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"52123301", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"59301300", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61223767", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96913269", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"133963928", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"17085869", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"35075605", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35095077", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"39705354", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41537576", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44112495", "lang":"hu", "jira_id": "SIRT-3916"},
{"user_id":"45053162", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"45652535", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49784093", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51735099", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"58139224", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58732435", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"112683140", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21826246", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"27513941", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27904816", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36008059", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39588100", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"40346617", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43252130", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55266361", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51273018", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54529647", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"57988781", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"64653741", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65789344", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93610037", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"136992038", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"29908114", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31382176", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42269129", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44483511", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"48805958", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51480030", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55087297", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63989257", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"80424485", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"328190", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"12238248", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31980959", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32248731", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38217064", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47492265", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50314547", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50719694", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51353492", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"54153705", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62825841", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63110213", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70526540", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72940294", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75485130", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76414221", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"78154232", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"27986682", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32221811", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"37706983", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44736078", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"46665085", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59363225", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70135589", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"75427256", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31557869", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35898953", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38219433", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43066856", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43376898", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74640371", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86002237", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88353821", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"93669613", "lang":"th", "jira_id": "SIRT-3916"},
{"user_id":"19164607", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22309264", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"31452198", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"36537191", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"37922130", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38868762", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"60488181", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61918609", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"63453971", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74065784", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82068080", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26792592", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27135472", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40249124", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42930224", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43765263", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44216722", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"47634857", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"52057956", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"58307275", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"110047668", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"115458923", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"19384987", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30869124", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53007579", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71713666", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"84071807", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"117346781", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"26878327", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"31303395", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40085820", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"43411412", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48171902", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53562361", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"54160313", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58330198", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59655344", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71089294", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21037864", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36073633", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41122083", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43408913", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67152365", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85204523", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101696683", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"120777509", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25899952", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"32692084", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39482105", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44800867", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"50061668", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"50536946", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64910559", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95608983", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"135117526", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28127608", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37606624", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42878939", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61786552", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69681556", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89913406", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26745431", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"29407232", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"41212108", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44473716", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45088052", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53694293", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58192989", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60660894", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62780751", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71537607", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"157623732", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"29893875", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"32267199", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36491103", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37790399", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"39213607", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"46666688", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51256101", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"54211782", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57429587", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74113905", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31301121", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"41275397", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52492968", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53889083", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68994199", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"9257643", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25240659", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"46996750", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"50633138", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"25007683", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27498948", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29202922", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32660448", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"32672998", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36445457", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40528880", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"52392431", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69330013", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"87000971", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"11819690", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"21917335", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37057763", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52544330", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54380223", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57455761", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65288665", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70265650", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"11400438", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23826596", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23999994", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31382313", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39723115", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58039921", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"67475290", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72327760", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77236706", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85949762", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"87031189", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29297868", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"31607523", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"40003351", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46288856", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50207690", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54603113", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"17595424", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53163706", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57025731", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66084577", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85963154", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"32167816", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"34484719", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"37184522", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"38693915", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49946498", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55313230", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80276163", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"174772455", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23676948", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"47219972", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"79914537", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80677051", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"111553331", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"143509081", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"26338826", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29158381", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29992084", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30465480", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36278654", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"42384461", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44462309", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47770250", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56001080", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57151098", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"59621686", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63518857", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77231010", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88789705", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"99730839", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"121843440", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36990689", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42481140", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46142553", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"50020312", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50354531", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69077225", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"133981502", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30340662", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36200936", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"36259246", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"48379932", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66707175", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70696399", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91692937", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97307986", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"111695924", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"134590768", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"136413", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22261090", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31837289", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32569500", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"48289890", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55627124", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56857074", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59227155", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"62387392", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"134424589", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"32458208", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42765796", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"44577636", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"46316341", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"60672003", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62764387", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73825224", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74698916", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24671626", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"31374833", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36133741", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"40429847", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40532754", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"41311721", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41975011", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42383664", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57497856", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"59709702", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64620125", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"79716298", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"31537114", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32986644", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37611860", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"44244260", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"44577018", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"46987682", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"49792279", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51020763", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62947176", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65030695", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66207897", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"492793", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45892899", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62899201", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69776738", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76766500", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"84323900", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"89633293", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"31344568", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31692568", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35915827", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"38583666", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39288249", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41521068", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"47944827", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62877859", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68573760", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68599954", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68981535", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75827325", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"28482922", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38666881", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40531535", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"51022189", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53789203", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"127041579", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21853109", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26504754", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38365951", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43644220", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54010636", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55200428", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60043271", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62224743", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62509016", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63226249", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89716285", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24514734", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41161011", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47854095", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"51412372", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54353828", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79206046", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"16415075", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23390233", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30058522", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39211904", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44653513", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59036759", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"60475516", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76926834", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80744319", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"124005836", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36250622", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36412929", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39640292", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41839411", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"45144989", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"45161975", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45992608", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57326938", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63266769", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67130805", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"79612598", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"142297436", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"153563380", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22007596", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29959567", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39575929", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41845639", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52252552", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58293623", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89026688", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"113329463", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"115468840", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"144875851", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"25143054", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"29129865", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29700364", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"32234674", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"40055466", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40282698", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42986787", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45544501", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"4119052", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"13550776", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"30927686", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42061303", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53266805", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70884874", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"33711935", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42121494", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45012226", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46025627", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"118448764", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"145317518", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27528432", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"41946402", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"52329010", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52613550", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56597340", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58522440", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70836604", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72713437", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55910607", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59957050", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64384438", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"39020397", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"49626245", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57086330", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57945685", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58091933", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67563479", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"83275701", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"15964054", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"30975875", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36632949", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42157787", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52733287", "lang":"sk", "jira_id": "SIRT-3916"},
{"user_id":"60148832", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62251729", "lang":"zh-cn", "jira_id": "SIRT-3916"},
{"user_id":"66419848", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"111646971", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"25467016", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"26857029", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"30202083", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32199342", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38981760", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47486805", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78566608", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"12635767", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26746130", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29469109", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"31330131", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38196285", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42766916", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44145133", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"44393953", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"46269893", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"46651990", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69373296", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71285571", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"74803888", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94315120", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"103055859", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"112888579", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"142948040", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37947710", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"38828936", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40889355", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47880094", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"48981200", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36726553", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36886494", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43401654", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"49257829", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"51090604", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51857490", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60633203", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75817815", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"100998918", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22834143", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43253423", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50971248", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76711566", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"23415728", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"26144483", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29194769", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38690793", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41994624", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"46186344", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"46765907", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"50858575", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52786155", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"53404830", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"62441782", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64839629", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78755469", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95328576", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"17129626", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39261587", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41147692", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43253017", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49955757", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60562709", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63667763", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40647757", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"41632891", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"42931361", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"44111439", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50091689", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"52879765", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"76429949", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"123221014", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35877132", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39111954", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48008430", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"48080893", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59347529", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86408423", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"38445597", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40250854", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44733709", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46064025", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46438270", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61596543", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"68408265", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"73541022", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"117318933", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"164867", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39801058", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40481503", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43353500", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"44386625", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48712963", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53415405", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62268025", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"65976940", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21038428", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30538336", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37694853", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"47848021", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50076886", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53325912", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70890562", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"75950698", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20718223", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30535591", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44089447", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46006523", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"46148089", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53678818", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54626373", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"57847639", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"63817041", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63956826", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67392437", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76348229", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"114502220", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"4611488", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25628592", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30174127", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"38619881", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40672570", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45289700", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49652782", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53824942", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63286192", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78207822", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"122537506", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22350907", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"32629097", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47504546", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47705308", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53788418", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54454315", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55522823", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58829261", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62938877", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"75869774", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38961964", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40328180", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54545190", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22109969", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"26727894", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32430714", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"38459288", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"42035841", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51833787", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"89248240", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"90612296", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91157112", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37329746", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39596792", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41514121", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41845999", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"44768847", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46113489", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91322281", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"132388103", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"30896965", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"39270091", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44621936", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47587489", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49448574", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50490529", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"50911978", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"63007105", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64250839", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66884319", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74856932", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80414207", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"85995241", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"122103283", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"134193032", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"49194872", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24540193", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30204750", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29482666", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62115551", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"63413403", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"120573745", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"45031387", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"63785223", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80873359", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"142320392", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31819131", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65764272", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53145783", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26942301", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45739571", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"47042126", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27282926", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21931133", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39918984", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60105387", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22583926", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49861697", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"98200713", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25647533", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"56458038", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71261828", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36724143", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49592223", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"82770849", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"105262316", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"121140862", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26869778", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"36474834", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88095251", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29653032", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"47002867", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52067168", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"43702945", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27762014", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38439944", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"473396", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29126162", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32826263", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46185958", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62835394", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60406933", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60890325", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"36625020", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36938623", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41531248", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"29617931", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21295004", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59834403", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"70936643", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70607390", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78577605", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"137341357", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"31565647", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52903685", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55253962", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"61007570", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28274082", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45379816", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27230451", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"37668867", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"57152833", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78378177", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39799471", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"60165089", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22973818", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29736884", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31368372", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"154914226", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"45094553", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"55081413", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25147530", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62178548", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"26190139", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82856208", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56490087", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72214956", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"34064632", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25921015", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"38162637", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"52883380", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"9568922", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47940485", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"44328954", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60622963", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"20833613", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23576598", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49021466", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36724634", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52578190", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56328162", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79733505", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"27913340", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"92927904", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"132241782", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"30144915", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39125473", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39070271", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40066436", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"62319707", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52052657", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54863432", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47405747", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63606272", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94267175", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"105670021", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"43516698", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"152004510", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"152812146", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"131142012", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"69500891", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40498235", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42783540", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"45662343", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64081037", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51204124", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44926005", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38418856", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38789731", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53241586", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66614688", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24270443", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"27474662", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"60097118", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50004679", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27855440", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45843741", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25057190", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61110870", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91787022", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"143603255", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"614394", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37880253", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"36254486", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42022680", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59910572", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"67263266", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"29159178", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58450280", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41927892", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49124442", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49041631", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50805501", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64831342", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68299691", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"79493285", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55425054", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"111019286", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52233274", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31461391", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"114433912", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45907103", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"48394404", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60176035", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52180689", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"99696886", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63834072", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"118372277", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"2338479", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49517758", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"52925091", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25141752", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35860916", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55898022", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"44970298", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46000809", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"49086782", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95897742", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"29530640", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"43383619", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39546731", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65192082", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"74992314", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77920359", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47878289", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"54485787", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45151981", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"146921353", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"44290087", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"41477461", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50957788", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53266702", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26227724", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45874057", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"152942812", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"161779945", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"59155999", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40207962", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"68601559", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22350012", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68083492", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"102447358", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"29857551", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31794289", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"35086696", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"51252501", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58022605", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22932685", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25534711", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61170606", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44186733", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60652790", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"61814352", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79254500", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"921585", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60722602", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48907548", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"65286734", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29003599", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78694914", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"65050259", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"71581086", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71750562", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"80731605", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37902650", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51920657", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"54972821", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56631377", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57875842", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57658688", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"24134839", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"42196084", "lang":"hu", "jira_id": "SIRT-3916"},
{"user_id":"136425752", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27888415", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29011572", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"41318878", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"74438634", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41353192", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57806536", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28374200", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36966312", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"61759152", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77089852", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"5690948", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29787406", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"7891017", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31165342", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50122159", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64467764", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42366695", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"77588893", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57512890", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36375104", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51815168", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"91491448", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27039466", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"43431874", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"67813184", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72925737", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"132098869", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"32080485", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"36790434", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29478945", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47715475", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"152987597", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"70314276", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92629178", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43664665", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"47014008", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36509728", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37334385", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53220859", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"48725578", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55689824", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25492880", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"47009953", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"51246328", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40336240", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"48509358", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88657735", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51335020", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"102229697", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36099348", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36599216", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32092099", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48064468", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57865089", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58474286", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45875400", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52289700", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"59544052", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"154417941", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64248240", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68900181", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90978745", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37912012", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39067018", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"78331837", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49927504", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74427777", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51483255", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"85959140", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"63671931", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39990054", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50961550", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55124599", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56062774", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43536728", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81697126", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30782393", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50837078", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"61199878", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"63667596", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58060519", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"72140803", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96224146", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"36647766", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"108755177", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25148769", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51418567", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52266906", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"104435562", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"27239319", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38162204", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80624679", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"81200171", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31980935", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"47459761", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39657886", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47648787", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45022836", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"62713689", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40193222", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68362545", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"136520024", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48935401", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56961306", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74458850", "lang":"ja", "jira_id": "SIRT-3916"},
{"user_id":"27626604", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28045739", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39095820", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65357267", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74172363", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78641609", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84237450", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"30143082", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35924164", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"43428679", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63909589", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91433897", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"38114186", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"85528419", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39815403", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43608052", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"14409324", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"32624908", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57957773", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37011861", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45292257", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38084694", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"41583160", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"52022019", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56900522", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"62635917", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68814570", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87721907", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"36000129", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37677641", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75118107", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81359054", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40924500", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"48770496", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23473656", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36262538", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55407413", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78455574", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"73407775", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"28618583", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36823260", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58629351", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"144906191", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"49956132", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57796879", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58803830", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"39317849", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"113941874", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"172070107", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"6455482", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45263688", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73465412", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89848897", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45810111", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"46022044", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"16980547", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27236722", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"73760640", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"49887638", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45007859", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52174360", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40077635", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"46137890", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39897806", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52059086", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"39927831", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53587571", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67339522", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"1845263", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49333353", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"52980453", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59178889", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"66897642", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90676927", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21534457", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"58157101", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"15936304", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25847949", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"30850104", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63544500", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27917797", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31424713", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84751370", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"13592500", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36305533", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72948875", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27381857", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39042878", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38053557", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22343314", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89985559", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"47632384", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54415616", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"113859160", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"31492900", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37586920", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"50904808", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"54768272", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"56536858", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72171787", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"73366487", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"41456199", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41610787", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"110667908", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"144538024", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51418071", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54355558", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25501230", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"32192372", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"27096538", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39649670", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66780064", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35077780", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55288295", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39815365", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"44363533", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48971902", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65217002", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"23497029", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31339081", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41925551", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81238238", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67259064", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86197071", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"134903339", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41211183", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47220129", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54414934", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56263239", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59181947", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"59074431", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"85712042", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45338899", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"97693348", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23548184", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"71405671", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60142131", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63532168", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40643739", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"82885872", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25583830", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41234684", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23708070", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35687686", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"9563658", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"100526878", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41355309", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21013799", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51525093", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"153295638", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37993514", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28994265", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36590639", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"65560166", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82897245", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"91107852", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"23870277", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45797148", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48286022", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"2128782", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26583534", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40114509", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43876074", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61944985", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40918065", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"123241872", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50093797", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"48661792", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36672185", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64345341", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38763482", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47819270", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57628608", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"73043067", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"24749972", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30584504", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42021135", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41819495", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58672415", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43146803", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"58459429", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"62071294", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55861194", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28051426", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"40091726", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55261322", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20676280", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"29140617", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58420177", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"46670379", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73013125", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"146124246", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68386044", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"51441294", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64816475", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37655319", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"90520927", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48994260", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37383975", "lang":"hu", "jira_id": "SIRT-3916"},
{"user_id":"14125332", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39392583", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35669413", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46562628", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"52673631", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62202339", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50645644", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"83221877", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"12892962", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20888852", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48090174", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23435589", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"24154789", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68885340", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28094779", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39088596", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58903326", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37470393", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"71874997", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72851418", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"42546512", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80262996", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51274522", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52280587", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"135889436", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"137288344", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51290223", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84749411", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"37253982", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53565768", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64320110", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69222217", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"51099881", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69775219", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52221091", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51597897", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"136055773", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"51060676", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73684074", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28231779", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42442017", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32921101", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"62565370", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"10754", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27075176", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40923619", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87369002", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"109380178", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31737436", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38661584", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88677614", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49249609", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"93402860", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"54159368", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57239384", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59872880", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69365394", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"46666957", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74217500", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39023862", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56178054", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29119486", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46786592", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48569495", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66065788", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72570683", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23807135", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28759504", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57535603", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41479206", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46241872", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38262358", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40716742", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40821649", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56256576", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75300386", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"14671319", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31469671", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"32694613", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"35835591", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57642765", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"40214663", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"42628053", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45793804", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26196139", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21517999", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54399915", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58726958", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43279984", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"77035838", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29660366", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"37769252", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"56585060", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"35843390", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36324135", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67200917", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"146878312", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"16225411", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56224180", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30108319", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50328865", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60140970", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24831133", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46692760", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46997249", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"71370820", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78311430", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"147762090", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45708234", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84810316", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"60227713", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59598606", "lang":"zh-cn", "jira_id": "SIRT-3916"},
{"user_id":"41524999", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82717929", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"5023547", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56634537", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"7366932", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"76211979", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35681930", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42609972", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"112369291", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53904043", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54643498", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58464374", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62262830", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"17898789", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91592288", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"133449901", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49563008", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54127935", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"58590666", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62123272", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44916646", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54838681", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"149222949", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"38730101", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"97260170", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56331105", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25733177", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"11152602", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44331891", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"23461102", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30944163", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22452136", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"78138889", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"26901444", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40561505", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"47578243", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"51052302", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54554044", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60460445", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"13163461", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"31837604", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63891891", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23786064", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"52146983", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55650933", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"114685937", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28624581", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44934854", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49342188", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36949801", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"53621996", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"31557216", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42649947", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68489058", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39922384", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"51738692", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"19683544", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27621153", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"40323999", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"57388084", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28168431", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"52160263", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"55113294", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46567437", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"37268006", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36359200", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48798945", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"49996521", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"32566842", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"53060537", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55439458", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25037698", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50787915", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45999012", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55986524", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72096794", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101955762", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"45468154", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51081644", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"31125501", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50069723", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"62192400", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75046061", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"4991609", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75230612", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38266265", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56989897", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32281329", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40444956", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"50565754", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58652395", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90211133", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"64159500", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"41826903", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"66222984", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36686556", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64219931", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"38259425", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75155299", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25879363", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59283970", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"50420963", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"11524649", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"29315319", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"43007289", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47866559", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"51846220", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44410137", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54817135", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"62900158", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23522881", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37785131", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45471987", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26472489", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51267793", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44815829", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62157597", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77690218", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"160654314", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38833745", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29823254", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63692631", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"40048608", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"105826023", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56779562", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"29289784", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31664476", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38515113", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29285735", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43755351", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54182239", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"59454122", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29202379", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"40094031", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"52969767", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"27878940", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36120105", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26712985", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21098561", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39114541", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41982101", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49186178", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"57735520", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"94191978", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"47134035", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"28668191", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30297435", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32212973", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41687787", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"46077653", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30283283", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"38161578", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80683910", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"153677800", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"54070901", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64532851", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37392471", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44578210", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"38510403", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44393147", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74208443", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"45066774", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61694409", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41479421", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52969741", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"81298211", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"7188795", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50520243", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"966460", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29735729", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42214935", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65716481", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40631316", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"111639383", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"68006995", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"34406129", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40814070", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58894605", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"36523035", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40609781", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"40845897", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23295301", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66601652", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"107537873", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63244028", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42941398", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"65777956", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95238150", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32985240", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43445044", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25531339", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"62384716", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30661033", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44365075", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89848676", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50110112", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"52376996", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46473345", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77820803", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53798747", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"42052568", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21799581", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43131877", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69433303", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80827137", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42050156", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"79787770", "lang":"th", "jira_id": "SIRT-3916"},
{"user_id":"134048978", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39721692", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44184340", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"63763886", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71066501", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76077685", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73292029", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37651297", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46121787", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68824275", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45412253", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101109698", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24347789", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38525425", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"68051549", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39139258", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39776179", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"46608555", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"67887464", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63687192", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"83393513", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"90953233", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"147212779", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44154558", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46504652", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20993498", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27756147", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27772972", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37407015", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91415238", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"62813165", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"44932814", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37972878", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47289967", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"89077758", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93002867", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35097779", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51661241", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74618858", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47530493", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49457435", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"85510354", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85513703", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"111125219", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44615474", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40853996", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50526040", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43450194", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38784785", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62020028", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"94408137", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"40211538", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"147848668", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"162132466", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"51870554", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55453653", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"98241265", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"58121050", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56105444", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"63794563", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36171461", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40159293", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45651336", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"134631930", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"56104959", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49814522", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59157171", "lang":"ja", "jira_id": "SIRT-3916"},
{"user_id":"61292850", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65564148", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85569457", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27802484", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45249966", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35782417", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53025113", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"61764056", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51676909", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48068025", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"99378472", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"43706695", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39441632", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"42524716", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"21435646", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46733809", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38252136", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75250001", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91545745", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26689949", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"30811881", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38752844", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"62021065", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69159153", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43188766", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51196123", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24116775", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28002242", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28152440", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41699970", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"45877376", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"55141391", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"47016689", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39820125", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"76773863", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27949137", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58479102", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48883064", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38551428", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"53831525", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66430939", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48509463", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"89165416", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27979360", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32187216", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36138196", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"53534557", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"74647786", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79844044", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31556883", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72375310", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72140129", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39791074", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71656715", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90101803", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61143607", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37677572", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76010277", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58711158", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21389053", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43652912", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"26964510", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"80631075", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"37004781", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"58275365", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"31725133", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62689262", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"157812373", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"17785930", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68236564", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"86387663", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39020737", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"62515007", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"110311966", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"35849262", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48655611", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86250518", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48792671", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54934793", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60299557", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54603834", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"114068733", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"30030212", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"91631563", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28252159", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40712693", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"32618814", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40094401", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43368432", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"144604994", "lang":"ja", "jira_id": "SIRT-3916"},
{"user_id":"114206738", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28319753", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"54444883", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"142515458", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93797665", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"26459510", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26584984", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62837052", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37793103", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"44783748", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59654450", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29584205", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"45001559", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46419413", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"46569178", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"30080660", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"137319166", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21956159", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"54144375", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51388964", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92946510", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"31213330", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56385814", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45693071", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"47605593", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40883644", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50860977", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"99402918", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"35970116", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48816734", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72021878", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50270016", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"58998655", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59262498", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"111656847", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26404636", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48564490", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50005421", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"91171732", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"152761462", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"31723226", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74653912", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"103907256", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"114305865", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56866293", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"10366864", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31064719", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"103154646", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42702467", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51333878", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31483165", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"100438963", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"37957151", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52034865", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"55472047", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96442691", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"51831867", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37524676", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28502580", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37207452", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"6307389", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62693130", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45471929", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51694017", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"118970121", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"28861556", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"27584091", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36482384", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44872073", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32008248", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"32368452", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"60097383", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77250404", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44836465", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51623710", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"41187099", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26582966", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41610821", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48905937", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60424719", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"31168860", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"71953757", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49331161", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"82481331", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"35862854", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"37511695", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39110898", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"65239753", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29731095", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57140431", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80902544", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"39244429", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45458090", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"166623303", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"29273855", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58974461", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22480377", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85499821", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31586081", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53482251", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40328746", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51110501", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"137333777", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"37109194", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29025586", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72043264", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"23608195", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"25590253", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39393623", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51893077", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64715776", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29621034", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"59628526", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61359978", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"31697938", "lang":"hu", "jira_id": "SIRT-3916"},
{"user_id":"41096569", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73065195", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35229420", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44621458", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66833680", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45606210", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56280951", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26991127", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39577001", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54361207", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"90610255", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36971673", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66798834", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44154311", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92082485", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38723074", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45560753", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71083749", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"8017989", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"59497527", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80134591", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80182088", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"55547023", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"36457323", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56712394", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21332028", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93146679", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"23397590", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"24086697", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28834300", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"37154279", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39324720", "lang":"ko", "jira_id": "SIRT-3916"},
{"user_id":"42597914", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48069455", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62533125", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"20248706", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31705294", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"140099266", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39213127", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29244446", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38506854", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"16877026", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45828845", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80789302", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"45720889", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61827911", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38695414", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47277567", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51433205", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"38388807", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22787452", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75712893", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"76069291", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39918661", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"72568199", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42174353", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65201061", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26080116", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"47187788", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51989286", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54634983", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47170270", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"472079", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24763578", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43813578", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22772726", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52325321", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62170966", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"65506625", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"114974136", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"29671467", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40393228", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"98246708", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21922561", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31052954", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54960789", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"53695787", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67621873", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"73097942", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"67510814", "lang":"sk", "jira_id": "SIRT-3916"},
{"user_id":"124101140", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25031306", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"146106840", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46727253", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37099688", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48487214", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"78044783", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67416099", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"70276919", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"22694812", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54301670", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48130842", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"48660027", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50835257", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40398902", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47386413", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60752373", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61472334", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"66044360", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69232675", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64504818", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37033903", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"106125347", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"112151906", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51655587", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44337862", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96263283", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43934558", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64397516", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38117112", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"57147569", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63457700", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"38066362", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42789331", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"56640329", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"54758167", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57298852", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"63321379", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50091296", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29522037", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47895476", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"61358694", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70692276", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"43142998", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"27809948", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76443613", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53789569", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"79741963", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"38427623", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56821808", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"86622327", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54657541", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26642223", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38245306", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67883798", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"85272288", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93263540", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46726372", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51245091", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61767838", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43351949", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"132143327", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"27229980", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"38248210", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77364372", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68001417", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"26005367", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37485041", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49380035", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73840282", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37326553", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65315823", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"49555187", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51427034", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"68627896", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"30200982", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39346925", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67138135", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"84569419", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38400823", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"54089345", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"61328412", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"68624541", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"26776830", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55457106", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"128096244", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52124428", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58017759", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74531078", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"30994103", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41231098", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65638766", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73596022", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"63791354", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56798896", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43645906", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"13095396", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49348999", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37332781", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"54566157", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44892066", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"85756815", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"23131667", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95488123", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"47374593", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48168224", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87939112", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"17935738", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20381277", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84641605", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"31985780", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27335718", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39580110", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"85395340", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46303574", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"31251842", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36047650", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54602190", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28205787", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42798193", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60637153", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"135296970", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"28843418", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63591762", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67692896", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45941398", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42950121", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78200514", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36330751", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47198196", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27403477", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"25306951", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38884820", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"72903219", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73439392", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"137493935", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31514066", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42041380", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"55214836", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83064485", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"32099757", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"41982731", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44707459", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35634447", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"38108639", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63530639", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"44960180", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"17975979", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22902567", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24025192", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50360750", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"48571059", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37385873", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40540704", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62382978", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74958358", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69414106", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39646599", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66820908", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28134978", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"30923466", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22582337", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"35760247", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"52041931", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25077819", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49994591", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58661576", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77891859", "lang":"es-mx", "jira_id": "SIRT-3916"},
{"user_id":"30274426", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38379293", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62025682", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"135559124", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"135648277", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21529222", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"48463632", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45949077", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41546661", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"43145911", "lang":"ar", "jira_id": "SIRT-3916"},
{"user_id":"54654573", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36267328", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36327846", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77566877", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"10983043", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49317338", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39744668", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"48615030", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"40961249", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"82286643", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32800936", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41253001", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44333816", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29297972", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"37674166", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"63594659", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"100131949", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"55754083", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29869768", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101283361", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39537163", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"81416701", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28868142", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"49720725", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62072088", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"18421254", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"63383264", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"126227251", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60917582", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44967292", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"102201823", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"37347846", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"5548436", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"28583282", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58470701", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59020624", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"154328831", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"45636509", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"46634979", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70065138", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"19482242", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40060353", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59160958", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"41473266", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"44258557", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30985390", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"62314683", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"79866518", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"115421429", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60849303", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80824190", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21802544", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"31601842", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57677065", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"86340096", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31432463", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47805777", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58086183", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30564986", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"12875057", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"40545141", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"64518856", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73395804", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"37613440", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"47843630", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50085693", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78072629", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91885393", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40724707", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"14847828", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27895592", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42269580", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"55138855", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73154104", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"68081187", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"111029660", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"20693004", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26659088", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51037581", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36942570", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40936330", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73690035", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36359856", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42613283", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55689624", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31570112", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"79263341", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"36928606", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"127942256", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"36914476", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"93630589", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47199411", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35833016", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"52567696", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30820152", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"52784363", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"38516051", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"22719016", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30946646", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78046701", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"21991090", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"27243963", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"36517927", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39217206", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50106286", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54309017", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"114342197", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23309431", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64448318", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"72454098", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81406933", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48699603", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51417693", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"15697166", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"25766032", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42728050", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52386983", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"54654321", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55187116", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"144635860", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"59349058", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31896084", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58173316", "lang":"en-us", "jira_id": "SIRT-3916"},
{"user_id":"58286563", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35043466", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"36806486", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"48849241", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36029255", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60656545", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"37669926", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51311926", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"80635790", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"25246541", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"32100762", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41969724", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43930764", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48657225", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52143580", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74409691", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"16824228", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41788595", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71842823", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21672866", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40616281", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46021592", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57062207", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66915629", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63487152", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"32809565", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48793697", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50633060", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"53738211", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58427248", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62268263", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42124980", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60158394", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71141483", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29694211", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38099781", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"38197393", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49780481", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"71854455", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"29150320", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42476213", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"67020025", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30522961", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"32622555", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39151567", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"58231808", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"142916814", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27155503", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"136263052", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44390048", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45210827", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60685213", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88083374", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"60627706", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24499175", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55692172", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58463185", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29370075", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75218184", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"59626261", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"9066496", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25273188", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50813570", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28732370", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60037314", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70321867", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"119777929", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"126072307", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"1491925", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49054463", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54747202", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"60078668", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61791160", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"45454146", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68167432", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"9687759", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26692942", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70762161", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29032367", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32954018", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"54717769", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55385391", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67063112", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88666142", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59778258", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"59929805", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61632339", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48841346", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52629809", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35619253", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42622886", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28073728", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"56044198", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37592922", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"14185514", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32712193", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31778135", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57626930", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35062595", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"40134083", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29703679", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"49081562", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61910088", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"80133156", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58927331", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45604516", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"46604549", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"31316207", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35665925", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39309945", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"50744753", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"124049615", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30539430", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44897343", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46287129", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54964703", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36645929", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69959502", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53113333", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"8439338", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25904933", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48494321", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23406701", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54716819", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"6685275", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43309047", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55507569", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58667531", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81618083", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"50807767", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70291193", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36477004", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"50756843", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54167713", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74595779", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"32963537", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63158494", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32666641", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46012597", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71356429", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27877716", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59680262", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68889821", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88914415", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37825596", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"30889474", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57277910", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65394540", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29251649", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"38851789", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"48163049", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"79610134", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88907318", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28064707", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68688290", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53822842", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26767579", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"32904790", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35617521", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35044089", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37474540", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"50915720", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31067382", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"31231130", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40680267", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49443275", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26485145", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52224211", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"92372244", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61243780", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"49822927", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45697335", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21841789", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32295204", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55760451", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25476928", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27286027", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64146127", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"156365307", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"36296589", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39850245", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"43878556", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"74105236", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"89973783", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54221727", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43515277", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"11198364", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40956379", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58150183", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"36456851", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60363944", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29418120", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67159605", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"52236283", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"133356668", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"29836516", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"28038576", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37665882", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54933969", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90099800", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"58988178", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82392399", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"61974931", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"64671119", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44115013", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54573289", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"36419749", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41786004", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45631773", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"42584189", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83399952", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"44202791", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38076862", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82947607", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"133435515", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48096505", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56051242", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28672681", "lang":"hu", "jira_id": "SIRT-3916"},
{"user_id":"32698831", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55433957", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82381880", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"62200249", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"66597990", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"24931727", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71282952", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"129558340", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"32551961", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41778823", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54467427", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67416474", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56175519", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28807670", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41463522", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47206228", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"55211121", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"67273477", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"142487599", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"25478709", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"27520338", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54943576", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"91240195", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68976536", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77923068", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46029849", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"30187632", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32218053", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48523046", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51762594", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60097989", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38750796", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"40331623", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45053039", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58731970", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"43965811", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"144812430", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"36618040", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41209890", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52780272", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30694797", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47447196", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49839515", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"49904597", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73864981", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"41753801", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46220080", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"114505935", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36636596", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"37645703", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"54787285", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91745103", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"30134766", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30910695", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65894562", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"91646426", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"3668560", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45945239", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35723820", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37967725", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43350377", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47792719", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"51281986", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51289010", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56148610", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"14089813", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38763608", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53911044", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26029049", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"7030126", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46288362", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28194376", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51897614", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62476697", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"73566990", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"135281268", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43815935", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"98177482", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"110065486", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31298665", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"22720027", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62834621", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38923526", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58106019", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71147541", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"81974731", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"54676525", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72272202", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84287102", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54807477", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"44263967", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29159207", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57538886", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36831129", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37763190", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"54423462", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61062393", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"116351049", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67898462", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52961457", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53275603", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23291717", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36045589", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"79836537", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32252725", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"77168469", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"24824514", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29410948", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"52103932", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"158952053", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"28895649", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"53157840", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"45708168", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"40761721", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44680102", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"92022486", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32102746", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"71986366", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20045574", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71910151", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"38921163", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"47618439", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54600005", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"67965898", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38863150", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"123162780", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"505017", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"13279462", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54553169", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59607299", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"136738604", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47601600", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94696204", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"42230049", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45336655", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"8320362", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63518502", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55374172", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61326326", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76465019", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49883763", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"52673103", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55675195", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80138048", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"83355950", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"12225127", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39619703", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"49996267", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57925882", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41149715", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"69036402", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"41965618", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56685119", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42613886", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"76503356", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"113291338", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"64788017", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"25329229", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"22031595", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72183988", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"72945252", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37159461", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78569381", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80221442", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42641739", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85262831", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"83839992", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50556608", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"19039084", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32639684", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"70904712", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44511619", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56116032", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61193345", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"61636339", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"53228752", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"79292253", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26011969", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41728487", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"18533596", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"32694816", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51446417", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"40207583", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"101448627", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35040115", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"96410032", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"96766208", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21369127", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27574942", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31977437", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50353874", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"58210745", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"58944946", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"26148912", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29079542", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29571852", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55541548", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68243649", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45737430", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44871595", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"43959617", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"44061802", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"80491182", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"49658617", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38708489", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63383070", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"147263263", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"41662751", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71211261", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"37179987", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67300598", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76174007", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51384132", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35981281", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51503826", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74180781", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27979087", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"65182350", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"94784185", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"110560013", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"123555135", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32092670", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36049878", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"37360247", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"116308415", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"40967776", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31966424", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55976364", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"77729443", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"144511392", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"12552879", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"66623123", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"125204264", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37080694", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41807419", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49593588", "lang":"bg", "jira_id": "SIRT-3916"},
{"user_id":"56573609", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30546412", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81443204", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"36977439", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32734001", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54135998", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56572851", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"44271035", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26047945", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26339526", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28481422", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"144140969", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"529578", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35403166", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"41408563", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"99325914", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70072594", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30976401", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31455057", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"48179756", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45689997", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59742036", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"61947046", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39928205", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36116870", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60918822", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28914510", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44741426", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82599589", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32429498", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39458027", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"4239239", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57517190", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"135729464", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37191269", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"121188654", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36696338", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"67041384", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25326788", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"80533904", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55898523", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"14157050", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23130030", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"31167624", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54386894", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22410390", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38082716", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"10648984", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43747649", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"53633109", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57868163", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58486016", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"134234472", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70343384", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84858550", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24389801", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32651413", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51244489", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"35994069", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"98749732", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43755954", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61248627", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67565441", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"115002586", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"30978503", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"58892365", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"73194509", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"101793899", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37388930", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"49890268", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58530188", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94447730", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44547240", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69079955", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"36004609", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"51806585", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45622696", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43976398", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"134130405", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91476634", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51373029", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68423626", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43569360", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"91289275", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52016660", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71336917", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"41382045", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67346029", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32036706", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37922941", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62224129", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65351462", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"12292934", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38693048", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"72834918", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"67062927", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"6409448", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51994090", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21326131", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25152893", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35034115", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"42425635", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"50155852", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39537774", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26651189", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"39249545", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55262092", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75961142", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36173712", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38075924", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"37038427", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48256390", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"59225385", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"44781195", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"22291444", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40861256", "lang":"zh-cn", "jira_id": "SIRT-3916"},
{"user_id":"28566702", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101935543", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26772740", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"59074958", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"1729976", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"26513896", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39862585", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45321522", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53552720", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"54399330", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"136990924", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"50648763", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78617110", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"17056955", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47288020", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64863489", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"21558438", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52088005", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23598828", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37104752", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78129108", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"41547772", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44301729", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62269509", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81002924", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43989030", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"13720407", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44725245", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"64080081", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"91482629", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30573404", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61614056", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28640725", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43043367", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68237844", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"37069316", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"51119490", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31918761", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36335019", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40861253", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"44042765", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"49373432", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76457689", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88653179", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"35984906", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53426785", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42390909", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36483360", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"50289854", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"44520736", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59344112", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28809551", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37549278", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54969305", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62999069", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44858482", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"104847356", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39637338", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42662565", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"82995892", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46974247", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22408497", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51118491", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74021002", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"73967892", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"40574086", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40344081", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62192991", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32342477", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"36769477", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"29758918", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85950872", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22592058", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28672743", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"50276718", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36038406", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"43447114", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"43179491", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50485301", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39252175", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26144028", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29362534", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70119726", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"31058118", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42112072", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65480120", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31272325", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39999632", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"55688313", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"44345314", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"68698357", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"76910419", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40664650", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41042262", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44018574", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45953832", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49817701", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"43428039", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57690457", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58989575", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"33233988", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29549896", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43296121", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30569552", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67691587", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27626530", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"41895678", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47400410", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62861309", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"38197226", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"11995157", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28234601", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"41405422", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69926143", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"33059680", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58473190", "lang":"no", "jira_id": "SIRT-3916"},
{"user_id":"65718025", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"12833278", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55844663", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"56441468", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71024762", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"7323220", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39380325", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31488771", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"80268336", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"7674614", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29970769", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50011414", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55189410", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"30482118", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40112825", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"73823677", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"20829274", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44785651", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"68279243", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"3052662", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"94188526", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"17699380", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36319591", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"52284130", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23560538", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37010751", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45987631", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"62037199", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"82589154", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25349284", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"24108347", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50039087", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"56277785", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48929516", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28065207", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39905112", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90900116", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"57790713", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21433597", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52939532", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55854791", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35995087", "lang":"hu", "jira_id": "SIRT-3916"},
{"user_id":"28036688", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53766299", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69065162", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26114226", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51178644", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"53085563", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61835737", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"93656392", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62391017", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38085353", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45898674", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52323725", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53471648", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"81054372", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40531175", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"66711076", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23829426", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39946187", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26028913", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38332147", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38830470", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"46771103", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47038403", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58078464", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58351323", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"89391337", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40851580", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"45911707", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31987217", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22781250", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50561208", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"71057143", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25845059", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38385116", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39772139", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50918671", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"53403704", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32947748", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43372860", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48298171", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75802639", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"39367256", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50129079", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49900517", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"38213257", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43929335", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23364603", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37135427", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54655721", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56894117", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61154043", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67730803", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"71849411", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"113702241", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38412514", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45163821", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"22891476", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"28581571", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51805851", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53441539", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"41127564", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"46254514", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"47389228", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60646020", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27399029", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52922621", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38316879", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"39207348", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"41126803", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"89404709", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91167969", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"2185655", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69917922", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63328492", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29356378", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36961241", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71487020", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"48803927", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"51522291", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54650497", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"96160745", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53495682", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"62563870", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"157579349", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38860306", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64724735", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"9768377", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"12784166", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84275403", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"65729230", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91188072", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28754395", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75470013", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"82345580", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"65014601", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"134982879", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58716016", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56859630", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"162174302", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32701435", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"80260232", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"115629128", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"32882103", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40229884", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42820122", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"43304514", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52351499", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"35954574", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50942169", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"127596852", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41142856", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50773022", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50943802", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"66930091", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"39382883", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88756250", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"110922521", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"32890404", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32923869", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40786027", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"46696727", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"51017734", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"56710837", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"74422264", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90660773", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"35609642", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26574136", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42528085", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"107194213", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"43441157", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61737397", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61307054", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"135913941", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29202841", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52854257", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81304000", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46801187", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"92008498", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"103594622", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"54179883", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"23543294", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43822802", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71461862", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26989487", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"79797407", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"84306655", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"32528169", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38232146", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"13031022", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"30721712", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71766922", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"79200599", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56863028", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"41555568", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46885617", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"53690484", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"37360768", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59869703", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70481644", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36377878", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32376777", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"106843019", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"10507978", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"65304165", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"23213371", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32058534", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54004018", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"68325571", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53108116", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37602250", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"15158605", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43155989", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56521337", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76385901", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"90736131", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48523436", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29471392", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"24758079", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"79514635", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"91398390", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53958172", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60152347", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49782959", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70268802", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49938264", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"65433299", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"80176144", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"51950404", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"30530079", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"36428650", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45054029", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69818706", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38835217", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43800303", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30656206", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"107494553", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31072030", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38784144", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"57326213", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54226588", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40990797", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44805240", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"35935121", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41343682", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"50396404", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71097724", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"119611214", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"37407650", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40506372", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"30652415", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"76508554", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"109439861", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"39771127", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"42747997", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36384088", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39428492", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63803637", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85701199", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31073781", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"86066576", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"28440467", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"68414246", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45345601", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31111891", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"39409972", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"64094972", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"94690551", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"63581291", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32381067", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"87249972", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"40293707", "lang":"cs", "jira_id": "SIRT-3916"},
{"user_id":"54184733", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"41990709", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45923619", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61086098", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53053448", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"52047495", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62741780", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"69414308", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"9013547", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30956761", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55437466", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101726322", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"40358678", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29181653", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50334563", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"53859125", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21442544", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30609839", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"119881279", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"27131643", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46305270", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"13610836", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62152596", "lang":"zh-cn", "jira_id": "SIRT-3916"},
{"user_id":"67739190", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32119284", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"42495447", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61045097", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"80468088", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"32351954", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48419013", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"43691359", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59167178", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"11692694", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"59335966", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49748575", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66469250", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44120046", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30972220", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60655603", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"40900786", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"69658466", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73385105", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20721044", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31876489", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"66191250", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"154530027", "lang":"it", "jira_id": "SIRT-3916"},
{"user_id":"55048158", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"31390469", "lang":"sv", "jira_id": "SIRT-3916"},
{"user_id":"57984560", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38535213", "lang":"fi", "jira_id": "SIRT-3916"},
{"user_id":"37668188", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"47600594", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30039638", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37222561", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"48599447", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44654014", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38791558", "lang":"zh-tw", "jira_id": "SIRT-3916"},
{"user_id":"52959589", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"37461264", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"95146616", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"102547842", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24249259", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62552229", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74950359", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21671310", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43087183", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21639150", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"51533859", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"84669725", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63646819", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32322886", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"63600026", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57087058", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"67554810", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64463839", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"120772316", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62545627", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"109391962", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21302211", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45480041", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36473600", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50464018", "lang":"en-us", "jira_id": "SIRT-3916"},
{"user_id":"50249362", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"67079436", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37262020", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51925282", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"67392092", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"51399427", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"70702223", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45885642", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41048789", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"52394366", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42013601", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"90565336", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"29210936", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"64326273", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21961978", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66786630", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"39361669", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"40314809", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38623083", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59787582", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"62868150", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"77850134", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"30707514", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"41083343", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"55562993", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55771986", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"36354387", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51014046", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37434469", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"63547567", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38388583", "lang":"ko", "jira_id": "SIRT-3916"},
{"user_id":"32323871", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"52906305", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42128344", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"57691031", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"39059883", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55174809", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"158935348", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"37316531", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36313064", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"23529525", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71672341", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"21597866", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52023747", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"26305851", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"41845984", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"29632971", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51142112", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60106053", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25387511", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42894068", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"37127307", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"61275350", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38185717", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"143865773", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37588024", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50347651", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"61302106", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"78354809", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"35748595", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"20705850", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"157682735", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59847459", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60746974", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27918356", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31355524", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"77945140", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"59181140", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"153636237", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"74108034", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72962384", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"72971254", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"123782776", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49928305", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30825676", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"131070", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"45261275", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"24958132", "lang":"pt", "jira_id": "SIRT-3916"},
{"user_id":"44676714", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"81154210", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"49936178", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"50267359", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"32216446", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36694401", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"88663428", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"75594896", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32952503", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46528695", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"59189713", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42266509", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"29110338", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"30777889", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"66895428", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36168333", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"76146397", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51656221", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60780549", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45021639", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56854732", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37175686", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42272445", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"54793441", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44269385", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"36265860", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"73566169", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"47210471", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"70215306", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"81543019", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50884838", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"22000491", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"29222397", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35611981", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"31820862", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50481153", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"60986082", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"86326344", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"57330981", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"62207830", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"146941171", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45804486", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"41371224", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"1839975", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43573614", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"53072635", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"116574089", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"61946174", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"26362955", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"53414893", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"85394084", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"78466330", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"60096543", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"71074982", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"28914969", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54178105", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"36708337", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"100505712", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"9305848", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"8698076", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27228315", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32274885", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43532187", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37332162", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"58955856", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25093116", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42883916", "lang":"nl", "jira_id": "SIRT-3916"},
{"user_id":"36056546", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"46864514", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"60854701", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"67449090", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"79770330", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31359168", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44405068", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"38433240", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32452204", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"76680698", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43267058", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"90408550", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"32154468", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30868543", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"54274094", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44627167", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43455372", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"30992692", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50215345", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"31483052", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52724810", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55603391", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"55901322", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58526833", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"57924693", "lang":"el", "jira_id": "SIRT-3916"},
{"user_id":"40774977", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"79785706", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"75259050", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"54189527", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"38107810", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"25322194", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"27928898", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"18543543", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45223771", "lang":"ru", "jira_id": "SIRT-3916"},
{"user_id":"48120353", "lang":"es-mx", "jira_id": "SIRT-3916"},
{"user_id":"23179163", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"51298387", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"45900764", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"35099910", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44106097", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55114741", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32404839", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"29849268", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"37714170", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"50786816", "lang":"es", "jira_id": "SIRT-3916"},
{"user_id":"42225521", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44603290", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"50010922", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"31066621", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45072885", "lang":"pt-br", "jira_id": "SIRT-3916"},
{"user_id":"25014993", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"58046991", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"9062092", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"55167717", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"32949104", "lang":"pl", "jira_id": "SIRT-3916"},
{"user_id":"43790534", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"1875857", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"44729442", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"101968558", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"10066000", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"52205617", "lang":"da", "jira_id": "SIRT-3916"},
{"user_id":"155042124", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"41903806", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"56350836", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"142488061", "lang":"fr", "jira_id": "SIRT-3916"},
{"user_id":"27924936", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"45626523", "lang":"en-gb", "jira_id": "SIRT-3916"},
{"user_id":"21379188", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"89700966", "lang":"de", "jira_id": "SIRT-3916"},
{"user_id":"57385297", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"38935585", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"43346934", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"4123340", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"42253438", "lang":"tr", "jira_id": "SIRT-3916"},
{"user_id":"55162446", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"8464732", "lang":"en", "jira_id": "SIRT-3916"},
{"user_id":"16312051", "lang":"en", "jira_id": "SIRT-3916"},
{"u