package twitch

import (
	"context"
	"fmt"
	"strconv"
	"strings"

	"github.com/pkg/errors"

	"code.justin.tv/chat/golibs/ptr"
	"code.justin.tv/common/config"
	desk "code.justin.tv/cs/desk-client"
	"code.justin.tv/cs/pong/logger"
	"code.justin.tv/cs/pong/tickets"
	salesforce "code.justin.tv/cs/salesforce-client"
)

const (
	deskBodySummaryMaxWords  = 10
	deskBodySummaryMaxLength = 100

	deskToEmail = "support@twitch.tv"

	deskLabelAmazonTransferIn          = "Amazon - Transfer In"
	deskLabelAmazonTransferOutComplete = "Amazon - Transfer Out Complete"

	deskFieldTwitchUserID       = "twitch_userid"
	deskFieldTwitchLogin        = "twitch_login"
	deskFieldTwitchDisplayName  = "twitch_display_name"
	deskFieldTwitchPartner      = "twitch_partner"
	deskFieldTwitchTurbo        = "twitch_turbo"
	deskFieldAmazonTicketID     = "twitch_amazon_case_id"
	deskFieldAmazonMediaID      = "twitch_amazon_media_id"
	deskFieldAmazonLinked       = "twitch_amazon_linked"
	deskFieldAmazonPrime        = "twitch_amazon_prime"
	deskFieldAmazonPrimeCountry = "twitch_amazon_prime_country"
)

func (c *client) CreateTicket(ctx context.Context, t tickets.Twitch) error {
	user, err := c.usersClient.GetUserByID(ctx, t.TwitchUserID, nil)
	if err != nil {
		return errors.Wrapf(err, "unable to retrieve information for user %s", t.TwitchUserID)
	}
	if *user.Email == "" {
		return errors.Wrapf(err, "unable to find email for user %s", t.TwitchUserID)
	}

	//Defaults
	hasTurbo := false
	hasPrime := false
	primeCountry := "US"

	premiumStatus, err := c.turboClient.PremiumStatus(ctx, t.TwitchUserID)
	if err != nil {
		logger.Errorf(err.Error(), "unable to retrieve Premium Status information for user %s, setting Turbo&Prime to false", t.TwitchUserID)
	} else {
		if premiumStatus.HasTurbo {
			hasTurbo = true
		}
		if premiumStatus.HasPrime {
			hasPrime = true
		}
	}

	primeStatus, err := c.samusClient.GetPrimeStatus(ctx, t.TwitchUserID, nil) // TODO: RIP Pending PRIME-3504
	if err != nil {
		logger.Errorf(err.Error(), "unable to retrieve prime country for user %s, falling back to US", t.TwitchUserID)
	} else {
		primeCountry = primeStatus.CountryCode
	}

	bodySummary := ticketBodySummary(t.Body, deskBodySummaryMaxWords, deskBodySummaryMaxLength)
	subject := fmt.Sprintf("Help request from Amazon User %s regarding: %s", *user.Displayname, bodySummary)

	deskCase := desk.Case{
		Type:    desk.TypeEmail,
		Labels:  []string{deskLabelAmazonTransferIn},
		Status:  desk.StatusNew,
		Subject: subject,
		CustomFields: map[string]interface{}{
			deskFieldTwitchUserID:       user.ID,
			deskFieldTwitchLogin:        *user.Login,
			deskFieldTwitchDisplayName:  *user.Displayname,
			deskFieldTwitchPartner:      desk.ConvertToDeskBoolean(false),
			deskFieldTwitchTurbo:        desk.ConvertToDeskBoolean(hasTurbo),
			deskFieldAmazonTicketID:     t.AmazonTicketID,
			deskFieldAmazonLinked:       desk.ConvertToDeskBoolean(true), // TODO: twitchtv.atlassian.net/browse/API-508
			deskFieldAmazonPrime:        desk.ConvertToDeskBoolean(hasPrime),
			deskFieldAmazonPrimeCountry: primeCountry,
		},
	}

	deskCaseMessage := desk.CaseMessage{
		Direction: desk.DirectionIn,
		From:      *user.Email,
		To:        deskToEmail,
		Subject:   subject,
		Body:      t.Body,
	}

	deskResult, err := c.deskClient.CreateCase(ctx, &deskCase, &deskCaseMessage)
	if err != nil {
		return errors.Wrapf(err, "failed to create Desk case for user %s", t.TwitchUserID)
	}

	logger.Infof("created Desk ticket %d from Amazon ticket %s", deskResult.ID, t.AmazonTicketID)

	salesforceCase := salesforce.CustomerCase{
		Origin:             ptr.String(config.App()),
		Subject:            ptr.String(subject),
		Description:        ptr.String(t.Body),
		TwitchUserID:       ptr.String(user.ID),
		TwitchUsername:     ptr.String(*user.Login),
		TwitchDisplayName:  ptr.String(*user.Displayname),
		TwitchTurbo:        ptr.Bool(hasTurbo),
		AmazonCaseID:       ptr.String(t.AmazonTicketID),
		TwitchAmazonLinked: ptr.Bool(true),
		TwitchPrime:        ptr.Bool(hasPrime),
		TwitchPrimeCountry: ptr.String(primeCountry),
		Labels:             ptr.String(deskLabelAmazonTransferIn),
	}

	//TODO: add a call to the service cloud client here as well
	serviceCloudResult, err := c.serviceCloudClient.CreateCase(ctx, salesforceCase)
	if err != nil {
		return errors.Wrapf(err, "failed to create Service Cloud case for user %s", t.TwitchUserID)
	}

	logger.Infof("created Service Cloud ticket %d from Amazon ticket %s", serviceCloudResult, t.AmazonTicketID)
	return nil
}

func (c *client) MarkTicketTransferred(ctx context.Context, id string, amazonMediaID string) error {
	parsedID, err := strconv.Atoi(id)
	if err != nil {
		return errors.Wrapf(err, "unable to parse Desk id %s for Amazon id %s", id, amazonMediaID)
	}

	deskCase := desk.Case{
		ID:          parsedID,
		LabelAction: desk.LabelActionAppend,
		Labels:      []string{deskLabelAmazonTransferOutComplete},
		Status:      desk.StatusResolved,
		CustomFields: map[string]interface{}{
			deskFieldAmazonMediaID: amazonMediaID,
		},
	}
	//TODO: add a call to the service cloud client here as well
	_, err = c.deskClient.UpdateCase(ctx, &deskCase)
	if err != nil {
		return errors.Wrapf(err, "failed to update Desk id %s with Amazon id %s", id, amazonMediaID)
	}

	logger.Infof("marked Desk ticket %s as transferred to Amazon", id)

	// salesforceCase := salesforce.CustomerCase{
	// 	Origin:        ptr.String(config.App()),
	// 	Labels:        ptr.String(deskLabelAmazonTransferOutComplete),
	// 	Status:        ptr.String("resolved"),
	// 	AmazonMediaID: ptr.String(amazonMediaID),
	// }

	// err = c.serviceCloudClient.UpdateCase(ctx, string(parsedID), salesforceCase)
	// if err != nil {
	// 	return errors.Wrapf(err, "failed to update ServiceCloud id %s with Amazon id %s", id, amazonMediaID)
	// }

	// logger.Infof("marked Service Cloud ticket %s as transferred to Amazon", id)
	return nil
}

func ticketBodySummary(body string, numWords, maxLength int) string {
	if numWords < 1 || maxLength < 3 {
		return ""
	}

	words := strings.Split(body, " ")
	if len(words) > numWords {
		words = words[:numWords]
	}

	summary := strings.Join(words, " ")
	if len(summary) > maxLength {
		summary = summary[:maxLength-3] + "..."
	}
	return summary
}
