package ptr

import (
	"github.com/golang/protobuf/ptypes"
	"github.com/golang/protobuf/ptypes/duration"
	"github.com/golang/protobuf/ptypes/timestamp"
	"github.com/golang/protobuf/ptypes/wrappers"

	"time"
)

// StringValue returns a Protobuf string wrapper around a string pointer
func StringValue(v *string) *wrappers.StringValue {
	if v == nil {
		return nil
	}

	return &wrappers.StringValue{
		Value: *v,
	}
}

// FromStringValue return a string pointer from a Protobuf string wrapper
func FromStringValue(val *wrappers.StringValue) *string {
	if val == nil {
		return nil
	}

	s := val.GetValue()
	return &s
}

// BoolValue returns a Protobuf bool wrapper around a bool pointer
func BoolValue(v *bool) *wrappers.BoolValue {
	if v == nil {
		return nil
	}

	return &wrappers.BoolValue{
		Value: *v,
	}
}

// FromBoolValue return a bool pointer from a Protobuf bool wrapper
func FromBoolValue(val *wrappers.BoolValue) *bool {
	if val == nil {
		return nil
	}

	b := val.GetValue()
	return &b
}

// Int32Value returns a Protobuf wrapper around a int32 pointer
func Int32Value(v *int32) *wrappers.Int32Value {
	if v == nil {
		return nil
	}

	return &wrappers.Int32Value{
		Value: *v,
	}
}

// FromInt32Value return a int32 pointer from a Protobuf int32 wrapper
func FromInt32Value(val *wrappers.Int32Value) *int {
	if val == nil {
		return nil
	}

	b := int(val.GetValue())
	return &b
}

// ProtobufTimestamp returns a Protobuf timestamp wrapper around a time
func ProtobufTimestamp(val *time.Time) (*timestamp.Timestamp, error) {
	if val == nil {
		return nil, nil
	}

	ts, err := ptypes.TimestampProto(*val)
	if err != nil {
		return nil, err
	}

	return ts, nil
}

// ProtobufTimestampUnsafe returns a Protobuf timestamp wrapper around a time. Instead of erroring,
// returns nil.
func ProtobufTimestampUnsafe(val *time.Time) *timestamp.Timestamp {
	if val == nil {
		return nil
	}

	ts, err := ptypes.TimestampProto(*val)
	if err != nil {
		return nil
	}

	return ts
}

// FromProtobufTimestamp return a time pointer from a Protobuf timestamp
func FromProtobufTimestamp(val *timestamp.Timestamp) (*time.Time, error) {
	if val == nil {
		return nil, nil
	}

	t, err := ptypes.Timestamp(val)
	if err != nil {
		return nil, err
	}

	return &t, nil
}

// FromProtobufTimestampUnsafe return a time pointer from a Protobuf timestamp. Instead of erroring,
// returns nil.
func FromProtobufTimestampUnsafe(val *timestamp.Timestamp) *time.Time {
	if val == nil {
		return nil
	}

	t, err := ptypes.Timestamp(val)
	if err != nil {
		return nil
	}

	return &t
}

// FromProtobufDuration returns a duration pointer from a Protobuf duration
func FromProtobufDuration(val *duration.Duration) (*time.Duration, error) {
	if val == nil {
		return nil, nil
	}

	d, err := ptypes.Duration(val)
	if err != nil {
		return nil, err
	}

	return &d, nil
}

// FromProtobufDurationUnsafe return a duration pointer from a Protobuf duration. Instead of erroring,
// returns nil.
func FromProtobufDurationUnsafe(val *duration.Duration) *time.Duration {
	if val == nil {
		return nil
	}

	d, err := ptypes.Duration(val)
	if err != nil {
		return nil
	}

	return &d
}
