package salesforce

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"net/url"
	"strings"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/foundation/twitchclient"
	"github.com/pkg/errors"
)

// APIVersion is the version of the Salesforce API.
const APIVersion = "v43.0"

type Client interface {
	CreateCase(ctx context.Context, cl CustomerCase) (string, error)
	UpdateCase(ctx context.Context, salesforceID string, cl CustomerCase) error
}

func NewSalesforceClient(c *Config) (Client, error) {
	httpClient := twitchclient.NewHTTPClient(twitchclient.ClientConf{})

	self := &client{
		url:          c.URL,
		clientID:     c.ClientID,
		clientSecret: c.ClientSecret,
		username:     c.Username,
		password:     c.Password,
		httpClient:   httpClient,
	}

	apiURL := fmt.Sprintf("%s/services/oauth2/token", self.url)

	data := url.Values{}

	data.Set("grant_type", "password")
	data.Add("client_id", self.clientID)
	data.Add("client_secret", self.clientSecret)
	data.Add("username", self.username)
	data.Add("password", self.password)

	resp, err := self.httpClient.PostForm(apiURL, data)
	if err != nil {
		return nil, err
	}

	defer func() {
		err = resp.Body.Close()
		if err != nil {
			logx.Error(context.Background(), errors.Wrap(err, "closing response body"))
		}
	}()

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, err
	}

	var token TokenResponse
	jsonErr := json.Unmarshal(body, &token)
	if jsonErr != nil {
		return nil, jsonErr
	}

	if token.Error != "" {
		return nil, errors.Wrap(errors.New(token.Error), token.ErrorDescription)
	}

	self.InstanceURL = token.InstanceURL
	self.Token = token.AccessToken
	self.jsonClient = twitchclient.NewJSONClient(self.InstanceURL, self.httpClient)
	return self, nil
}

func (c *client) CreateCase(ctx context.Context, cl CustomerCase) (string, error) {
	path := fmt.Sprintf("%s/services/data/%s/sobjects/Case", c.InstanceURL, APIVersion)
	buffer := new(bytes.Buffer)

	if err := json.NewEncoder(buffer).Encode(cl); err != nil {
		return "", err
	}

	req, err := http.NewRequest(http.MethodPost, path, buffer)
	if err != nil {
		return "", err
	}

	req.Header.Set("Authorization", fmt.Sprintf("Bearer %s", c.Token))
	req.Header.Set("Content-Type", "application/json")

	resp, err := c.httpClient.Do(req.WithContext(ctx))
	if err != nil {
		return "", errors.Wrap(err, "salesforce: failed to create case")
	}

	defer func() {
		if err = resp.Body.Close(); err != nil {
			logx.Error(ctx, "salesforce: failed to close response body", logx.Fields{"error": err})
		}
	}()

	switch resp.StatusCode {
	case http.StatusOK, http.StatusCreated:
		return handlePostOK(resp.Body)
	default:
		return "", handleResponseError(resp.StatusCode, resp.Body)
	}
}

func (c *client) UpdateCase(ctx context.Context, salesforceID string, cl CustomerCase) error {
	path := fmt.Sprintf("%s/services/data/%s/sobjects/Case/%s", c.InstanceURL, APIVersion, salesforceID)
	buffer := new(bytes.Buffer)

	if err := json.NewEncoder(buffer).Encode(cl); err != nil {
		return err
	}

	req, err := http.NewRequest(http.MethodPatch, path, buffer)
	if err != nil {
		return err
	}

	req.Header.Set("Authorization", fmt.Sprintf("Bearer %s", c.Token))
	req.Header.Set("Content-Type", "application/json")

	resp, err := c.httpClient.Do(req.WithContext(ctx))
	if err != nil {
		return errors.Wrap(err, "salesforce: failed to create case")
	}

	defer func() {
		if err = resp.Body.Close(); err != nil {
			logx.Error(ctx, "salesforce: failed to close response body", logx.Fields{"error": err})
		}
	}()

	switch resp.StatusCode {
	case http.StatusOK, http.StatusCreated, http.StatusNoContent:
		return nil
	default:
		return handleResponseError(resp.StatusCode, resp.Body)
	}
}

func handlePostOK(body io.ReadCloser) (string, error) {
	var resp CreateApplicationResponse

	if err := json.NewDecoder(body).Decode(&resp); err != nil {
		return "", errors.Wrap(err, "salesforce: failed to decode status 200 response body")
	}

	if resp.Success {
		return resp.ID, nil
	}

	return "", fmt.Errorf("salesforce: unsuccessful post: %s", strings.Join(resp.Errors, ", "))
}

func handleResponseError(status int, body io.ReadCloser) error {
	bytes, err := ioutil.ReadAll(body)
	if err != nil {
		return errors.Wrapf(err, "salesforce: failed to decode status %d response body", status)
	}

	return fmt.Errorf("salesforce: unexpected response (status: %d): %s", status, string(bytes))
}
