package salesforce

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"net/url"
	"time"

	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/foundation/twitchclient"
	"github.com/pkg/errors"
)

// APIVersion is the version of the Salesforce API.
const APIVersion = "v43.0"

// Client provides the methods necessary to create and update salesforce cases
type Client interface {
	CreateCase(ctx context.Context, cc CustomerCase) (string, error)
	UpdateCase(ctx context.Context, ccu CustomerCaseUpdate) (string, error)
}

func (c *client) updateOauthToken(ctx context.Context) error {
	apiURL := fmt.Sprintf("%s/services/oauth2/token", c.url)

	data := url.Values{}

	data.Set("grant_type", "password")
	data.Add("client_id", c.clientID)
	data.Add("client_secret", c.clientSecret)
	data.Add("username", c.username)
	data.Add("password", c.password)

	resp, err := c.httpClient.PostForm(apiURL, data)
	if err != nil {
		return err
	}

	defer func() {
		err = resp.Body.Close()
		if err != nil {
			logx.Error(context.Background(), errors.Wrap(err, "closing response body"))
		}
	}()

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return errors.Wrap(err, "reading body of token response")
	}

	var token TokenResponse
	jsonErr := json.Unmarshal(body, &token)
	if jsonErr != nil {
		return errors.Wrap(jsonErr, "unmarshaling update token response")
	}

	if token.Error != "" {
		return errors.Wrap(errors.New(token.Error), token.ErrorDescription)
	}

	c.InstanceURL = token.InstanceURL
	c.Token = token.AccessToken
	c.jsonClient = twitchclient.NewJSONClient(c.InstanceURL, c.httpClient)
	return nil
}

// NewSalesforceClient provides a client to salesforce
func NewSalesforceClient(config *Config) (Client, error) {
	httpClient := twitchclient.NewHTTPClient(twitchclient.ClientConf{})
	self := &client{
		url:          config.URL,
		clientID:     config.ClientID,
		clientSecret: config.ClientSecret,
		username:     config.Username,
		password:     config.Password,
		httpClient:   httpClient,
	}

	if err := self.updateOauthToken(context.Background()); err != nil {
		logx.Error(context.Background(), err, logx.Fields{
			"error": "error initally creating salesforce client token",
		})
	}

	// persistently keep token refreshed
	go func() {
		for {
			time.Sleep(30 * time.Minute)
			if err := self.updateOauthToken(context.Background()); err != nil {
				logx.Error(context.Background(), err, logx.Fields{
					"error": "error updating salesforce client token",
				})
			}
		}
	}()

	return self, nil
}

// CreateCase creates a new salesforce case
func (c *client) CreateCase(ctx context.Context, cc CustomerCase) (string, error) {
	path := fmt.Sprintf("%s/services/apexrest/PONG", c.InstanceURL)
	buffer := new(bytes.Buffer)

	if err := json.NewEncoder(buffer).Encode(cc); err != nil {
		return "", err
	}

	req, err := http.NewRequest(http.MethodPost, path, buffer)
	if err != nil {
		return "", err
	}

	req.Header.Set("Authorization", fmt.Sprintf("Bearer %s", c.Token))
	req.Header.Set("Content-Type", "application/json")

	resp, err := c.httpClient.Do(req.WithContext(ctx))
	if err != nil {
		return "", errors.Wrap(err, "salesforce: failed to create case")
	}

	defer func() {
		if err = resp.Body.Close(); err != nil {
			logx.Error(ctx, "salesforce: failed to close response body", logx.Fields{"error": err})
		}
	}()

	switch resp.StatusCode {
	case http.StatusOK, http.StatusCreated:
		return handleCaseCreateSuccess(resp.Body)
	default:
		return "", handleResponseError(resp.StatusCode, resp.Body)
	}
}

func (c *client) UpdateCase(ctx context.Context, ccu CustomerCaseUpdate) (string, error) {
	path := fmt.Sprintf("%s/services/apexrest/PONG", c.InstanceURL)
	buffer := new(bytes.Buffer)

	if err := json.NewEncoder(buffer).Encode(ccu); err != nil {
		return "", err
	}

	req, err := http.NewRequest(http.MethodPatch, path, buffer)
	if err != nil {
		return "", err
	}

	req.Header.Set("Authorization", fmt.Sprintf("Bearer %s", c.Token))
	req.Header.Set("Content-Type", "application/json")

	resp, err := c.httpClient.Do(req.WithContext(ctx))
	if err != nil {
		return "", errors.Wrap(err, "salesforce: failed to create case")
	}

	defer func() {
		if err = resp.Body.Close(); err != nil {
			logx.Error(ctx, "salesforce: failed to close response body", logx.Fields{"error": err})
		}
	}()

	switch resp.StatusCode {
	case http.StatusOK, http.StatusCreated, http.StatusNoContent:
		return handleCaseUpdateSuccess(resp.Body)
	default:
		return "", handleResponseError(resp.StatusCode, resp.Body)
	}
}

func handleCaseCreateSuccess(body io.ReadCloser) (string, error) {
	var resp CustomerCaseCreated

	if err := json.NewDecoder(body).Decode(&resp); err != nil {
		return "", errors.Wrap(err, "salesforce: failed to decode status 200 response body")
	}

	return string(resp), nil
}

func handleCaseUpdateSuccess(body io.ReadCloser) (string, error) {
	var resp CustomerCaseUpdateSuccess

	if err := json.NewDecoder(body).Decode(&resp); err != nil {
		return "", errors.Wrap(err, "salesforce: failed to decode status 200 response body")
	}

	return string(resp), nil
}

func handleResponseError(status int, body io.ReadCloser) error {
	bytes, err := ioutil.ReadAll(body)
	if err != nil {
		return errors.Wrapf(err, "salesforce: failed to decode status %d response body", status)
	}

	return fmt.Errorf("salesforce: unexpected response (status: %d): %s", status, string(bytes))
}
