package salesforce

import (
	"context"
	"fmt"
	"log"
	"os"
	"testing"

	twitchConfig "code.justin.tv/common/config"
	"code.justin.tv/cs/salesforce-client/config"

	"github.com/pkg/errors"
	"github.com/stretchr/testify/assert"
)

func configurate(t *testing.T) *config.Config {

	t.Helper()

	log.SetPrefix("TRACE: ")
	log.SetFlags(log.LstdFlags | log.Llongfile)
	log.Println("log prefix set.")

	// environment type isn't known until Parse() is called
	environment, exists := os.LookupEnv("ENVIRONMENT")
	if !exists {
		t.Fatal("ENVIRONMENT is not set")
	}

	t.Logf("Configuring with environment: %s", environment)

	err := config.SetEnv(environment)
	if err != nil {
		t.Fatal(errors.Wrapf(err, "Failed to configure environment %s:", environment))
	}

	t.Log("Parsing common config")
	err = twitchConfig.Parse()
	if err != nil {
		t.Fatal(errors.Wrapf(err, "Failed to parse common config"))
	}
	if environment != twitchConfig.Environment() {
		t.Fatalf("Environment mismatch: %s vs %s", environment, twitchConfig.Environment())
	}

	// Environment() and App() are now available for use

	t.Log("Fetching app config")
	conf, err := config.GetConfig(twitchConfig.Environment())
	if err != nil {
		t.Fatal(errors.Wrapf(err, "Failed to get config for %s:", twitchConfig.Environment()))
	}

	t.Log("Populating config secrets")
	err = conf.PopulateSecrets()
	if err != nil {
		t.Fatal(errors.Wrapf(err, "Failed to populate secrets for %s:", twitchConfig.Environment()))
	}

	return conf
}

func TestClient(t *testing.T) {

	conf := configurate(t)

	salesforceConfig := Config{
		URL:          conf.Secrets["service_cloud.instance_url"].Value,
		ClientID:     conf.Secrets["service_cloud.client_id"].Value,
		ClientSecret: conf.Secrets["service_cloud.client_secret"].Value,
		// a service cloud API password is an account password+security_security
		Password: fmt.Sprintf("%s%s", conf.Secrets["service_cloud.password"].Value, conf.Secrets["service_cloud.security_token"].Value),
		Username: conf.Secrets["service_cloud.username"].Value,
	}

	client, clientCreateErr := NewSalesforceClient(&salesforceConfig)

	assert.Nil(t, clientCreateErr)
	assert.NotNil(t, client)

	testSubject := "cs/salesforce-client: test, please ignore"
	testDescription := "testing from cs/salesforce-client"
	testOrigin := "Pong" // origin needs to be defined on salesforce's side first, so we need a generic one for salesforce-client later

	testAmazonID := "1234567ABC"

	testCase := CustomerCase{
		Subject:      &testSubject,
		Description:  &testDescription,
		Origin:       &testOrigin,
		AmazonCaseID: &testAmazonID,
	}

	id, caseCreateErr := client.CreateCase(context.Background(), testCase)

	assert.Nil(t, caseCreateErr)
	assert.NotEmpty(t, id)
	t.Logf("creation result: %+v \n", id)
	if caseCreateErr != nil {
		t.Errorf("create err: %+v\n\n", caseCreateErr)
	}

	testObjectID := id

	testCaseUpdate := CustomerCaseUpdate{
		CaseID:       &testObjectID,
		AmazonCaseID: &testAmazonID,
	}

	success, caseUpdateErr := client.UpdateCase(context.Background(), testCaseUpdate)

	assert.Nil(t, caseUpdateErr)
	assert.NotEmpty(t, success)
	t.Logf("update result: %+v \n", success)
	if caseUpdateErr != nil {
		t.Errorf("update err: %+v\n\n", caseUpdateErr)
	}

}
