package config

import (
	"log"
	"os"

	"github.com/pkg/errors"
)

type env struct {
	name  string
	value string
}

type envVars []env
type secret struct {
	Value   string
	fetched bool
}

type secrets map[string]secret

// Config wraps connection information for kinesis, redis, and the ports this app serves traffic on
type Config struct {
	Secrets     secrets
	environment envVars
}

var configs = map[string]*Config{
	"development": {
		environment: envVars{
			{name: "APP", value: "pong"}, // this lets common config's App() return the correct app name when the binary lives in a subfolder
			{name: "STATSD_HOST_PORT", value: "localhost:8125"},
		},

		Secrets: secrets{
			"service_cloud.instance_url":   {},
			"service_cloud.client_id":      {},
			"service_cloud.client_secret":  {},
			"service_cloud.username":       {},
			"service_cloud.password":       {},
			"service_cloud.security_token": {},
		},
	},

	"test": {
		environment: envVars{
			{name: "APP", value: "pong"},
		},

		Secrets: secrets{
			"service_cloud.instance_url":   {},
			"service_cloud.client_id":      {},
			"service_cloud.client_secret":  {},
			"service_cloud.username":       {},
			"service_cloud.password":       {},
			"service_cloud.security_token": {},
		},
	},

	"staging": {
		environment: envVars{
			{name: "APP", value: "pong"},
			{name: "STATSD_HOST_PORT", value: "localhost:8125"},
		},

		Secrets: secrets{
			"service_cloud.instance_url":   {},
			"service_cloud.client_id":      {},
			"service_cloud.client_secret":  {},
			"service_cloud.username":       {},
			"service_cloud.password":       {},
			"service_cloud.security_token": {},
		},
	},

	"production": {
		environment: envVars{
			{name: "APP", value: "pong"}, // this lets common config's App() return the correct app name when the binary lives in a subfolder
		},

		Secrets: secrets{
			"service_cloud.instance_url":   {},
			"service_cloud.client_id":      {},
			"service_cloud.client_secret":  {},
			"service_cloud.username":       {},
			"service_cloud.password":       {},
			"service_cloud.security_token": {},
		},
	},
}

// GetConfig loads the configuration for a given environment
func GetConfig(env string) (*Config, error) {
	log.Printf("Getting config with environment: %s", env)

	conf, ok := configs[env]

	if !ok {
		return nil, errors.New("no config found for env: " + env)
	}

	return conf, nil
}

// SetEnv stands up environmental variables. Call prior to the environment being parsed.
func SetEnv(env string) error {

	log.Printf("Setting environment variables for env: %s", env)

	conf, ok := configs[env]

	if !ok {
		return errors.New("no config found for env: " + env)
	}

	for _, e := range conf.environment {
		err := os.Setenv(e.name, e.value)
		if err != nil {
			return errors.Wrapf(err, "setting env var %s", e.name)
		}
	}

	return nil
}

// PopulateSecrets stands up secret values. Call after the environment is parsed.
func (conf *Config) PopulateSecrets() error {
	set := make(secrets)

	for k := range conf.Secrets {
		value, err := populateSecret(k)
		if err != nil {
			return errors.Wrapf(err, "populating secret: %s", k)
		}
		set[k] = secret{Value: value, fetched: true}
	}

	conf.Secrets = set
	return nil
}
