package config

import (
	"bytes"
	"fmt"
	"log"
	"text/template"

	twitchConfig "code.justin.tv/common/config"
	"code.justin.tv/systems/sandstorm/manager"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sts"
	"github.com/pkg/errors"
)

func getSecret(sandstormManager manager.API, secretName string) (string, error) {
	secretNameTemplate := "{{.Org}}/{{.App}}/{{.Environment}}/{{.SecretName}}"
	params := struct {
		Org         string
		App         string
		Environment string
		SecretName  string
	}{
		Org:         twitchConfig.App(),
		App:         "app",
		Environment: twitchConfig.Environment(),
		SecretName:  secretName,
	}

	tmpl, err := template.New("secretNameTemplate").Parse(secretNameTemplate)
	if err != nil {
		return "", errors.Wrapf(err, "parsing secretName template for %s", secretName)
	}

	var key bytes.Buffer
	err = tmpl.Execute(&key, params)
	if err != nil {
		return "", errors.Wrapf(err, "executing secretName template for %s", secretName)
	}

	secret, err := sandstormManager.Get(key.String())
	if err != nil {
		return "", err
	}

	return string(secret.Plaintext), nil

}

// populateSecret takes a name and returns the value fetched from sandstorm.
func populateSecret(secretName string) (string, error) {

	serviceName := fmt.Sprintf("safety-%s-%s", twitchConfig.App(), twitchConfig.Environment())

	session, err := session.NewSession()
	if err != nil {
		return "", errors.Wrap(err, "creating aws session")
	}

	// Get credentials to access information on the sandstorm account
	creds := credentials.NewCredentials(&stscreds.AssumeRoleProvider{
		// This is the ARN to the IAM role retrieved from https://dashboard-v2.internal.justin.tv/sandstorm/manage-roles
		RoleARN: "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/" + serviceName,
		Client:  sts.New(session),
	})

	m := manager.New(manager.Config{
		AWSConfig:   aws.NewConfig().WithRegion("us-west-2").WithCredentials(creds),
		ServiceName: serviceName,
	})

	svc := sts.New(session)
	input := &sts.GetCallerIdentityInput{}
	_, err = svc.GetCallerIdentity(input)
	if err != nil {
		if aerr, ok := err.(awserr.Error); ok {
			switch aerr.Code() {
			default:
				log.Println("@@@@ GetCallerIdentity error1:", aerr.Error())
			}
		} else {
			log.Println("@@@@ GetCallerIdentity error2:", err.Error())
		}
		log.Println("@@@@@ here 3")
	}

	defer func() {
		_ = m.CleanUp()
	}()

	value, err := getSecret(m, secretName)
	if err != nil {
		return "", err
	}

	return value, nil
}
