package main

import (
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"log"
	"net/http"
	"reflect"
	"strings"
	"time"

	"code.justin.tv/common/config"
	"github.com/cactus/go-statsd-client/statsd"
)

type DatabaseWrite struct {
	databaseName string
	databaseHost string
}

func (write DatabaseWrite) String() string {
	return fmt.Sprintf("%s|%s", write.databaseName, write.databaseHost)
}

var dbLogChannel chan DatabaseWrite = make(chan DatabaseWrite, 100)

func LogDatabaseWrite(write DatabaseWrite) {
	select {
	case dbLogChannel <- write:
	default:
	}
}

func SpinWriteProgress(graphiteQueryURL string) {
	databases := make(map[string]DatabaseWrite)
	ticker := time.Tick(5 * time.Minute)

	for {
		select {
		case write := <-dbLogChannel:
			writeName := write.String()
			_, ok := databases[writeName]
			if !ok {
				databases[writeName] = write
			}
			continue
		case <-ticker:
			WriteAllValues(databases, graphiteQueryURL)
			databases = make(map[string]DatabaseWrite)
			continue
		}
	}
}

func WriteAllValues(databases map[string]DatabaseWrite, graphiteQueryURL string) {
	for _, value := range databases {
		err := WriteGraphiteValue(config.Statsd(), graphiteQueryURL, config.Environment(), repoName(), value.databaseName, metricify(value.databaseHost), "upper_90", 50)
		if err != nil {
			log.Println(err)
		}

		err = WriteGraphiteValue(config.Statsd(), graphiteQueryURL, config.Environment(), repoName(), value.databaseName, metricify(value.databaseHost), "upper_99", 500)
		if err != nil {
			log.Println(err)
		}
	}
}

func WriteGraphiteValue(stats statsd.Statter, graphiteUrl string, environment string, repo string, database string, host string, metric string, healthThreshold int) error {
	justUrlIndex := strings.Index(graphiteUrl, ":")
	if justUrlIndex >= 0 {
		graphiteUrl = graphiteUrl[0:justUrlIndex]
	}

	url := fmt.Sprintf(
		"http://%s/render?target=offset(scale(changed(diffSeries(maxSeries(stats.timers.alucard.%s.%s.%s.%s.queries.%s),removeAboveValue(maxSeries(stats.timers.alucard.%s.%s.%s.%s.queries.%s),%d))),-1),1)&format=json&from=-10min&to=-5min",
		graphiteUrl,
		environment,
		repo,
		database,
		host,
		metric,
		environment,
		repo,
		database,
		host,
		metric,
		healthThreshold)

	resp, err := http.Get(url)
	if err != nil {
		return err
	}

	defer func() {
		err := resp.Body.Close()
		if err != nil {
			log.Println(err)
		}
	}()

	body, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return err
	}

	if resp.StatusCode/100 != 2 {
		return fmt.Errorf("Received %d response when accessing %s", resp.StatusCode, url)
	}

	graphiteData := make([]map[string]interface{}, 1)
	err = json.Unmarshal(body, &graphiteData)
	if err != nil {
		return err
	}

	if len(graphiteData) < 1 {
		return errors.New("No graphite metrics returned from the json API.")
	}

	metricField := graphiteData[0]

	datapoints, ok := metricField["datapoints"]
	if !ok {
		return errors.New("No graphite datapoints returned for the first metric from the json API.")
	}

	structDatapoints, ok := datapoints.([]interface{})
	if !ok {
		return fmt.Errorf("The graphite datapoints field was the wrong type: instead of []interface{}, was %s", reflect.TypeOf(datapoints))
	}

	if len(structDatapoints) < 1 {
		return errors.New("No graphite datapoints returned for the first metric from the json API.")
	}

	var total float32
	for _, datapoint := range structDatapoints {
		structDatapoint, ok := datapoint.([]interface{})
		if !ok {
			return fmt.Errorf("Found a datapoint that was the wrong type: instead of []interface{}, was %s", reflect.TypeOf(datapoint))
		}

		if len(structDatapoint) != 2 {
			return fmt.Errorf("Was expecting a single data point to be in the format [value, timestamp], instead got %v.", structDatapoint)
		}

		value, ok := structDatapoint[0].(float64)
		if !ok {
			return fmt.Errorf("Datapoint value was not float32 as expected.  Instead was %s.", reflect.TypeOf(structDatapoint[0]))
		}

		normalizedValue := float32(value) / float32(len(structDatapoints))
		total += normalizedValue
	}

	metricName := fmt.Sprintf("%s.%s.health.%s", database, host, metric)
	return stats.Gauge(metricName, int64(total*100000), 1.0)
}
