package main

import (
	"io"
	"log"
	"strings"
	"time"

	"golang.org/x/net/context"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"

	"code.justin.tv/common/chitin"
	"code.justin.tv/common/config"
	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/release/trace/api"
)

var firehoseSampleRate float64 = 0.01

func main() {
	config.Register(map[string]string{
		"FirehoseHostPort":      "", // The host/port of the trace firehose we're going to read from
		"Service":               "", //The name of the service to watch for firehose events
		"Databases":             "", //Comma-separated list of databases to watch
		"GraphiteQueryHostPort": "graphite-web.internal.justin.tv",
	})

	//Pull config
	err := config.Parse()
	if err != nil {
		log.Fatal(err)
	}

	firehoseAddr := config.MustResolve("FirehoseHostPort")
	databases := config.MustResolve("Databases")
	watchService := config.MustResolve("Service")
	graphiteQueryURL := config.MustResolve("GraphiteQueryHostPort")

	var databaseList = map[string]string{}
	for _, dbName := range strings.Split(databases, ",") {
		databaseList[dbName] = dbName
	}

	//Auto-trace
	err = chitin.ExperimentalTraceProcessOptIn()
	if err != nil {
		log.Fatal(err)
	}

	go SpinWriteProgress(graphiteQueryURL)

	go func() {
		for {
			err = runTrace(firehoseAddr, watchService, databaseList)

			time.Sleep(200 * time.Millisecond)

			if err == nil || grpc.Code(err) == codes.DeadlineExceeded || err == io.EOF {
				continue
			}

			if config.RollbarErrorLogger() != nil {
				config.RollbarErrorLogger().Error(err)
			}
			log.Println(err)
		}
	}()

	//Launch the healthcheck endpoint- this will return current health status on :8000/debug/running
	log.Fatal(twitchhttp.ListenAndServe(twitchhttp.NewServer()))
}

func runTrace(firehoseAddr string, watchService string, databaseList map[string]string) error {
	//Open a GRPC connection to the trace URL
	log.Println("Dialing")
	traceConn, err := grpc.Dial(firehoseAddr, grpc.WithInsecure())
	if err != nil {
		log.Fatalf("Could not reach firehose: %v", err)
	}
	defer attemptClose(traceConn)

	//Build a firehose siphon to pipe into stats
	firehoseClient := api.NewTraceClient(traceConn)
	stats := NewStatsWriter()
	siphon := NewFirehoseSiphon(firehoseClient, stats, watchService, databaseList)
	defer siphon.CloseFireHose()
	//Run the siphon - if we have an error, keep rebuilding & trying again
	return siphon.PullTransactions(context.Background())
}

func attemptClose(grpcConn *grpc.ClientConn) {
	err := grpcConn.Close()

	if err != nil {
		log.Println("Error closing conn: ", err)
	}
}
