variable "template_name" {
    default = "user_data.tpl"
}

# Basic IAM Role
resource "aws_iam_role" "iam_role" {
    name = "${var.service}-${var.role}-${var.environment}"
    path = "/"
    assume_role_policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Action": "sts:AssumeRole",
            "Principal": {
                "Service": "ec2.amazonaws.com"
            },
            "Effect": "Allow",
            "Sid": ""
        }
    ]
}
EOF
}

resource "aws_iam_instance_profile" "instance_profile" {
    name = "${var.service}-${var.role}-${var.environment}"
    role = "${aws_iam_role.iam_role.id}"
    depends_on = ["aws_iam_role.iam_role"]
}

data "template_file" "user_data" {
    template = "${file(join("/", list(path.module, var.template_name)))}"
    vars {
        role = "${var.role}"
        cluster = "${var.cluster == "" ? var.role : var.cluster }"
        env = "${lookup(var.environment_tagname, var.environment)}"
        owner = "${var.owner}"
        environment = "${var.environment}"
        puppet = "${lookup(var.puppet_server, var.puppet)}"
        dns_region = "${lookup(var.dns_region, var.region)}"
    }
}

resource "aws_launch_configuration" "asg_launch" {

    image_id = "${var.ami}"
    associate_public_ip_address = "false"
    instance_type = "${var.instance_type}"
    security_groups = "${var.vpc_security_group_ids}"
    iam_instance_profile = "${aws_iam_instance_profile.instance_profile.id}"

    root_block_device {
        volume_size = "${var.root_block_device_size}"
    }

    lifecycle {
        create_before_destroy = true
    }
    
    user_data = "${data.template_file.user_data.rendered}"
}

resource "aws_autoscaling_group" "asg" {

    name = "${var.service}-${var.role}-${var.environment}"
    launch_configuration = "${aws_launch_configuration.asg_launch.name}"

    max_size = "${var.max_count}"
    min_size = "${var.min_count}"
    desired_capacity = "${var.desired_capacity}"
    vpc_zone_identifier = "${var.subnet_ids}"

    load_balancers = ["${var.load_balancers}"]
    min_elb_capacity = "${length(var.load_balancers) > 0 ? var.desired_capacity : 0}"

    tag {
        key = "Name"
        value = "${var.role}-${var.environment}"
        propagate_at_launch = true
    }

    tag {
        key = "Environment"
        value = "${lookup(var.environment_tagname, var.environment)}"
        propagate_at_launch = true
    }

    tag {
        key = "Owner"
        value = "${var.owner}@twitch.tv"
        propagate_at_launch = true
    }

    tag {
        key = "Service"
        value = "${var.owner}/${var.service}/${var.role}"
        propagate_at_launch = true
    }
}
