package queue

import (
	"fmt"
	"os"
	"path"
	"strings"

	"code.justin.tv/systems/sandstorm/util"
	log "github.com/Sirupsen/logrus"
	"github.com/aws/aws-sdk-go/aws"
	uuid "github.com/satori/go.uuid"
	"github.com/vaughan0/go-ini"
)

const (
	sectionHeader          = "sandstorm-agent"
	maxQueueHostnameLength = 42
	defaultTopicArn        = "arn:aws:sns:us-west-2:734326455073:sandstorm-production"
	defaultQueueConfigPath = "/var/run/sandstorm/"
	defaultQueueFileName   = "queue_arn"
	defaultQueueNamePrefix = "sandstorm"
)

//Config for Queue
type Config struct {
	QueueArn        string
	QueueURL        string
	TopicArn        string
	AWSConfig       *aws.Config
	QueueConfigPath string
	QueueFileName   string
	QueueNamePrefix string
}

// buildQueueIdentifier returns a newly generated UUID and a FQDN
// Queuenames have a maximum length of 80 chars and and we prefix with 'sandstorm'
// and suffix wth a UUID. Hostnames are truncated down to 43-len(prefix) chars to fit
// name length
func buildQueueIdentifiers(namePrefix string) (uid, name string, err error) {
	name, err = os.Hostname()
	if err != nil {
		return
	}
	uid = uuid.NewV4().String()
	if len(name) > maxQueueHostnameLength-len(namePrefix) {
		name = name[0:(maxQueueHostnameLength - len(namePrefix) - 1)]
	}
	fqdn := fmt.Sprintf("%s-%s-%s", namePrefix, name, uid)
	fqdn = strings.Replace(fqdn, ".", "_", -1)
	return uid, fqdn, nil
}

// DefaultConfig creates a config with default values
func DefaultConfig() Config {
	return Config{
		TopicArn:        defaultTopicArn,
		AWSConfig:       &aws.Config{Region: aws.String("us-west-2")},
		QueueConfigPath: defaultQueueConfigPath,
		QueueFileName:   defaultQueueFileName,
		QueueNamePrefix: defaultQueueNamePrefix,
	}
}

// replaces nil values in provided config with defaultConfig
func mergeConfig(provided *Config, defaultConfig Config) {
	if provided.AWSConfig == nil {
		provided.AWSConfig = defaultConfig.AWSConfig
	}

	if provided.QueueConfigPath == "" {
		provided.QueueConfigPath = defaultConfig.QueueConfigPath
	}

	if provided.QueueFileName == "" {
		provided.QueueFileName = defaultConfig.QueueFileName
	}

	if provided.QueueNamePrefix == "" {
		provided.QueueNamePrefix = defaultConfig.QueueNamePrefix
	}
}

// loads queue config data from file
func (q *Queue) loadQueueConfig() error {
	queueConfigPath := path.Join(q.Config.QueueConfigPath, q.Config.QueueFileName)
	if util.FileExists(queueConfigPath) {
		QueueFile, err := ini.LoadFile(queueConfigPath)
		if err != nil {
			return fmt.Errorf("Error loading agent queue config: %s", err.Error())
		}

		q.Config.QueueArn, _ = QueueFile.Get(sectionHeader, "QueueArn")
		q.Config.QueueURL, _ = QueueFile.Get(sectionHeader, "QueueURL")

	} else {

		log.Info(fmt.Sprintf(
			"No queue config found at %s, creating now.",
			queueConfigPath,
		))
	}
	return nil
}
