package utils

import (
	"fmt"
	"os"
	"os/exec"

	"github.com/pkg/errors"
	"github.com/spf13/cobra"
	"code.justin.tv/d8a/buddy/cmd/buddy-cli/setup"
	"code.justin.tv/d8a/buddy/lib/zenyatta"
)

var etlFmtCmd = &cobra.Command{
	Use:   "etl-format [cluster] [format]",
	Short: "Update etl output format of cluster [cluster] to csv, parquet, default or no output",
	Long:  "`buddy etl-format [cluster] [format]` will update the output format for a cluster, where [format] is either csv, parquet, default or no_output",
	RunE: func(command *cobra.Command, args []string) error {

		if len(args) < 2 {
			return command.Usage()
		}

		clusterId := args[0]
		dbFormat := args[1]

		cmdArgs := make([]string, 0)
		cmdArgs = append(cmdArgs, "/opt/twitch/zenyatta/current/scripts/update_dag_defaults.py", "--rds-id", clusterId)

		switch dbFormat {
		case "csv", "parquet":
			fmt.Printf("db snapshot will be saved as %v files.\n", dbFormat)
			cmdArgs = append(cmdArgs, "--output-format", dbFormat)
		case "default":
			fmt.Printf("db snapshot will be restored to both CSV and Parquet files.\n")
			cmdArgs = append(cmdArgs, "--output-format", "csv parquet")
		case "no_output":
			fmt.Printf("no db snapshot will be produced\n")
			cmdArgs = append(cmdArgs, "--remove")
		default:
			fmt.Printf("%v is not a valid option. Please pick one from 'csv', 'parquet', 'default' or 'no_output'", dbFormat)
			return nil
		}

		cmd := exec.Command("/opt/virtualenvs/airflow/bin/python", cmdArgs...)
		env, err := zenyatta.ZenyattaEnvironment()
		if err != nil {
			return err
		}
		cmd.Env = env
		cmd.Stderr = os.Stderr
		cmd.Stdin = os.Stdin
		cmd.Stdout = os.Stdout
		err = cmd.Run()
		if err != nil {
			return errors.Wrap(err, "failed to run update_dag_defaults.py script")
		}
		fmt.Println("Buddy updated db snapshot format. Please restart zenyatta service to pick up the change.")
		return nil

	},
}

func init() {
	setup.RootCmd.AddCommand(etlFmtCmd)
}
