package api

import (
	"encoding/json"
	"log"
	"net/http"
	"time"

	"golang.org/x/net/context"
)

type OutputBulkStatus struct {
	Name     string     `json:"name"`
	Complete bool       `json:"complete"`
	NextRow  int        `json:"nextRow,omitempty"`
	LastRun  *time.Time `json:"lastRun,omitempty"`
	Cluster  string     `json:"cluster"`
}

type BulkStatusResponse struct {
	BulkStatuses     []*OutputBulkStatus `json:"bulkStatuses"`
	CurrentlyRunning *bool               `json:"currentlyRunning,omitempty"`
	NextRun          *time.Time          `json:"nextRun,omitempty"`
}

func (s *Server) BulkStatus(c context.Context, w http.ResponseWriter, r *http.Request) {
	resp := &BulkStatusResponse{}

	now := time.Now()
	nextRun := s.schedule.NextRun(now)

	if nextRun.Before(now.Add(time.Minute)) {
		b := true
		resp.CurrentlyRunning = &b
	} else {
		resp.NextRun = &nextRun
	}
	resp.BulkStatuses = make([]*OutputBulkStatus, 0)
	statuses := s.cache.AllStatuses()

	for _, status := range statuses {
		outStatus := &OutputBulkStatus{
			Name:     status.Name,
			Complete: status.Complete,
			NextRow:  status.NextRow,
			Cluster:  status.Cluster,
		}

		if !status.ExecutedAt.IsZero() {
			outStatus.LastRun = &status.ExecutedAt
		}

		resp.BulkStatuses = append(resp.BulkStatuses, outStatus)
	}

	w.Header().Set("Content-Type", "application/json;charset=utf-8")
	err := json.NewEncoder(w).Encode(resp)
	if err != nil {
		log.Println(w.Write([]byte(err.Error())))
	}
}
