package alerts

import (
	"fmt"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/cloudwatch"
	"github.com/aws/aws-sdk-go/service/rds"
)

var requestedAlarms = make(map[string]*AlarmConfig)
var instanceMemory = map[string]float64{
	"db.t1.micro":    615,
	"db.m1.small":    1700,
	"db.m1.medium":   3750,
	"db.m1.large":    7500,
	"db.m1.xlarge":   15000,
	"db.m2.xlarge":   17100,
	"db.m2.2xlarge":  34200,
	"db.m2.4xlarge":  68400,
	"db.m3.medium":   3750,
	"db.m3.large":    7500,
	"db.m3.xlarge":   15000,
	"db.m3.2xlarge":  30000,
	"db.m4.large":    8000,
	"db.m4.xlarge":   16000,
	"db.m4.2xlarge":  32000,
	"db.m4.4xlarge":  64000,
	"db.m4.10xlarge": 160000,
	"db.r3.large":    15000,
	"db.r3.xlarge":   30500,
	"db.r3.2xlarge":  61000,
	"db.r3.4xlarge":  122000,
	"db.r3.8xlarge":  244000,
	"db.t2.micro":    1000,
	"db.t2.small":    2000,
	"db.t2.medium":   4000,
	"db.t2.large":    8000,
}

type AlarmPriority int

const (
	//NoAlarms should be set as the priority of a cluster to not apply any alarms
	NoAlarms AlarmPriority = iota
	//MinimumAlarms indicates alarms that should be monitored under every circumstance, like CPU utilization & memory
	MinimumAlarms AlarmPriority = iota
	//ModerateAlarms indicates alarms that have a high amount of signal for DB outages, such as IOPS
	ModerateAlarms AlarmPriority = iota
	//HighAlarms indicates alarms that don't have false positives but may not always be helpful in finding problem states
	HighAlarms AlarmPriority = iota
	//VerboseAlarms indicates alarms that are inappropriate for some databases because of false positives, such as latency alarms (large recently-cloned staging DB's frequently have really bad latencies)
	VerboseAlarms AlarmPriority = iota
)

func (alarm AlarmPriority) String() string {
	switch alarm {
	case NoAlarms:
		return "noalarms"
	case MinimumAlarms:
		return "minimal"
	case ModerateAlarms:
		return "moderate"
	case HighAlarms:
		return "high"
	case VerboseAlarms:
		return "verbose"
	default:
		return "unknown"
	}
}

func addAlarm(priority AlarmPriority, metricName string, comparisonOperator string, threshold float64, description string) {
	requestedAlarms[metricName] = &AlarmConfig{
		metricName:         metricName,
		comparisonOperator: comparisonOperator,
		period:             60,
		evaluationPeriods:  3,
		statistic:          aws.String("Average"),
		threshold:          &ConstantThresholdCalculator{threshold: threshold},
		alarmDescription:   description,
		alarmPriority:      priority,
	}
}

func addAdvancedAlarm(priority AlarmPriority, metricName string, comparisonOperator string, threshold float64, description string, period int64, evaluationPeriods int64, statistic *string, extendedStatistic *string) {

	requestedAlarms[alarmName(metricName, extendedStatistic)] = &AlarmConfig{
		metricName:         metricName,
		comparisonOperator: comparisonOperator,
		period:             period,
		evaluationPeriods:  evaluationPeriods,
		statistic:          statistic,
		threshold:          &ConstantThresholdCalculator{threshold: threshold},
		alarmDescription:   description,
		extendedStatistic:  extendedStatistic,
		alarmPriority:      priority,
	}
}

func addPercentAlarm(priority AlarmPriority, metricName string, comparisonOperator string, metric string, percent float64, minValue float64, description string) {
	requestedAlarms[metricName] = &AlarmConfig{
		metricName:         metricName,
		comparisonOperator: comparisonOperator,
		period:             60,
		evaluationPeriods:  3,
		statistic:          aws.String("Average"),
		threshold:          &MetricPercentThresholdCalculator{Metric: metric, Percent: percent, MinValue: minValue},
		alarmDescription:   description,
		alarmPriority:      priority,
	}
}

type ThresholdCalculator interface {
	Threshold(instance *rds.DBInstance) (float64, error)
}

type AlarmConfig struct {
	metricName         string
	comparisonOperator string
	period             int64
	evaluationPeriods  int64
	statistic          *string
	alarmDescription   string
	extendedStatistic  *string

	threshold ThresholdCalculator

	alarmPriority AlarmPriority
}

func (cfg *AlarmConfig) Matches(alarm *cloudwatch.MetricAlarm, instance *rds.DBInstance) (bool, error) {

	if alarm == nil {
		return false, nil
	}

	if alarm.MetricName == nil || alarm.ComparisonOperator == nil || alarm.Period == nil || alarm.EvaluationPeriods == nil ||
		alarm.AlarmDescription == nil || alarm.Threshold == nil {
		return false, nil
	}

	if (alarm.ExtendedStatistic == nil) != (cfg.ExtendedStatistic() == nil) || (alarm.ExtendedStatistic != nil && *alarm.ExtendedStatistic != *cfg.ExtendedStatistic()) {
		return false, nil
	}

	if (alarm.Statistic == nil) != (cfg.Statistic() == nil) || (alarm.Statistic != nil && *alarm.Statistic != *cfg.Statistic()) {
		return false, nil
	}

	a := *alarm
	threshold, err := cfg.Threshold(instance)
	if err != nil {
		return false, err
	}

	return cfg.MetricName() == *a.MetricName && cfg.ComparisonOperator() == *a.ComparisonOperator && cfg.Period() == *a.Period &&
		cfg.EvaluationPeriods() == *a.EvaluationPeriods && cfg.Description(instance) == *a.AlarmDescription &&
		threshold == *a.Threshold, nil
}

func (cfg *AlarmConfig) MetricName() string {
	return cfg.metricName
}

func (cfg *AlarmConfig) ComparisonOperator() string {
	return cfg.comparisonOperator
}

func (cfg *AlarmConfig) Period() int64 {
	return cfg.period
}

func (cfg *AlarmConfig) EvaluationPeriods() int64 {
	return cfg.evaluationPeriods
}

func (cfg *AlarmConfig) Statistic() *string {
	return cfg.statistic
}

func (cfg *AlarmConfig) ExtendedStatistic() *string {
	return cfg.extendedStatistic
}

func (cfg *AlarmConfig) Description(instance *rds.DBInstance) string {
	return fmt.Sprintf(cfg.alarmDescription, *instance.DBInstanceIdentifier)
}

func (cfg *AlarmConfig) Threshold(instance *rds.DBInstance) (float64, error) {
	return cfg.threshold.Threshold(instance)
}

func (cfg *AlarmConfig) Priority() AlarmPriority {
	return cfg.alarmPriority
}

type ConstantThresholdCalculator struct {
	threshold float64
}

func (calc *ConstantThresholdCalculator) Threshold(instance *rds.DBInstance) (float64, error) {
	return calc.threshold, nil
}

type MetricPercentThresholdCalculator struct {
	Metric   string
	Percent  float64
	MinValue float64
}

func (calc *MetricPercentThresholdCalculator) Threshold(instance *rds.DBInstance) (float64, error) {
	value, err := getMetricValue(instance, calc.Metric)
	value *= calc.Percent
	if value < calc.MinValue {
		value = calc.MinValue
	}

	return value, err
}

func getMetricValue(instance *rds.DBInstance, metric string) (float64, error) {
	switch metric {
	case "memory":
		return getMemory(instance)
	case "disk":
		return getDiskSpace(instance)
	case "iops":
		return getIops(instance)
	default:
		return 0, fmt.Errorf("Unknown percent metric '%s'.", metric)
	}
}

func getMemory(instance *rds.DBInstance) (float64, error) {
	if instance.DBInstanceClass == nil {
		return 0, fmt.Errorf("Cannot build a memory-derived threshold for %s: instance class is nil", *instance.DBInstanceIdentifier)
	}

	memory, ok := instanceMemory[*instance.DBInstanceClass]
	if !ok {
		return 0, fmt.Errorf("Cannot build a memory-derived threshold for %s: could not retrieve memory value for instance class %s, because the code has never heard of it", *instance.DBInstanceIdentifier, *instance.DBInstanceClass)
	}

	return memory, nil
}

func getDiskSpace(instance *rds.DBInstance) (float64, error) {
	if instance.AllocatedStorage == nil {
		return 0, fmt.Errorf("Cannot build a disk-derived threshold for %s: allocated storage value is nil", *instance.DBInstanceIdentifier)
	}

	return float64(*instance.AllocatedStorage), nil
}

func getIops(instance *rds.DBInstance) (float64, error) {
	if instance.Iops != nil {
		return float64(*instance.Iops), nil
	}

	//If no PIOPS, then we're using something that can do 3 IOPS for 1 GB and burst up to 10:1
	//Since cloudwatch buckets are 60s, we can pretty easily treat 3:1 as the max,
	//since if you're sitting near 3:1 over the course of a minute, you're probably
	//capped out
	disk, err := getDiskSpace(instance)
	if err != nil {
		return 0, err
	}

	diskGb := disk
	return diskGb * 3, nil
}
