package awsutil

import (
	"fmt"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/autoscaling"
	"github.com/aws/aws-sdk-go/service/autoscaling/autoscalingiface"
	"github.com/aws/aws-sdk-go/service/ec2"
	"github.com/aws/aws-sdk-go/service/ec2/ec2iface"
)

type Metadata interface {
	GetMetadata(path string) (string, error)
}

func TerminateInstance(ec2Client ec2iface.EC2API, instance *ec2.Instance) error {
	_, err := ec2Client.TerminateInstances(&ec2.TerminateInstancesInput{
		InstanceIds: []*string{instance.InstanceId},
	})
	return err
}

func FetchInstance(ec2Client ec2iface.EC2API, name string, instanceState string) (*ec2.Instance, error) {
	resp, err := ec2Client.DescribeInstances(&ec2.DescribeInstancesInput{
		Filters: []*ec2.Filter{
			{
				Name: aws.String("tag:Name"),
				Values: []*string{aws.String(name)},
			},
			{
				Name: aws.String("instance-state-name"),
				Values: []*string{aws.String(instanceState)},
			},
		},
	})
	if err != nil {
		return nil, err
	}

	couldNotFind := fmt.Errorf(fmt.Sprintf("Couldn't find the instance with tag:Name: %s", name))
	if len(resp.Reservations) < 1 {
		return nil, couldNotFind
	}

	if len(resp.Reservations[0].Instances) < 1 {
		return nil, couldNotFind
	}

	return resp.Reservations[0].Instances[0], nil

}

func FetchLocalInstance(ec2Client ec2iface.EC2API, metadataClient Metadata) (*ec2.Instance, error) {
	instanceId, err := metadataClient.GetMetadata("instance-id")
	if err != nil {
		return nil, err
	}

	resp, err := ec2Client.DescribeInstances(&ec2.DescribeInstancesInput{
		InstanceIds: []*string{
			aws.String(instanceId),
		},
	})
	if err != nil {
		return nil, err
	}

	couldNotFind := fmt.Errorf("Couldn't find the local instance using the local metadata.  Is something wrong?")
	if len(resp.Reservations) < 1 {
		return nil, couldNotFind
	}

	if len(resp.Reservations[0].Instances) < 1 {
		return nil, couldNotFind
	}

	return resp.Reservations[0].Instances[0], nil
}

func FetchASG(autoscalingClient autoscalingiface.AutoScalingAPI, instanceId *string) (*autoscaling.Group, error) {
	var instanceInfo *autoscaling.InstanceDetails
	instanceResp, err := autoscalingClient.DescribeAutoScalingInstances(&autoscaling.DescribeAutoScalingInstancesInput{
		InstanceIds: []*string{instanceId},
	})

	if err != nil {
		return nil, err
	}

	if instanceResp != nil && len(instanceResp.AutoScalingInstances) > 0 {
		instanceInfo = instanceResp.AutoScalingInstances[0]
	} else {
		return nil, fmt.Errorf("Couldn't find instance %s in any ASG.", *instanceId)
	}

	var asg *autoscaling.Group
	groupResp, err := autoscalingClient.DescribeAutoScalingGroups(&autoscaling.DescribeAutoScalingGroupsInput{
		AutoScalingGroupNames: []*string{instanceInfo.AutoScalingGroupName},
	})

	if groupResp != nil && len(groupResp.AutoScalingGroups) > 0 {
		asg = groupResp.AutoScalingGroups[0]
	}

	return asg, err
}
