package clusterdb

import (
	"database/sql"
	"fmt"

	"code.justin.tv/d8a/buddy/lib/config"
	"code.justin.tv/d8a/buddy/lib/sandstorm"
)

type ClusterDB struct {
	conn          *sql.DB
	driverQueries DriverQueries
}

// OpenDbConn opens a database connection to the given managed cluster, logged in as the given user
func OpenDbConn(cluster *config.Cluster, sandstormClient sandstorm.SandstormAPI, user string) (*ClusterDB, error) {
	password, err := sandstormClient.GetUserPassword(cluster, user)
	if err != nil {
		return nil, err
	}

	return OpenDbConnWithManualSecret(cluster, user, password)
}

// OpenDbConnWithManualSecret works like OpenDbConn but instead of accepting a sandstorm client which it uses
// to retrieve the password, it just accepts the password
func OpenDbConnWithManualSecret(cluster *config.Cluster, user string, password string) (*ClusterDB, error) {
	dbname := cluster.Database
	host := cluster.Host
	port := cluster.Port

	var dbURL string
	var driverQueryFactory DriverQueryFactory
	switch cluster.Driver {
	case "postgres":
		dbURL = fmt.Sprintf("user=%s dbname=%s host=%s port=%d sslmode=disable password=%s", user, dbname, host, port, password)
		driverQueryFactory = &PostgresQueriesFactory{}
		break
	case "mysql":
		dbURL = fmt.Sprintf("%s:%s@(%s:%d)/%s", user, password, host, port, dbname)
		driverQueryFactory = &MySQLQueriesFactory{}
		break
	}

	conn, err := sql.Open(cluster.Driver, dbURL)
	if err != nil {
		return nil, err
	}

	queries, err := driverQueryFactory.BuildDriverQueries(conn)
	if err != nil {
		return nil, err
	}

	clusterDb := &ClusterDB{
		conn:          conn,
		driverQueries: queries,
	}

	return clusterDb, clusterDb.driverQueries.TestConnection(conn)
}

// ClusterUsers retrieves all users in the cluster DB, as a map of username -> LOGIN/NOLOGIN.  If true, LOGIN, if false NOLOGIN
func (clusterDb *ClusterDB) ClusterUsers() (map[string]bool, error) {
	users := make(map[string]bool)
	rows, err := clusterDb.driverQueries.ClusterUsers(clusterDb.conn)
	if err != nil {
		return users, err
	}

	for rows.Next() {
		var username string
		var canlogin bool
		err = rows.Scan(&username, &canlogin)
		if err != nil {
			return users, err
		}
		users[username] = canlogin
	}

	return users, nil
}

// GrantedRoles returns a list of role names who have grants in the cluster DB
func (clusterDb *ClusterDB) GrantedRoles() ([]string, error) {
	var roles []string
	rows, err := clusterDb.driverQueries.GrantedRoles(clusterDb.conn)
	if err != nil {
		return roles, err
	}

	for rows.Next() {
		var username string
		err = rows.Scan(&username)
		if err != nil {
			return roles, err
		}
		roles = append(roles, username)
	}

	return roles, err
}

// ActiveUsers returns a list of users found in pg_stat_activity
func (clusterDb *ClusterDB) ActiveUsers() ([]string, error) {
	var users []string
	rows, err := clusterDb.driverQueries.ActiveUsers(clusterDb.conn)
	if err != nil {
		return users, err
	}

	for rows.Next() {
		var username string
		err = rows.Scan(&username)
		if err != nil {
			return users, err
		}
		users = append(users, username)
	}

	return users, err
}

// RoleParents returns a list of role inheritance relationships in the clusterDb, as a map
// of role name -> list of member role names
func (clusterDb *ClusterDB) RoleParents() (map[string][]string, error) {
	parents := make(map[string][]string)
	rows, err := clusterDb.driverQueries.RoleParents(clusterDb.conn)
	if err != nil {
		return parents, err
	}

	for rows.Next() {
		var role string
		var user string
		err = rows.Scan(&role, &user)
		if err != nil {
			return parents, err
		}

		allRoles, ok := parents[user]
		if !ok {
			allRoles = make([]string, 0)
		}
		allRoles = append(allRoles, role)
		parents[user] = allRoles
	}

	return parents, err
}

// CreateUser makes a new DB user in the cluster db, with password and an inherited role
func (clusterDb *ClusterDB) CreateUser(rolename string, parentRole string, password string) error {
	_, err := clusterDb.driverQueries.CreateUser(clusterDb.conn, rolename, password, parentRole)
	return err
}

// CreateRole makes a new blank DB role in the cluster db
func (clusterDb *ClusterDB) CreateRole(roleName string) error {
	_, err := clusterDb.driverQueries.CreateRole(clusterDb.conn, roleName)
	return err
}

// SetUserLogin sets the DB user to LOGIN or NOLOGIN in the cluster db
func (clusterDb *ClusterDB) SetUserLogin(username string, userLogin bool) error {
	_, err := clusterDb.driverQueries.SetUserLock(clusterDb.conn, username, !userLogin)
	return err
}

// SetUserPassword modifies the DB user's password in the clusterDB
func (clusterDb *ClusterDB) SetUserPassword(username string, password string) error {
	_, err := clusterDb.driverQueries.AlterPassword(clusterDb.conn, username, password)
	return err
}

func (clusterDb *ClusterDB) CanUseRolePairs() bool {
	return clusterDb.driverQueries.CanUseRolePairs()
}

func (clusterDb *ClusterDB) CanLockUsers() bool {
	return clusterDb.driverQueries.CanLockUsers()
}
