package clusters

import (
	"errors"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/rds"
	"github.com/aws/aws-sdk-go/service/rds/rdsiface"
	multierror "github.com/hashicorp/go-multierror"

	"code.justin.tv/d8a/buddy/lib/awsutil"
	"code.justin.tv/d8a/buddy/lib/config"
)

// FetchMeta retrieves cluster metadata from the AWS API and populates it into RDSInfo
func FetchMeta(clusters config.ClusterList, client rdsiface.RDSAPI) (map[string]*rds.DBInstance, map[string]*rds.DBCluster, error) {
	rdsClusters := make(map[string]*rds.DBCluster)
	rdsInstances := make(map[string]*rds.DBInstance)
	outErrors := new(multierror.Error)

	if len(clusters) == 0 {
		return rdsInstances, rdsClusters, outErrors
	}

	//Grab first available client - if it's not there, something's gone terribly wrong so we can probably panic?
	if client == nil {
		outErrors = multierror.Append(outErrors, errors.New("The masterinstance client had not been initialized yet- probably FetchMeta is being called on an uninitialized cluster list?"))
		return rdsInstances, rdsClusters, outErrors
	}

	var masterClusters []*string
	var masterInstances []*string
	for _, cluster := range clusters {
		if cluster.IsAurora {
			masterClusters = append(masterClusters, aws.String(cluster.RootIdentifier))
		} else {
			masterInstances = append(masterInstances, aws.String(cluster.RootIdentifier))
		}
	}

	var fetchError error
	if len(masterClusters) > 0 {
		var auroraInstances []*string

		rdsClusters, auroraInstances, fetchError = fetchClusterData(client, masterClusters)
		masterInstances = append(masterInstances, auroraInstances...)
		outErrors = multierror.Append(outErrors, fetchError)
	}

	if len(masterInstances) > 0 {
		rdsInstances, fetchError = fetchInstanceData(client, masterInstances)
		outErrors = multierror.Append(outErrors, fetchError)
	}

	// And... that's it!
	return rdsInstances, rdsClusters, outErrors.ErrorOrNil()
}

// fetchClusterData will fetch RDS metadata for all cluster identifiers passed in
func fetchClusterData(client rdsiface.RDSAPI, clusterIds []*string) (map[string]*rds.DBCluster, []*string, error) {
	clusterMap := make(map[string]*rds.DBCluster)
	var clusterInstances []*string
	errors := new(multierror.Error)

	clusterObjs, err := awsutil.FetchClusters(client, clusterIds)
	if err != nil {
		errors = multierror.Append(errors, err)
	}

	var replicaClusterIds []*string
	for _, clusterObj := range clusterObjs {
		clusterMap[*clusterObj.DBClusterIdentifier] = clusterObj
		for _, replicaCluster := range clusterObj.ReadReplicaIdentifiers {
			replicaClusterIds = append(replicaClusterIds, replicaCluster)
		}
		for _, clusterMember := range clusterObj.DBClusterMembers {
			clusterInstances = append(clusterInstances, clusterMember.DBInstanceIdentifier)
		}
	}

	if len(replicaClusterIds) > 0 {
		subClusters, subReplicaIds, subErrors := fetchClusterData(client, replicaClusterIds)

		for subClusterName, subCluster := range subClusters {
			clusterMap[subClusterName] = subCluster
		}
		clusterInstances = append(clusterInstances, subReplicaIds...)
		errors = multierror.Append(errors, subErrors)
	}

	return clusterMap, clusterInstances, errors
}

func fetchInstanceData(client rdsiface.RDSAPI, instanceIds []*string) (map[string]*rds.DBInstance, error) {
	errors := new(multierror.Error)
	instanceMap := make(map[string]*rds.DBInstance)

	instanceObjs, err := awsutil.FetchInstancesByIds(client, instanceIds)
	if err != nil {
		errors = multierror.Append(errors, err)
	}

	var replicaIds []string
	for _, instanceObj := range instanceObjs {
		instanceMap[*instanceObj.DBInstanceIdentifier] = instanceObj
		for _, replicaID := range instanceObj.ReadReplicaDBInstanceIdentifiers {
			replicaIds = append(replicaIds, *replicaID)
		}
	}

	var subFetchIds []*string
	for _, replicaID := range replicaIds {
		_, ok := instanceMap[replicaID]
		if !ok {
			subFetchIds = append(subFetchIds, aws.String(replicaID))
		}
	}

	if len(subFetchIds) > 0 {
		subInstances, subErrors := fetchInstanceData(client, subFetchIds)

		for subInstanceName, subInstance := range subInstances {
			instanceMap[subInstanceName] = subInstance
		}
		errors = multierror.Append(errors, subErrors)
	}

	return instanceMap, errors
}
