package config

import (
	"io/ioutil"
	"os"
	"path/filepath"

	"github.com/naoina/toml"
)

// ConfigFile is a struct representing the contents of buddy.cfg, as parsed from toml
type ConfigFile struct {
	StoreIdentifier string         // This is only used in debug environments so we can find the buddy datastore
	SandstormTeam   string         // This is the token used to identify the owning team in sandstorm key paths & AWS resource identifiers
	GithubKey       string         // This is the OAuth2 token buddy uses to operate in github enterprise
	GithubTeam      string         // This is the github org that the owning team wants to use by default
	SlackRoom       string         // This is the owning team's default slack room where they'd like to receive deploy/build messages
	SandstormRole   string         // This is the sandstorm role buddy should use to communicate with the sandstorm API
	Environment     string         // This is the box's environment (staging, production, etc.)
	Cluster         ClusterList    `toml:"-"` // List of buddy clusters to manage
	BulkSchedule    []BulkSchedule // The list of cron schedules that bulk operations will trigger on if one is not already running
	StopBulkOps     bool           // This is the flag set to stop bulk operations default to false to allow bulk
}

// ReadConfig is a method that will build a ConfigFile struct from the contents of a toml file
// located at the given path.
func ReadConfig(filePath string) (*ConfigFile, error) {
	data, err := ioutil.ReadFile(filePath)
	if err != nil {
		return nil, err
	}

	configFile := &ConfigFile{}
	err = toml.Unmarshal(data, configFile)

	return configFile, err
}

// FindConfigFile is a method that retrieves the absolute path of the best buddy config file to use
// this method returns a blank path if a config file could not be found
func FindConfigFile() string {
	localCfg, absErr := filepath.Abs("./buddy.cfg")
	etcCfg := "/etc/buddy/buddy.cfg"

	if _, err := os.Stat(localCfg); absErr == nil && err == nil {
		return localCfg
	}

	if _, err := os.Stat(etcCfg); err == nil {
		return etcCfg
	}

	return ""
}

// GetCluster is a method that retrieves the buddy Cluster struct that has the
// given cluster name.  This method returns nil if no cluster with the given name exists
func (config *ConfigFile) GetCluster(name string) *Cluster {
	return config.Cluster.GetCluster(name)
}
