package config

import (
	"fmt"
	"path/filepath"
	"strconv"

	"code.justin.tv/d8a/buddy/lib/alerts"
	"code.justin.tv/d8a/iceman/lib/dbconf"
)

const GitRoot string = "/etc/buddy/iceman/"

type User struct {
	Id       int
	Name     string
	Secret   string
	RolePair *RolePair
}

type RolePair struct {
	Id           int
	Name         string
	RoleServices []*RoleService
	Users        []*User
}

type RoleService struct {
	Id             int
	Name           string
	UsernameSecret string
	PasswordSecret string
	RolePair       *RolePair
}

// Cluster is a struct that represents a single cluster of databases that buddy manages
// Generally it contains necessary login information for the master instance, a list of
// users with sandstorm data, and optionally some RDS metadata
type Cluster struct {
	Id               int
	Name             string // The name of this cluster
	Driver           string // The driver used to access the cluster, such as postgres/mysql/etc.
	Environment      string // The environment for the cluster, such as staging/production/etc.
	Host             string // Connection endpoint host
	Port             int    // Connection endpoint port
	Database         string // Main database (the one specified at RDS provision)
	Schema           string // The schema iceman should use when running (usually left blank, indicating the default schema)
	SuperUser        string // The RDS master user (the one specified at RDS provision)
	Repository       string // The iceman github repository for this cluster
	AlarmPriorityStr string //We use this string as a parse fixture for alarm priority so we can set a default value
	IsAurora         bool   // True if this is a cluster-based topology
	RootIdentifier   string // The identifier for the master instance if not aurora, or identifier of master cluster
	User             []*User
	RolePair         []*RolePair

	AlarmPriority alerts.AlarmPriority
}

// ClusterList is a type for array of clusters- the type exists so we
// can attach methods to it
type ClusterList []*Cluster

// GetUser retrieves a user from a username or returns nil
func (cluster *Cluster) GetUser(username string) *User {
	var foundUser *User

	for _, user := range cluster.User {
		if user.Name == username {
			foundUser = user
			break
		}
	}

	return foundUser
}

func (cluster *Cluster) GetRolePair(rolename string) *RolePair {
	var foundPair *RolePair

	for _, pair := range cluster.RolePair {
		if pair.Name == rolename {
			foundPair = pair
			break
		}
	}

	return foundPair
}

func (rolePair *RolePair) GetService(servicename string) *RoleService {
	var foundService *RoleService

	for _, service := range rolePair.RoleServices {
		if service.Name == servicename {
			foundService = service
			break
		}
	}

	return foundService
}

// RepoFolder retrieves the local file path to the folder where this cluster's iceman
// repository will be cloned to
func (cluster *Cluster) RepoFolder() string {
	return filepath.Clean(GitRoot + cluster.Repository + "/")
}

// GetDbConfMap retrieves a string -> string map used to build the dbconf in the iceman repository for this
// cluster
func (cluster *Cluster) GetDbConfMap() map[string]string {
	out := make(map[string]string)
	out["user"] = cluster.SuperUser
	out["dbname"] = cluster.Database
	out["host"] = cluster.Host
	out["port"] = strconv.Itoa(cluster.Port)
	out["sslmode"] = "disabled"
	return out
}

// GetIcemanDbConf builds an iceman DBConf object from the file system, or, if it is not
// present, builds one from scratch.
func (cluster *Cluster) GetIcemanDbConf(confType string) (*dbconf.DBConf, error) {
	return dbconf.NewDBConf(filepath.Join(cluster.RepoFolder(), "iceman"), cluster.Environment, cluster.Schema, confType)
}

type parseableCluster Cluster

func (cluster *Cluster) ParseAlarmPriority() error {
	if cluster.AlarmPriorityStr == "" {
		cluster.AlarmPriorityStr = "verbose"
	}

	switch cluster.AlarmPriorityStr {
	case "noalarms", "0":
		cluster.AlarmPriority = alerts.NoAlarms
		return nil
	case "minimum", "1":
		cluster.AlarmPriority = alerts.MinimumAlarms
		return nil
	case "moderate", "2":
		cluster.AlarmPriority = alerts.ModerateAlarms
		return nil
	case "high", "3":
		cluster.AlarmPriority = alerts.HighAlarms
		return nil
	case "verbose", "4":
		cluster.AlarmPriority = alerts.VerboseAlarms
		return nil
	default:
		return fmt.Errorf("Could not parse alarm priority '%s'", cluster.AlarmPriorityStr)
	}
}

func (cluster *Cluster) Equals(otherCluster *Cluster) ([]string, bool) {
	if cluster == nil || otherCluster == nil {
		return []string{"existence"}, false
	}

	var differences []string
	if cluster.Name != otherCluster.Name {
		differences = append(differences, fmt.Sprintf("Name: '%s' != '%s'", cluster.Name, otherCluster.Name))
	}
	if cluster.Driver != otherCluster.Driver {
		differences = append(differences, fmt.Sprintf("Driver: '%s' != '%s'", cluster.Driver, otherCluster.Driver))
	}
	if cluster.Environment != otherCluster.Environment {
		differences = append(differences, fmt.Sprintf("Environment: '%s' != '%s'", cluster.Environment, otherCluster.Environment))
	}
	if cluster.Host != otherCluster.Host {
		differences = append(differences, fmt.Sprintf("Host: '%s' != '%s'", cluster.Host, otherCluster.Host))
	}
	if cluster.Port != otherCluster.Port {
		differences = append(differences, fmt.Sprintf("Port: %d != %d", cluster.Port, otherCluster.Port))
	}
	if cluster.Schema != otherCluster.Schema {
		differences = append(differences, fmt.Sprintf("Schema: '%s' != '%s'", cluster.Schema, otherCluster.Schema))
	}
	if cluster.SuperUser != otherCluster.SuperUser {
		differences = append(differences, fmt.Sprintf("SuperUser: '%s' != '%s'", cluster.SuperUser, otherCluster.SuperUser))
	}
	if cluster.IsAurora != otherCluster.IsAurora {
		differences = append(differences, fmt.Sprintf("IsAurora: %t != %t", cluster.IsAurora, otherCluster.IsAurora))
	}
	if cluster.RootIdentifier != otherCluster.RootIdentifier {
		differences = append(differences, fmt.Sprintf("RootIdentifier: '%s' != '%s'", cluster.RootIdentifier, otherCluster.RootIdentifier))
	}

	return differences, len(differences) == 0
}

func (clusterList ClusterList) GetCluster(name string) *Cluster {
	for _, cluster := range clusterList {
		if cluster.Name == name {
			return cluster
		}
	}

	return nil
}
