package store

import (
	"database/sql"
	"fmt"

	"github.com/aws/aws-sdk-go/service/rds/rdsiface"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/rds"
	"github.com/pkg/errors"

	"code.justin.tv/d8a/buddy/lib/sandstorm"
)

// OpenDbConn will open a DB connection to the buddy store from an rds.DBInstance object retrieved from GetBuddyStore
func OpenDbConn(dbInstance *rds.DBInstance, sandstormClient sandstorm.SandstormAPI) (*sql.DB, error) {
	password, err := sandstormClient.GetStorePassword()
	if err != nil {
		return nil, err
	}

	user := *dbInstance.MasterUsername
	dbname := *dbInstance.DBName
	host := *dbInstance.Endpoint.Address
	port := int(*dbInstance.Endpoint.Port)

	dbURL := fmt.Sprintf("user=%s dbname=%s host=%s port=%d sslmode=disable password=%s", user, dbname, host, port, password)
	return sql.Open("postgres", dbURL)
}

func RefreshSuperUserPassword(rdsClient rdsiface.RDSAPI, dbInstance *rds.DBInstance, sandstormClient sandstorm.SandstormAPI) error {
	var err error

	password, err := sandstormClient.GetStorePassword()
	if err != nil {
		return err
	}

	_, err = rdsClient.ModifyDBInstance(&rds.ModifyDBInstanceInput{
		DBInstanceIdentifier: dbInstance.DBInstanceIdentifier,
		MasterUserPassword:   aws.String(password),
	})

	if err != nil {
		return errors.Wrap(err, "could not change the superuser password of buddy store")
	}

	return nil
}
