provider "aws" {
  profile = "${var.provider_profile}"
  region = "{{.Region}}"
}

variable "provider_profile" { default = "{{.InfraProfile}}" }
variable "service" { default = "{{.Service}}" }
variable "environment" { default = "{{.Environment}}" }
variable "region" { default = "{{.Region}}" }
variable "owner" { default = "{{.Owner}}" }
variable "data_count" { default = "{{.WorkerCount}}" }
variable "ebs_size" { default = "{{.EbsSize}}" }
variable "ebs_count" { default = "{{.WorkerCount}}" }
variable "vpc_id" { default = "{{.VpcId}}" }
variable "zone_id" { default = "{{.ZoneId}}" }
variable "instance_type" { default =  "{{.InstanceType}}" }
variable "ami" { default = "{{.Ami}}" }
variable "rds_identifier" { default ="{{.RdsIdentifier}}" }
variable "rds_username" { default ="{{.RdsUsername}}" }
variable "rds_password" { default ="{{.RdsPassword}}" }
variable "rds_subnet_group_name" { default ="{{.RdsSubnetGroupName}}" }
variable "subnet_ids" {
    type = "list"
    default = ["{{.SubnetIds}}"]
}

variable "subnet_cidrs" {
  type = "list"
  default = ["{{.SubnetCidrs}}"]
}

variable "azs" {
  type = "list"
  default = ["{{.AvailabilityZones}}"]
}


module "workers" {
  source                  = "git::git+ssh://git@git-aws.internal.justin.tv/terraform-modules/aws-instance-no-profile.git?ref=v1.1"
  account                 = "${var.provider_profile}"
  region                  = "${var.region}"
  environment             = "${var.environment}"
  vpc_security_group_ids = ["${aws_security_group.zenyatta-sg.id}"]
  subnet_ids              = ["${var.subnet_ids}"]
  service                 = "${var.service}"
  role                    = "${var.service}-airflow-worker"
  ami                     = "ami-a66dccc6"
  instance_type           = "${var.instance_type}"
  puppet                  = "puppet"
  count                   = "${var.data_count}"
  owner                   = "${var.owner}"

  instance_profile = "${aws_iam_instance_profile.zenyatta_profile.id}"
}

# RDS
resource "aws_db_instance" "postgres" {
  identifier = "${var.rds_identifier}"
  allocated_storage = "100" #space in gigs
  backup_retention_period = "7"
  engine = "postgres"
  instance_class = "db.m4.xlarge"
  name = "airflow"
  username = "${var.rds_username}"
  password = "${var.rds_password}"
  multi_az = true
  db_subnet_group_name = "${var.rds_subnet_group_name}"
  vpc_security_group_ids = ["${aws_security_group.zenyatta-sg.id}"]
  port = "5432"
  tags {
    Name = "${var.service}-${var.environment}"
    Environment = "${var.environment}"
    Owner = "${var.owner}@twitch.tv"
    Service = "${var.owner}/${var.service}"
  }
}

resource "aws_security_group" "zenyatta-sg" {
  name = "${var.service}-${var.environment}"
  description = "security group for redis, postgres, and celery"
  vpc_id = "${var.vpc_id}"

  egress {
    from_port = 0
    to_port = 0
    protocol = -1
    cidr_blocks = ["0.0.0.0/0"]
  }
  ingress {
    from_port = 22
    to_port = 22
    protocol = "tcp"
    cidr_blocks = "${var.subnet_cidrs}"
  }
  ingress { # airflow worker port
    from_port = 8793
    to_port = 8793
    protocol = "tcp"
    cidr_blocks = "${var.subnet_cidrs}"
  }
  ingress { # redis port
    from_port = 6379
    to_port = 6379
    protocol = "tcp"
    cidr_blocks = "${var.subnet_cidrs}"
  }
  ingress { # default webserver port
    from_port = 8080
    to_port = 8080
    protocol = "tcp"
    cidr_blocks = "${var.subnet_cidrs}"
  }
  ingress { # rds port
    from_port = 5432
    to_port = 5432
    protocol = "tcp"
    cidr_blocks = "${var.subnet_cidrs}"
  }
  ingress {
    # required for nagios monitoring
    from_port = 5666
    to_port = 5666
    protocol = "tcp"
    cidr_blocks = "${var.subnet_cidrs}"
  }
  ingress { # celery ui port
    from_port = 5555
    to_port = 5555
    protocol = "tcp"
    cidr_blocks = "${var.subnet_cidrs}"
  }
  tags {
    Name = "${var.service}-${var.environment}"
    Owner = "${var.owner}@twitch.tv"
  }
}

# REDIS
resource "aws_elasticache_subnet_group" "redis_subnet" {
  name = "${var.service}"
  description = "subnet for ${var.service}"
  subnet_ids = "${var.subnet_ids}"
}

resource "aws_elasticache_cluster" "redis" {
  cluster_id = "${var.service}-${var.environment}"
  engine = "redis"
  engine_version = "2.8.24"
  node_type = "cache.m3.medium"
  port = "6379"
  num_cache_nodes = "1"
  parameter_group_name  = "default.redis2.8"
  security_group_ids = ["${aws_security_group.zenyatta-sg.id}"]
  subnet_group_name = "${aws_elasticache_subnet_group.redis_subnet.name}"

  tags {
    Name = "${var.service}-${var.environment}"
    Environment = "${var.environment}"
    Owner = "${var.owner}@twitch.tv"
    Service = "${var.owner}/${var.service}"
  }
}

# manual part

# Create some EBS volumes

# Attach the EBS volumes (AZ A)
resource "aws_ebs_volume" "data" {
  availability_zone = "us-west-2a"
  size = "${var.ebs_size}"
  type = "gp2"
  count = "${var.data_count}"
}

resource "aws_volume_attachment" "data" {
  device_name = "/dev/xvdb"
  volume_id = "${element(aws_ebs_volume.data.*.id, count.index)}"
  instance_id = "${element(module.workers.instance_ids, count.index)}"
  count = "${var.data_count}"
}

resource "aws_iam_instance_profile" "zenyatta_profile" {
  name = "${var.service}-${var.environment}"
  roles = ["${aws_iam_role.airflow.id}"]
}

# MASTER ROLE
resource "aws_iam_role" "airflow" {
  name = "${var.service}-airflow"
  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "Service": "ec2.amazonaws.com"
      },
      "Effect": "Allow",
      "Sid": ""
    }
  ]
}
EOF
}

resource "aws_iam_role_policy" "d8a_airflow_policy" {
    name = "${var.service}-zenyatta-access"
    role = "${aws_iam_role.airflow.id}"
    policy = <<EOF
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Effect": "Allow",
            "Action": [
                "cloudwatch:*",
                "autoscaling:Describe",
                "logs:*",
                "ecr:*",
                "sns:*"
            ],
            "Resource": "*"
        },
        {
            "Effect": "Allow",
            "Action": [
                "s3:Get*",
                "s3:Put*",
                "s3:List*"
            ],
            "Resource": [
              {{.S3Arns}}
            ]
        },
        {
            "Effect": "Deny",
            "Action": [
                "cloudwatch:Delete*",
                "logs:Delete*",
                "sns:Delete*"
            ],
            "Resource": "*"
        },
        {
            "Effect": "Allow",
            "Action": "sts:AssumeRole",
            "Resource": [
            "{{.SandstormRole}}"
            ]
        }
    ]
}
EOF
}
