resource "aws_codedeploy_app" "app" {
    name = "deploy-${var.owner}-${var.service}-${var.environment}"
}

data "aws_iam_policy_document" "codedeploy_service_assume_policy_doc" {
    statement {
        actions = ["sts:AssumeRole"]

        principals {
            type = "Service"
            identifiers = ["codedeploy.amazonaws.com"]
        }
    }
}

data "aws_iam_policy_document" "codedeploy_service_policy_doc" {
    statement {
        actions = [
            "autoscaling:PutLifecycleHook",
            "autoscaling:DeleteLifecycleHook",
            "autoscaling:RecordLifeCycleActionHeartbeat",
            "autoscaling:CompleteLifecycleAction",
            "autoscaling:DescribeAutoscalingGroups",
            "autoscaling:EnterStandby",
            "autoscaling:ExitStandby",
            "autoscaling:DescribeLifecycleHooks"
        ]

        resources = ["*"]
    }

    statement {
        actions = ["ec2:Describe*"]
        resources = ["*"]
    }
}

resource "aws_iam_policy" "codedeploy_service_policy" {
    policy = "${data.aws_iam_policy_document.codedeploy_service_policy_doc.json}"
}

resource "aws_iam_role" "codedeploy_service_role" {
    name = "codedeploy-service-${var.owner}-${var.service}-${var.environment}"
    path = "/"
    assume_role_policy = "${data.aws_iam_policy_document.codedeploy_service_assume_policy_doc.json}"
}

resource "aws_iam_role_policy_attachment" "codedeploy_service_policy_attach" {
    role = "${aws_iam_role.codedeploy_service_role.name}"
    policy_arn = "${aws_iam_policy.codedeploy_service_policy.arn}"
}

resource "aws_codedeploy_deployment_group" "targets" {
    app_name = "${aws_codedeploy_app.app.name}"
    deployment_group_name = "${var.deploy_group}"
    service_role_arn = "${aws_iam_role.codedeploy_service_role.arn}"
    autoscaling_groups = ["${var.asgs}"]

    auto_rollback_configuration {
        enabled = true
        events  = ["DEPLOYMENT_FAILURE"]
    }
}


data "aws_iam_policy_document" "receive_deploy_policy_doc" {
    statement {
        actions = [
            "s3:GetObject",
            "s3:ListBucket",
            "s3:GetBucketLocation"
        ]

        resources = [
            "arn:aws:s3:::${data.null_data_source.config.inputs.bucket_name}",
            "arn:aws:s3:::${data.null_data_source.config.inputs.bucket_name}/*"
        ]
    }
}

resource "aws_iam_policy" "receive_deploy_policy" {
    policy="${data.aws_iam_policy_document.receive_deploy_policy_doc.json}"
}

resource "aws_iam_role_policy_attachment" "codedeploy_assume_attach" {
    count = "${var.instance_role_count}"

    role = "${element(var.instance_roles, count.index)}"
    policy_arn = "${aws_iam_policy.receive_deploy_policy.arn}"
}
